/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeDataRenderer.C
 *    Implementation of TapeDataRenderer.
 *
 * @author Tim Langner
 * @date   2012/01/01
 */

#include <TapeDataRenderer.h>

#include <QPainter>
#include <QMenu>
#include <QAction>

#include <serialization/Serialization.h>
#include <serialization/BinaryJSONConverter.h>

#include <widgets/MultiInputDialog.h>
#include <TapeEditor.h>
#include <TapeCommand.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

void TapeDataRenderer::fillContextMenu(QMenu* menu, TapeChannelInfo& info,
                                       TapeChannelInfo::DataMap::iterator& message,
                                       int64 time)
{
	if (hasFeature(TapeDataRenderer::CREATE))
		menu->addAction("Create message at this position");

	if (hasFeature(TapeDataRenderer::EDIT) &&
		message != info.data.end())
		menu->addAction("Edit message");
}

void TapeDataRenderer::executeAction(QAction* action, TapeChannelInfo& info,
                                     TapeChannelInfo::DataMap::iterator& message,
                                     int64 time)
{
	if (action->text() == "Create message at this position")
		addMessage(info, time);

	if (action->text() == "Edit message")
		editMessage(info, message);
}

QSize TapeDataRenderer::renderMessage(QPainter* painter, const QRect& maxRect,
                                      TapeChannelInfo& info,
                                      TapeChannelInfo::DataMap::iterator& message)
{
	return QSize();
}

QString TapeDataRenderer::getMessageInfo(TapeChannelInfo& info,
                                         TapeChannelInfo::DataMap::iterator& message)
{
	if (!info.meta)
		return "";
	try
	{
		JSONValue val;
		BinaryJSONConverter::binaryToJSON(message->second.getData(), false,
		                                  *info.meta, mEditor->getMetaDatabase(),
		                                  val);
		return json::write(val).c_str();
	}
	catch(Exception& ex)
	{
		return ex.message().c_str();
	}
	return "";
}

void TapeDataRenderer::editMessage(TapeChannelInfo& info,
                                   TapeChannelInfo::DataMap::iterator& message)
{
	// create an input dialog and fill it with the data of the message
	MultiInputDialog dialog("Editing message");
	dialog.addInputField("time", "Time offset us", QString("%1").arg(message->first));
	dialog.addInputField("frameid", "Frame ID", message->second.getFrameID().c_str());
	dialog.addInputField("sequenceid", "Sequence ID", QString("%1").arg(message->second.getSequenceID()));
	boost::optional<std::string> data;
	if (info.meta)
	{
		try
		{
			JSONValue val;
			BinaryJSONConverter::binaryToJSON(message->second.getData(), false,
			                                  *info.meta, mEditor->getMetaDatabase(),
			                                  val);
			data.reset(json::write(val));
			dialog.addInputField("data", "Data", data->c_str(), true);
		}
		catch(Exception& ex)
		{
		}
	}
	if (dialog.exec() == 0)
		return;

	int64 time = fromString<int64>(dialog.getInput("time").toStdString());
	std::string newFrameID = dialog.getInput("frameid").toStdString();
	uint32 newSequenceID = fromString<uint32>(dialog.getInput("sequenceid").toStdString());
	boost::optional<std::string> newData;
	if (data)
		newData.reset(dialog.getInput("data").toStdString());
	// did the user modify anything? if not nothing changed
	if (time == message->first &&
		newFrameID == message->second.getFrameID() &&
		newSequenceID == message->second.getSequenceID() &&
		(newData && *data == *newData))
		return;

	TapeChannelMessage newMessage(true);
	newMessage.setFrameID(newFrameID);
	newMessage.setSequenceID(newSequenceID);
	if (newData)
	{
		json::Value j;
		json::read(*newData, j);
		BinaryJSONConverter::JSONToBinary(j, *info.meta, mEditor->getMetaDatabase(),
		                                  const_cast<Buffer<uint8>&>(newMessage.getData()), false);
	}
	else
		newMessage.setData(message->second.getData());
	mEditor->executeCommand(new EditMessageCommand(info.name,
	                                               message->first, message->second,
	                                               time, newMessage,
	                                               mEditor));
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::TapeDataRenderer, mira::Object)
