/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

#include "app/MIRAPackageShell.h"
#include "core/Tools.h"

#include <utils/MakeString.h>

#include <iostream>

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

MIRAPackageShell::MIRAPackageShell(PromptProvider* promptProvider) : MIRAPackage(promptProvider) {}

MIRAPackageShell::~MIRAPackageShell() {}

void MIRAPackageShell::listPackages( string const& listOption )
{
	vector<PackageGroup*> packageGroups;
	foreach( PackageGroup* packageGroup, mDB.getRootPackage()->mSubPackages ) {
		if ( listOption == "all" ||
		     ( listOption == "installed" && mDB.isInstalled( packageGroup, false ) ) ||
		     ( listOption == "available" && !mDB.isInstalled( packageGroup, false ) ) )
			packageGroups.push_back(packageGroup);
	}

	string message = getListPackagesText(packageGroups, listOption);

	mPromptProvider->showMessage("", "List of " + listOption + " packages:\n" + message);
}

void MIRAPackageShell::installPackages( vector<string> const& packageNames )
{
	// process in reverse install sequence order, taking version requirements into account
	mCurrentActionSequence = getInstallSequence(packageNames);
	while (!mCurrentActionSequence.empty()) {
		string packageName = mCurrentActionSequence.back();
		mCurrentActionSequence.pop_back();
		installPackage(packageName);
	}
}

Package* MIRAPackageShell::installPackage( string const& packageName, Package::Version const& version /* = Package::Version() */, bool asDependency /* = false */, bool update /* = false */ )
{
	// skip if already installed (or is going to be),
	// but don't check this in update mode
	if (!update && mDB.isInstalled(packageName, true))
		return 0;
	// find all packages with that name
	vector<Package*> packages = mDB.getRootPackage()->findPackages(packageName, true);
	if (packages.size() == 0)
		MIRA_THROW(XLogical, "Cannot find package with name \"" + packageName + "\"!");
	if (find(mCurrentActionSequence.begin(), mCurrentActionSequence.end(), packageName) != mCurrentActionSequence.end())
		asDependency = false;
	Package* package = selectVersion(packages, version, asDependency);
	if (package) {
		selectSource(package);
		if (!update)
			addPackageToCheckoutPlan(package, true, false);
	}
	return package;
}

void MIRAPackageShell::deinstallPackages( vector<string> const& packageNames )
{
	// process in reverse install sequence order
	mCurrentActionSequence = getInstallSequence(packageNames);
	while (!mCurrentActionSequence.empty()) {
		string packageName = mCurrentActionSequence.back();
		mCurrentActionSequence.pop_back();
		deinstallPackage(packageName);
	}
}

void MIRAPackageShell::deinstallPackage( string const& packageName, bool update /* = false */ )
{
	// find all packages with that name
	vector<Package*> packages = mDB.getRootPackage()->findPackages( packageName, true );
	if (packages.size() == 0)
		MIRA_THROW(XLogical, "Cannot find package with name \"" + packageName + "\"!");
	// remove all packages with that name
	foreach( Package* package, packages ) {
		if (!update)
			addPackageForRemoval( package );
	}
}

///////////////////////////////////////////////////////////////////////////////

string MIRAPackageShell::selectMIRAPath( string const& defaultValue /* = "" */ ) {
	auto miraPaths = PathProvider::miraPaths();

	unsigned int selected = miraPaths.size() - 1; // choose latest path as install root

	// create options
	vector<string> options;
	unsigned int i = 0;
	foreach (Path const& path, miraPaths) {
		options.push_back(path.string());
		if (defaultValue == path.string())
			selected = i;
		++i;
	}

	// let the user select
	if (!noninteractive && options.size() > 1)
		selected = mPromptProvider->showSelectionMessage("", "Please select install root path:", options, selected);

	// return the selected path
	Path selectedPath = miraPaths[selected];
	MIRA_LOG(NOTICE) << "Selected " << selectedPath << " as install root path.";
	return selectedPath.string();
}

bool MIRAPackageShell::confirmDependencies(Package* rootPackage, Database* database) {
	foreach (auto package, rootPackage->mDependencies) {
		Dependency* dependency = dynamic_cast<Dependency*>(package);
		if (!mDB.isInstalled(dependency->mName, true)) {
			mPromptProvider->showMessage("", "Package \"" + rootPackage->mName + " (" + rootPackage->mVersion.str() + ")\" depends on the package \"" + dependency->mName + " (" + dependency->mVersion.str() + ")\", which is not installed yet.");
			installPackage(dependency->mName, dependency->mVersion, true);
		}
	}
	return true;
}

bool MIRAPackageShell::confirmCheckoutPlan() {
	vector<Package*> uninstallSequence = getPackageSequence( Database::UNINSTALL );
	vector<Package*> installSequence = getPackageSequence( Database::INSTALL );

	if (uninstallSequence.size() + installSequence.size() == 0) {
		mPromptProvider->showMessage("Info", "There are no packages to (un)install!");
		return true;
	}

	// check if a path needs reviewing
	foreach (auto& package, installSequence)
		if (package->mLocalPath == mInstallRoot || package->mLocalPath == mDepInstallPath) {
			mPromptProvider->showMessage("", "Cannot determine mountdir for package \"" + package->mName + " (" + package->mVersion.str() + ")\"!");
			selectMountDir(package);
		}

	if (noninteractive)
		return true;

	bool accepted = false;
	while (!accepted) {
		string message = getCheckoutPlanText(uninstallSequence, installSequence);

		accepted = !mPromptProvider->showYesNoMessage("", "Scheduled actions:\n" + message + "Edit before execution?", false);

		if (!accepted) {
			unsigned int selected = mPromptProvider->getSelection("Which package do you want to edit?", installSequence.size() - 1);
			Package* package = installSequence[selected];

			// get MIRA_PATH for the package
			Path miraPath = PathProvider::getAssociatedMiraPath(package->mLocalPath);
			Path subPath = relativizePath(package->mLocalPath, miraPath);

			vector<string> options;
			options.push_back("install root");
			options.push_back("mount dir");
			selected = mPromptProvider->showSelectionMessage("", "What do you want to change?", options);
			if (selected == 0) {
				miraPath = Path(selectMIRAPath(miraPath.string()));
				package->mLocalPath = miraPath / subPath;
			} else {
				package->mLocalPath = miraPath;
				selectMountDir(package);
			}
		}
	}

	return true;
}

bool MIRAPackageShell::confirmUpdatePlan(vector<pair<Package*, Package*> >& updatePlan) {
	if (updatePlan.size() == 0) {
		mPromptProvider->showMessage("Info", "There are no updates!");
		return true;
	}

	if (noninteractive)
		return true;

	bool accepted = false;
	while (!accepted) {
		string message = getUpdatePlanText(updatePlan);

		accepted = !mPromptProvider->showYesNoMessage("", "Update plan:\n" + message + "Edit before execution?", false);

		if (!accepted) {
			unsigned int selected = mPromptProvider->getSelection("Which package do you want to edit?", updatePlan.size() - 1);
			updatePlan[selected].second = installPackage(updatePlan[selected].first->mName, Package::Version(), false, true);
		}
	};

	return true;
}

bool MIRAPackageShell::confirmExportPackages(Database& ioDB, map<Path,string>& oPathMap) {
	return true; // TODO: Not implemented
}

///////////////////////////////////////////////////////////////////////////////

bool comparePackages(Package* a, Package* b) { return (*b < *a); }
Package* MIRAPackageShell::selectVersion(vector<Package*> packages, Package::Version const& version /* = Package::Version() */, bool acceptNone /* = false */) {
	sort(packages.begin(), packages.end(), comparePackages);

	// create options
	vector<string> options;
	foreach (Package* package, packages)
		options.push_back(package->mName + " (" + package->mVersion.str() + " - " + Package::getDevelStateString(*package) + " - " + (package->mType & PackageGroup::ETC ? "E" : "") + (package->mType & PackageGroup::BINARY ? "B" : "") + (package->mType & PackageGroup::SOURCE ? "S" : "") + (package->mType == PackageGroup::UNSPECIFIED ? "?" : "") + ")" + (mDB.isInstalled(package) ? " *" : ""));
	if (acceptNone)
		options.push_back("none");

	unsigned int selected = 0;

	// try to find the "best" package
	Package* bestPackage = mDB.stdSource(packages, version);
	if (bestPackage) {
		for (unsigned int i = 0; i < packages.size(); ++i) {
			if (packages[i] == bestPackage) {
				selected = i;
				break;
			}
		}
	}

	// let the user select
	if (!noninteractive && options.size() > 1)
		selected = mPromptProvider->showSelectionMessage("", "Please select version:", options, selected);

	// return selected package
	if (acceptNone && selected == options.size() - 1) {
		MIRA_LOG(NOTICE) << "Selected \"none\"" << (packages.size() > 0 ? " for package \"" + packages[0]->mName + "\"" : "") << ".";
		return 0;
	}
	Package* package = packages[selected];
	MIRA_LOG(NOTICE) << "Selected version \"" << options[selected] << "\".";
	return package;
}

void MIRAPackageShell::selectSource(Package* package) {
	auto sortedRepos = package->sortedRepos(&mDB);

	// create options
	vector<string> options;
	foreach (string const& repository, sortedRepos)
		options.push_back(mDB.getRepoFromUrl(repository)->name);

	unsigned int selected = 0; // already the "best" repo since they are sorted

	// let the user select
	if (!noninteractive && options.size() > 1)
		selected = mPromptProvider->showSelectionMessage("", "Please select source for package \"" + package->mName + " (" + package->mVersion.str() + ")\":", options, selected);

	// set selected repo as current
	package->mCurrentRepo = sortedRepos[selected];
	MIRA_LOG(NOTICE) << "Selected repo \"" << mDB.getRepoFromUrl(package->mCurrentRepo)->name << "\" for package \"" << package->mName << "\"";
}

void MIRAPackageShell::selectMountDir(Package* package) {
	if (noninteractive)
		MIRA_THROW(XRuntime, "Cannot set mountdir since we are in noninteractive mode!");

	Path installRoot = package->mLocalPath.empty() ? Path(mInstallRoot) : package->mLocalPath;

	// let the user decide
	string path = mPromptProvider->showInputMessage("", "Please complete the desired path:", installRoot.string() + "/");
	while (path.back() == '/')
		path.pop_back();

	// set selected path as local path
	package->mLocalPath = installRoot / Path(path);
}

///////////////////////////////////////////////////////////////////////////////

string MIRAPackageShell::getListPackagesText(vector<PackageGroup*> const& packageGroups, string const& listOption) const {
	bool showAll = (listOption == "all");
	bool showInstalled = (listOption == "installed");
	unsigned int maxPkgName  = 11;
	unsigned int maxCurrVLen = 11;
	unsigned int maxDescLen  = 13;

	foreach (PackageGroup* packageGroup, packageGroups) {
		string pkg = packageGroup->mName;
		if (pkg.length() > maxPkgName) maxPkgName = pkg.length();
		if (showInstalled) {
			Package* package = mDB.getInstalled( packageGroup, false );
			if (package) {
				string currV = package->mVersion.str();
				if (currV.length() > maxCurrVLen) maxCurrVLen = currV.length();
			}
		}
		string desc = packageGroup->mDescription;
		if (desc.length() > maxDescLen) maxDescLen = desc.length();
	}

	maxPkgName += 2;

	stringstream message;

	if (showAll)
		message << leftText(""     , 4);
	message << leftText("   Package ", maxPkgName);
	if (showInstalled)
		message << leftText("   Version ", maxCurrVLen);
	message << " Description " << endl;

	foreach (PackageGroup* packageGroup, packageGroups) {
		Package* package = mDB.getInstalled( packageGroup, false );
		if (showAll)
			message << centerText((package != 0) ? "ii" : "", 4);
		message << " " << leftText(packageGroup->mName, maxPkgName);
		if (showInstalled)
			message << " " << leftText(package->mVersion.str() , maxCurrVLen);
		message << packageGroup->mDescription << endl;
	}

	return message.str();
}

string MIRAPackageShell::getCheckoutPlanText(vector<Package*> const& uninstallSequence, vector<Package*> const& installSequence) const {
	unsigned int maxPkgName     = 17;
	unsigned int maxRepoName    = 12;
	unsigned int maxSubPathLen  =  9;
	unsigned int maxMountDirLen = 10;

	foreach (Package* package, uninstallSequence) {
		string pkg = package->mName + " (" + package->mVersion.str() + ")";
		if (pkg.length() > maxPkgName) maxPkgName = pkg.length();
		string repo = mDB.getRepoFromUrl(package->mCurrentRepo)->name;
		if (repo.length() > maxRepoName) maxRepoName = repo.length();
		string subpath = package->mRepoSubPath;
		if (subpath.length() > maxSubPathLen) maxSubPathLen = subpath.length();
		string mountdir = package->mLocalPath.string();
		if (mountdir.length() > maxMountDirLen) maxMountDirLen = mountdir.length();
	}

	foreach (Package* package, installSequence) {
		string pkg = package->mName + " (" + package->mVersion.str() + ")";
		if (pkg.length() > maxPkgName) maxPkgName = pkg.length();
		string repo = mDB.getRepoFromUrl(package->mCurrentRepo)->name;
		if (repo.length() > maxRepoName) maxRepoName = repo.length();
		string subpath = package->mRepoSubPath;
		if (subpath.length() > maxSubPathLen) maxSubPathLen = subpath.length();
		string mountdir = package->mLocalPath.string();
		if (mountdir.length() > maxMountDirLen) maxMountDirLen = mountdir.length();
	}

	stringstream message;

	message << "\u2554" << centerText(" Action "         , 9             , "\u2550");
	message << "\u2564" << centerText(" # "              , 5             , "\u2550");
	message << "\u2564" << centerText(" Package/Version ", maxPkgName    , "\u2550");
	message << "\u2564" << centerText(" Repository "     , maxRepoName   , "\u2550");
	message << "\u2564" << centerText(" SubPath "        , maxSubPathLen , "\u2550");
	message << "\u2564" << centerText(" MountDir "       , maxMountDirLen, "\u2550");
	message << "\u2557" << endl;

	if (uninstallSequence.size() > 0) {
		foreach (Package* package, uninstallSequence) {
			string pkg = package->mName + " (" + package->mVersion.str() + ")";
			string repo = mDB.getRepoFromUrl(package->mCurrentRepo)->name;
			string subpath = package->mRepoSubPath;
			string mountdir = package->mLocalPath.string();
			message << "\u2551" << centerText("Uninstall", 9);
			message << "\u2502" << leftText(""      , 5);
			message << "\u2502" << leftText(pkg     , maxPkgName);
			message << "\u2502" << leftText(repo    , maxRepoName);
			message << "\u2502" << leftText(subpath , maxSubPathLen);
			message << "\u2502" << leftText(mountdir, maxMountDirLen);
			message << "\u2551" << endl;
		}
	}

	if (uninstallSequence.size() > 0 && installSequence.size() > 0) {
		message << "\u255F" << leftText("", 9             , "\u2500");
		message << "\u253C" << leftText("", 5             , "\u2500");
		message << "\u253C" << leftText("", maxPkgName    , "\u2500");
		message << "\u253C" << leftText("", maxRepoName   , "\u2500");
		message << "\u253C" << leftText("", maxSubPathLen , "\u2500");
		message << "\u253C" << leftText("", maxMountDirLen, "\u2500");
		message << "\u2562" << endl;
	}

	if (installSequence.size() > 0) {
		unsigned int i = 0;
		foreach (Package* package, installSequence) {
			string pkg = package->mName + " (" + package->mVersion.str() + ")";
			string repo = mDB.getRepoFromUrl(package->mCurrentRepo)->name;
			string subpath = package->mRepoSubPath;
			string mountdir = package->mLocalPath.string();
			message << "\u2551" << centerText("Install", 9);
			message << "\u2502" << rightText(MakeString() << i++ << " ", 5);
			message << "\u2502" << leftText(pkg     , maxPkgName);
			message << "\u2502" << leftText(repo    , maxRepoName);
			message << "\u2502" << leftText(subpath , maxSubPathLen);
			message << "\u2502" << leftText(mountdir, maxMountDirLen);
			message << "\u2551" << endl;
		}
	}

	message << "\u255A" << leftText("", 9             , "\u2550");
	message << "\u2567" << leftText("", 5             , "\u2550");
	message << "\u2567" << leftText("", maxPkgName    , "\u2550");
	message << "\u2567" << leftText("", maxRepoName   , "\u2550");
	message << "\u2567" << leftText("", maxSubPathLen , "\u2550");
	message << "\u2567" << leftText("", maxMountDirLen, "\u2550");
	message << "\u255D" << endl;

	return message.str();
}

string MIRAPackageShell::getUpdatePlanText(vector<pair<Package*, Package*>> const& updatePlan) const {
	// get longest package name length
	unsigned int maxPkgName    = 11;
	unsigned int maxCurrVLen   =  9;
	unsigned int maxRecommVLen = 15;

	foreach (auto const& update, updatePlan) {
		string pkgName = update.first->mName;
		if (pkgName.length() > maxPkgName) maxPkgName = pkgName.length();
		string currV = update.first->mVersion.str();
		if (currV.length() > maxCurrVLen) maxCurrVLen = currV.length();
		string recommV = update.second->mVersion.str();
		if (recommV.length() > maxRecommVLen) maxRecommVLen = recommV.length();
	}

	stringstream message;

	message << "\u2554" << centerText(" # "          , 5            , "\u2550");
	message << "\u2564" << centerText(" Package "    , maxPkgName   , "\u2550");
	message << "\u2564" << centerText(" Current "    , maxCurrVLen  , "\u2550");
	message << "\u2564" << centerText(" Recommended ", maxRecommVLen, "\u2550");
	message << "\u2557" << endl;

	unsigned int i = 0;
	foreach (auto const& update, updatePlan) {
		message << "\u2551" << rightText(MakeString() << i++ << " "      , 5);
		message << "\u2502" << centerText(update.first->mName          , maxPkgName);
		message << "\u2502" << centerText(update.first->mVersion.str() , maxCurrVLen);
		message << "\u2502" << centerText(update.second->mVersion.str(), maxRecommVLen);
		message << "\u2551" << endl;
	}

	message << "\u255A" << centerText("", 5            , "\u2550");
	message << "\u2567" << centerText("", maxPkgName   , "\u2550");
	message << "\u2567" << centerText("", maxCurrVLen  , "\u2550");
	message << "\u2567" << centerText("", maxRecommVLen, "\u2550");
	message << "\u255D" << endl;

	return message.str();
}

///////////////////////////////////////////////////////////////////////////////

string MIRAPackageShell::leftText(string const& input, unsigned int space, string const& fillchar) const {
	string output;
	output += input;
	for (unsigned int i = 0; i < space - input.length(); ++i)
		output += fillchar;
	return output;
}

string MIRAPackageShell::centerText(string const& input, unsigned int space, string const& fillchar) const {
	string output;
	for (unsigned int i = 0; i < (space - input.length()) / 2; ++i)
		output += fillchar;
	output += input;
	for (unsigned int i = 0; i < (space - input.length() + 1) / 2; ++i)
		output += fillchar;
	return output;
}

string MIRAPackageShell::rightText(string const& input, unsigned int space, string const& fillchar) const {
	string output;
	for (unsigned int i = 0; i < space - input.length(); ++i)
		output += fillchar;
	output += input;
	return output;
}

///////////////////////////////////////////////////////////////////////////////

}
