/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CmdLineOptions.C
 *    Description.
 *
 * @author Ronny Stricker
 * @date   2012/02/29
 */

#include <app/CmdLineOptions.h>

#include <core/Repository.h>
#include <core/Tools.h>

#include <utils/MakeString.h>

#include <boost/regex.hpp>

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

string replace_all(string text) {
	boost::replace_all(text, " ", "_");
	boost::replace_all(text, ".", "_");
	boost::replace_all(text, "-", "_");
	boost::replace_all(text, "(", "_");
	boost::replace_all(text, ")", "_");
	boost::replace_all(text, "/", "_");
	return text;
}

void usage( vector<string> const& repos )
{
	std::cout <<
			"mirapackage <command> [parameters]\n\n"
			"Simplifies checking out branches and their dependencies, by\n"
			"examining the available branches on the specified repositories.\n"
			"The dependencies are obtained from the *.package files within the\n"
			"branches. The checkout location is determined using the mountdir.xml\n"
			"files in the branches. A mountdir.xml file contains the path in the\n"
			"source tree, that the branch corresponds to. Subdirectories of the\n"
			"branch are checked out relative to that path.\n"
			"\n"
			"By default the following repositories are added:\n";
	foreach( string path, repos ) {
		cout << "   " << path << std::endl;
	}
	cout << "\n"
			"To add further repositories, type:\n"
			"  > mirapackage --addrepo [name] [URL of branch within repository]\n"
			<< endl;
}

bool listPackages( string const& listOption, MIRAPackageShell& core )
{
	if ( listOption != "all" && listOption != "installed" &&
	     listOption != "available" && listOption != "export") {
		return false;
	}

	core.listPackages( listOption );
	return true;
}

bool exportPackages( string const& fileName, MIRAPackageShell& core )
{
	core.exportDB( fileName );
	return true;
}

bool importPackages( string const& fileName, MIRAPackageShell& core )
{
	// try to get working mira path, if not already selected
	if ( core.mInstallRoot.empty() )
		core.resolveMIRAPath();

	core.importDB( fileName );

	core.doIt();
	core.save();
	return true;
}

bool showPackageDetails( string const& packageName, MIRAPackageShell& core )
{
	vector<Package*> tPackages = core.mDB.getRootPackage()->
			findPackages( packageName, true );

	cout << "The following packages have been found:\n\n";
	foreach( Package* tPackage, tPackages ) {
		cout << "----------------------\n";
		cout << "Name: " << tPackage->mName << "\n";
		cout << "Version: " << tPackage->mVersion.str() << "\n";
		cout << "Description: " << tPackage->mDescription << "\n";
		cout << "Author: " << tPackage->mAuthor << "\n";
		cout << "Dependencies: ";
		foreach( Package* dep, tPackage->mDependencies ) {
			cout << dep->mName << " ";
		}
		cout << "\n";
		cout << "Repository: " << core.mDB.getRepoNameFromUrl(tPackage->mCurrentRepo) << "\n";
		cout << "RepoSubPath: " << tPackage->mRepoSubPath << "\n";
		cout << "FileName: " << tPackage->mFileName << "\n";
		cout << "LocalPath: " << tPackage->mLocalPath.string() << "\n";
		cout << "\n\n";
	}
	return true;
}

bool installPackages( std::vector<std::string> packageNames, MIRAPackageShell& core )
{
	// try to get working mira path, if not already selected
	if ( core.mInstallRoot.empty() )
		core.resolveMIRAPath();

	core.installPackages( packageNames );
	core.doIt();
	core.save();
	return true;
}

bool deinstallPackages( vector<string> packageNames, MIRAPackageShell& core )
{
	// if "AllPackages" is given as an option, the program tries
	// to deinstall all packages
	if ( packageNames.size() == 1 && packageNames[0] == "AllPackages" ) {
		// add all known packages
		packageNames.clear();
		foreach( Package* tPackage, core.mDB.mInstalledPackages ) {
			packageNames.push_back( tPackage->mName );
			cout << " deinstall " << tPackage->mName << endl;
		}
	}
	core.deinstallPackages( packageNames );
	core.doIt();
	core.save();
	return true;
}

void collectDependencies( string const& packageName, MIRAPackageShell& core, const bool recursive, set<string>& deps )
{
	vector<Package*> tPackages = core.mDB.getRootPackage()->
			findPackages( packageName, true );
	Package* tPackage = core.mDB.stdSource( tPackages );
	if ( tPackage ) {
		foreach( Package* dep, tPackage->mDependencies ) {
			if (deps.find(dep->mName) != deps.end())
				continue;
			deps.insert( dep->mName );
			if (recursive)
				collectDependencies( dep->mName, core, recursive, deps );
		}
	}
}

bool listDependencies( string const& packageName, MIRAPackageShell& core, const bool recursive )
{
	set<string> tDependencies;
	collectDependencies( packageName, core, recursive, tDependencies );
	if ( tDependencies.size() )
		listInstallSequence( vector<string>(tDependencies.begin(), tDependencies.end()), core );
	return true;
}

void iterateDependencyTree( string const& packageName, MIRAPackageShell& core, const bool prune, const string& prefix, set<string>& ignore )
{
	vector<Package*> tPackages = core.mDB.getRootPackage()->
			findPackages( packageName, true );
	Package* tPackage = core.mDB.stdSource( tPackages );
	if ( tPackage ) {
		foreach( Package* dep, tPackage->mDependencies ) {
			cout << prefix << dep->mName;
			if (prune && ignore.find(dep->mName) != ignore.end()) {
				cout << "*" << endl;
				continue;
			}
			cout << endl;
			ignore.insert( dep->mName );
			if ( dep->mName != "MIRABase" && dep->mName != "MIRAFramework" ) {
				iterateDependencyTree( dep->mName, core, prune, prefix + "  ", ignore );
			}
		}
	}
}

bool showDependencyTree( string const& packageName, MIRAPackageShell& core, const bool prune, const string& prefix )
{
	set<string> ignore;
	iterateDependencyTree( packageName, core, prune, prefix, ignore );
	return true;
}

void showDependencyGraphCluster( DependencyGraphCluster* cluster, string const& prefix = "\t" )
{
	foreach( auto package, cluster->packages )
		cout << prefix << package.first << (package.second.empty() ? "" : " " + package.second) << endl;
	foreach( DependencyGraphCluster* subcluster, cluster->subcluster ) {
		cout << prefix << "subgraph cluster_" << replace_all(subcluster->label) << " {" << endl;
		cout << prefix << "\tlabel=\"" << subcluster->label << "\"" << endl;
		cout << prefix << "\tstyle=rounded" << endl;
		cout << prefix << "\tpenwidth=3"<< endl;
		showDependencyGraphCluster(subcluster, prefix + "\t");
		cout << prefix << "}" << endl;
	}
}

bool showDependencyGraph( vector<string> const& packageNames, MIRAPackageShell& core )
{
	cout << "digraph " << replace_all(boost::algorithm::join(packageNames, "_")) << "_dependencies" << endl;
	cout << "{" << endl;

	list<string> processed;
	DependencyGraphCluster rootCluster;
	list<string> dependencies;

	bool res = false;
	if ( packageNames.size() == 1 && packageNames[0] == "installed" ) {
		foreach( PackageGroup* packageGroup, core.mDB.getRootPackage()->mSubPackages ) {
			Package* package = core.mDB.getInstalled( packageGroup, false );
			if ( package )
				res |= getInstalledDependencyGraph( package->mName, core, &rootCluster, dependencies );
		}
	} else {
		foreach( string const& packageName, packageNames ) {
			if ( find(processed.begin(), processed.end(), packageName) == processed.end() ) {
				res |= getDependencyGraph( packageName, core, processed, &rootCluster, dependencies );
			}
		}
	}

	showDependencyGraphCluster(&rootCluster);

	foreach( const string& dependency, dependencies ) {
		cout << "\t" << dependency << endl;
	}

	cout << "}" << endl;

	return res;
}

bool getDependencyGraph( string const& packageName, MIRAPackageShell& core, list<string>& processed, DependencyGraphCluster* rootCluster, list<string>& dependencies)
{
	vector<Package*> tPackages = core.mDB.getRootPackage()->
			findPackages( packageName, true );
	Package* tPackage = core.mDB.stdSource( tPackages );
	if ( tPackage ) {
		processed.push_back( tPackage->mName );
		tPackage->selectStdRepo( &core.mDB );
		stringstream packageString;
		packageString << "[shape=box,label=\"" << tPackage->mName;
		packageString << "\\n" << tPackage->mVersion.str();
		RepositoryPtr repo = core.mDB.getRepoFromUrl(tPackage->mCurrentRepo);
		packageString << "\"";
		if ( tPackage->mDevelState == Package::DEPRECATED )
			packageString << ",style=dashed,color=grey";
		else if ( tPackage->mDevelState == Package::EXPERIMENTAL )
			packageString << ",style=dashed";
		else if ( tPackage->mDevelState == Package::RELEASE )
			packageString << ",style=bold";
		packageString << "]";
		string packageNameReplaced = replace_all(tPackage->mName);
		rootCluster->getSubcluster(repo->name)->packages[packageNameReplaced] = packageString.str();
		foreach( Package* dep, tPackage->mDependencies ) {
			if ( dep->mName != "MIRABase" && dep->mName != "MIRAFramework" ) {
				stringstream dependencyString;
				dependencyString << packageNameReplaced << " -> " << replace_all(dep->mName);
				Dependency* dep2 = dynamic_cast<Dependency*>(dep);
				if ( dep2 && dep2->mDepFlag & (Dependency::FACULTATIVE | Dependency::RUNTIME) ) {
					dependencyString << " [style=dashed";
					if ( dep2->mDepFlag & Dependency::FACULTATIVE )
						dependencyString << ",color=grey";
					dependencyString << "]";
				}
				dependencies.emplace_back(dependencyString.str());
				if ( find(processed.begin(), processed.end(), dep->mName) == processed.end() ) {
					getDependencyGraph( dep->mName, core, processed, rootCluster, dependencies );
				}
			}
		}
	} else {
		processed.push_back( packageName );
		rootCluster->packages[packageName] = MakeString() << "[shape=box,label=\"" << packageName << "\",style=dotted]";
	}
	return true;
}

bool getInstalledDependencyGraph( string const& packageName, MIRAPackageShell& core, DependencyGraphCluster* rootCluster, list<string>& dependencies)
{
	Package* tPackage = core.mDB.getInstalled( packageName, false );
	if ( tPackage ) {
		tPackage->selectStdRepo( &core.mDB );
		stringstream packageString;
		packageString << "[shape=box,label=\"" << tPackage->mName;
		packageString << "\\n" << tPackage->mVersion.str();
		packageString << "\"";
		if ( tPackage->mDevelState == Package::DEPRECATED )
			packageString << ",style=dashed,color=grey";
		else if ( tPackage->mDevelState == Package::EXPERIMENTAL )
			packageString << ",style=dashed";
		else if ( tPackage->mDevelState == Package::RELEASE )
			packageString << ",style=bold";
		packageString << "]";
		string packageName = replace_all(tPackage->mName);
		Path miraPath = PathProvider::getAssociatedMiraPath(tPackage->mLocalPath);
		DependencyGraphCluster* cluster = rootCluster->getSubcluster(miraPath.string());
		string path = relativizePath(tPackage->mLocalPath, miraPath).string();
		while (true) {
			auto pos = path.find("/");
			if (pos == string::npos)
				break;
			cluster = cluster->getSubcluster(path.substr(0, pos));
			path = path.substr(pos + 1);
		}
		cluster->packages[packageName] = packageString.str();
		foreach( Package* dep, tPackage->mDependencies ) {
			if ( dep->mName != "MIRABase" && dep->mName != "MIRAFramework" ) {
				stringstream dependencyString;
				dependencyString << packageName << " -> " << replace_all(dep->mName);
				Dependency* dep2 = dynamic_cast<Dependency*>(dep);
				if ( dep2 && dep2->mDepFlag & (Dependency::FACULTATIVE | Dependency::RUNTIME) ) {
					dependencyString << " [style=dashed";
					if ( dep2->mDepFlag & Dependency::FACULTATIVE )
						dependencyString << ",color=grey";
					dependencyString << "]";
				}
				dependencies.emplace_back(dependencyString.str());
			}
		}
	} else {
		rootCluster->packages[packageName] = MakeString() << "[shape=box,label=\"" << packageName << "\",style=dotted]";
	}
	return true;
}

bool listDependents( string const& packageName, MIRAPackageShell& core, const bool recursive )
{
	vector<Package*> tPackages = core.mDB.getRootPackage()->
			findPackages( packageName, true );
	Package* tPackage = core.mDB.stdSource( tPackages );
	if ( tPackage ) {
		set<string> tDependents;
		tDependents.insert( packageName );
		unsigned long oldSz = 0, sz = 1;
		while (oldSz != sz) {
			set<Package*> tAll = core.mDB.getRootPackage()->getPackages(true);
			set<string> tNewDependents;
			foreach( Package* p, tAll ) {
				foreach( Package* dep, p->mDependencies ) {
					foreach( string depd, tDependents ) {
						if ( dep->mName == depd ) {
							tNewDependents.insert( p->mName );
							break;
						}
					}
				}
			}
			tDependents.insert(tNewDependents.begin(), tNewDependents.end());
			oldSz = sz;
			sz = tDependents.size();
			if (!recursive)
				oldSz = sz;
		}
		tDependents.erase( packageName );
		vector<string> tDependentList( tDependents.begin(), tDependents.end() );
		listInstallSequence( tDependentList, core );
	}
	return true;
}

bool showDependentTree( string const& packageName, MIRAPackageShell& core, const string& prefix )
{
	vector<Package*> tPackages = core.mDB.getRootPackage()->
			findPackages( packageName, true );
	Package* tPackage = core.mDB.stdSource( tPackages );
	if ( tPackage ) {
		set<string> tDependents;
		set<Package*> tAll = core.mDB.getRootPackage()->getPackages(true);
		foreach( Package* p, tAll ) {
			foreach( Package* dep, p->mDependencies ) {
				if ( dep->mName == packageName ) {
					tDependents.insert( p->mName );
					break;
				}
			}
		}
		foreach( string const& name, tDependents ) {
			cout << prefix << name << endl;
			showDependentTree( name, core, prefix + "  " );
		}
	}
	return true;
}

bool listInstallSequence( vector<string> packageNames, MIRAPackageShell& core )
{
	vector<string> tInstallSequence = core.getInstallSequence( packageNames );
	foreach( string tPackage, tInstallSequence ) {
		cout << " " << tPackage;
	}
	cout << endl;
	return true;
}

bool listDeinstallSequence( vector<string> packageNames, MIRAPackageShell& core )
{
	vector<string> tInstallSequence = core.getInstallSequence( packageNames );
	reverse(tInstallSequence.begin(), tInstallSequence.end());
	foreach( string tPackage, tInstallSequence ) {
		cout << " " << tPackage;
	}
	cout << endl;
	return true;
}

///////////////////////////////////////////////////////////////////////////////

} // namespace
