/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MIRAPackageShell.h
 *
 * @author Ronny Stricker
 * @date   2013/09/05
 */

#ifndef _MIRA_MIRAPACKAGESHELL_H_
#define _MIRA_MIRAPACKAGESHELL_H_

#include <core/MIRAPackage.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief Class implements the main functionality for package administration
 * and installation. Basically, this class consists of a database and provides
 * additional functions for package handling.
 */
class MIRAPackageShell : public MIRAPackage
{
public:
	MIRAPackageShell(PromptProvider* promptProvider);

	virtual ~MIRAPackageShell();

public:

	virtual void listPackages( std::string const& listOption );

	virtual void installPackages( std::vector<std::string> const& packageNames );

	Package* installPackage( std::string const& packageName, Package::Version const& version = Package::Version(), bool asDependency = false, bool update = false );

	virtual void deinstallPackages( std::vector<std::string> const& packageNames );

	void deinstallPackage( std::string const& packageName, bool update = false );

protected:

	/**
	 * @brief Show PathSelectDialog for user interaction.
	 */
	virtual std::string selectMIRAPath( std::string const& defaultValue = "" );

	/**
	 * @brief Show DependencyDialog and let the user decide.
	 * @see MIRAPackage::confirmDependencies()
	 */
	virtual bool confirmDependencies( Package* rootPackage, Database* database );

	/**
	 * @brief Show InstallOverviewDialog and let the user decide.
	 */
	virtual bool confirmCheckoutPlan();

	/**
	 * @brief Show UpdateDialog and let the user decide.
	 * @param updatePlan updatePlan which should be reviewed
	 * @return true if update should be done
	 */
	virtual bool confirmUpdatePlan( std::vector<std::pair<Package*,Package*> >& updatePlan );

	/**
	 * @brief Show ExportDialog and let the user decide.
	 * The packages which are not selected will be delete from the database.
	 * @param ioDB Database to work with
	 * @return true if the user has accepted the dialog.
	 */
	virtual bool confirmExportPackages( Database& ioDB, std::map<Path,std::string>& oPathMap );

private:
	Package* selectVersion(std::vector<Package*> packages, Package::Version const& version = Package::Version(), bool acceptNone = false);

	void selectSource(Package* package);

	void selectMountDir(Package* package);

private:
	std::string getListPackagesText(std::vector<PackageGroup*> const& packageGroups, std::string const& listOption) const;

	std::string getCheckoutPlanText(std::vector<Package*> const& uninstallSequence, std::vector<Package*> const& installSequence) const;

	std::string getUpdatePlanText(std::vector<std::pair<Package*, Package*>> const& updatePlan) const;

private:
	std::string leftText(std::string const& input, unsigned int space, std::string const& fillchar = " ") const;

	std::string centerText(std::string const& input, unsigned int space, std::string const& fillchar = " ") const;

	std::string rightText(std::string const& input, unsigned int space, std::string const& fillchar = " ") const;

private:
	std::vector<std::string> mCurrentActionSequence;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif /* _MIRA_MIRAPACKAGESHELL_H_ */
