/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MIRADaemon.C
 *    A linux daemon starting a framework.
 *
 * @author Tim Langner, Christian Martin
 * @date   2012/03/19
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <syslog.h>
#include <string.h>
#include <assert.h>
#include <signal.h>

#include <fw/Framework.h>

volatile sig_atomic_t gTermFlag = 0;
volatile sig_atomic_t gHUPFlag = 0;

void signalHandler(int sig)
{
	switch(sig) {
		case SIGHUP:
			syslog(LOG_WARNING, "Received SIGHUP signal.");
			gHUPFlag = 1;
			break;
		case SIGTERM:
			syslog(LOG_WARNING, "Received SIGTERM signal.");
			gTermFlag = 1;
			break;
		default:
			syslog(LOG_WARNING, "Unhandled signal (%s)", strsignal(sig));
			break;
	}
}

int main(int argc, char *argv[])
{
	signal(SIGHUP, signalHandler);
	signal(SIGTERM, signalHandler);
	signal(SIGINT, signalHandler);
	signal(SIGQUIT, signalHandler);

	syslog(LOG_INFO, "mirad starting up");

#if defined(DEBUG)
	setlogmask(LOG_UPTO(LOG_DEBUG));
	openlog("mirad", LOG_CONS | LOG_NDELAY | LOG_PERROR | LOG_PID, LOG_USER);
#else
	setlogmask(LOG_UPTO(LOG_INFO));
	openlog("mirad", LOG_CONS, LOG_USER);
#endif

	pid_t pid, sid;

	syslog(LOG_INFO, "starting the daemonizing process");

	pid = fork();
	if (pid < 0)
		exit(EXIT_FAILURE);

	if (pid > 0)
		exit(EXIT_SUCCESS);

	///////////////////////////////////////////////////////////////////////////
	// At this point we're running in the child process. This process runs a
	// loop, which starts and monitors the working process

	close(STDIN_FILENO);
	close(STDOUT_FILENO);
	close(STDERR_FILENO);

	// We have to open three dummy files, to ensure, that the file handles
	// 0, 1 and 2 are in use. Otherwise, the next opened files will get one of
	// this handles and the output of std::cout/std::cerr will go this file!
	// (See here: http://www.microhowto.info/howto/cause_a_process_to_become_a_daemon_in_c.html)
	int f0 = open("/dev/null", O_RDONLY);
	int f1 = open("/dev/null", O_WRONLY);
	int f2 = open("/dev/null", O_RDWR);

	sid = setsid();
	if (sid < 0)
		exit(EXIT_FAILURE);

	if ((chdir("/")) < 0)
		exit(EXIT_FAILURE);

	umask(0);

	///////////////////////////////////////////////////////////////////////////
	// Write the pid of the monitoring process to a file.

	int argc2 = 0;
	char** argv2 = new char*[argc];

	std::string pidFile;
	for(int i = 0; i < argc; i++) {
		if ((i < argc-1) && (std::string(argv[i]) == "--pidfile")) {
			pidFile = std::string(argv[i+1]);
			i++;
		} else {
			argv2[argc2++] = argv[i];
		}
	}
	if (pidFile.size() > 0) {
		try {
			std::ofstream pidFileStream;
			pidFileStream.open(pidFile, std::ios::out);
			pidFileStream << getpid() << std::endl;
			pidFileStream.close();
		} catch(...) {
			std::stringstream ss;
			ss << "mirad can't write pidfile: " << pidFile;
			syslog(LOG_INFO, "%s", ss.str().c_str());
		}
	}

	///////////////////////////////////////////////////////////////////////////

	while (!gTermFlag)
	{
		MIRA_SLEEP(500);

		pid_t workerPID = fork();
		if (workerPID < 0)
			exit(EXIT_FAILURE);

		if (workerPID > 0) {

			///////////////////////////////////////////////////////////////////
			// At this point we are in the daemon process, which monitors the
			// working process.

			bool workingProcessStillRunning = true;
			while (workingProcessStillRunning && !gTermFlag && !gHUPFlag) {
				MIRA_SLEEP(500);
				int waitStatus = 0;
				pid_t p = waitpid(workerPID, &waitStatus, WNOHANG);
				if (p < 0) {
					// An error occurred. The working process seems to be
					// disappeared. Therefore, we must restart the process.
					workingProcessStillRunning = false;
					syslog(LOG_INFO, "mirad worker process disappeared.");
				} else {
					if (waitStatus != 0) {
						// If the working process exited, we have to restart it.
						if (WIFEXITED(waitStatus))
							workingProcessStillRunning = false;
						if (WIFSIGNALED(waitStatus))
							workingProcessStillRunning = false;
						syslog(LOG_INFO, "mirad worker process exited.");
					}
				}
			}
			if (gTermFlag) {
				syslog(LOG_INFO, "Sending SIGTERM to worker process.");
				kill(workerPID, SIGTERM);
			}
			if (gHUPFlag) {
				syslog(LOG_INFO, "Sending SIGHUP to worker process.");
				kill(workerPID, SIGHUP);
			}
		} else {

			///////////////////////////////////////////////////////////////////
			// At this point we are in the process, which does the real work.

			syslog(LOG_INFO, "mirad worker process started.");
			try
			{
				mira::Framework framework(argc2, argv2);
				framework.load();
				framework.start();
				while(!gTermFlag && !gHUPFlag)
					MIRA_SLEEP(1000);
			} catch(std::exception& ex) {
				std::stringstream ss;
				ss << "mirad worker process exiting with error: " << ex.what();
				syslog(LOG_INFO, "%s", ss.str().c_str());
				exit(EXIT_FAILURE);
			} catch(...) {
				syslog(LOG_INFO, "mirad worker process exiting with error.");
				exit(EXIT_FAILURE);
			}
			syslog(LOG_INFO, "mirad worker process exiting with success.");
			exit(EXIT_SUCCESS);
		}
	}

	// Delete the pidfile
	if (pidFile.size() > 0)
		unlink(pidFile.c_str());

	syslog(LOG_INFO, "mirad exiting");

	exit(0);
}
