/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file AuthorityView.C
 *    Implementation of AuthorityView.h
 *
 * @author Tim Langner
 * @date   2011/01/16
 */

#include <QCoreApplication>
#include <QEvent>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QMenu>
#include <QToolBar>
#include <QFileDialog>
#include <QHelpEvent>
#include <QToolTip>

#include <views/AuthorityView.h>
#include <widgets/CollapsibleTreeHeader.h>
#include <widgets/QtUtils.h>
#include <serialization/Serialization.h>
#include <thread/ThreadMonitor.h>
#include <fw/MicroUnit.h>
#include <fw/UnitManager.h>

///////////////////////////////////////////////////////////////////////////////

MIRA_CLASS_SERIALIZATION(mira::AuthorityView, mira::ViewPart);

///////////////////////////////////////////////////////////////////////////////

namespace mira
{

///////////////////////////////////////////////////////////////////////////////

class AuthorityTreeWidget : public QTreeWidget
{
public:

	class Item : public QTreeWidgetItem
	{
	public:
		Item(QTreeWidget* parent, const std::string& name) :
			QTreeWidgetItem(parent),
			userExpanded(false),
			autoExpanded(false),
			subscribedChannelItem(NULL),
			publishedChannelItem(NULL),
			serviceInterfaceItem(NULL),
			infoItem(NULL)
		{
			setText(0, name.c_str());
			setData(0, (Qt::ItemDataRole)TreeViewFilter::FilterRole, name.c_str());
			setText(1, "");
		}

		Item(QTreeWidgetItem* parent, const std::string& name) :
			QTreeWidgetItem(parent),
			userExpanded(false),
			autoExpanded(false),
			subscribedChannelItem(NULL),
			publishedChannelItem(NULL),
			serviceInterfaceItem(NULL),
			infoItem(NULL)
		{
			setText(0, name.c_str());
			setData(0, (Qt::ItemDataRole)TreeViewFilter::FilterRole, name.c_str());
			setText(1, "");
		}

		std::string authorityID;
		bool userExpanded;
		bool autoExpanded;
		QTreeWidgetItem* subscribedChannelItem;
		QTreeWidgetItem* publishedChannelItem;
		QTreeWidgetItem* serviceInterfaceItem;
		QTreeWidgetItem* infoItem;
		std::map<std::string, QTreeWidgetItem*> subscribedItems;
		std::map<std::string, QTreeWidgetItem*> publishedItems;
		std::map<std::string, QTreeWidgetItem*> interfaceItems;
	};

public:

	AuthorityTreeWidget(QWidget* parent) : QTreeWidget(parent)
	{
		setColumnCount(2);
		QStringList l;
		l << "Name" << "Status";
		setHeaderLabels(l);
		QHeaderView* header = new CollapsibleTreeHeader(Qt::Horizontal,this);
		setHeader( header );
		setRootIsDecorated(true);

	}

};

///////////////////////////////////////////////////////////////////////////////

#define PROPERTY_CLEAR_EVENT_ID  ((QEvent::Type)(QEvent::User+1001))
#define PROPERTY_UPDATE_EVENT_ID ((QEvent::Type)(QEvent::User+1002))

class PropertyUpdateEvent : public QEvent
{
public:
	PropertyUpdateEvent() : QEvent(PROPERTY_UPDATE_EVENT_ID) {}
public:
	std::vector<boost::shared_ptr<PropertyNode>> nodes;
};

///////////////////////////////////////////////////////////////////////////////

AuthorityView::AuthorityView() :
	mControl(new PropertyViewPage()),
	mInterruptPropertiesUpdateFlag(false),
	mShowHidden(false),
	mShowRelative(false),
	mShowInternal(false),
	mUI(NULL)
{
	mHideFilterBarAct = new QAction("Hide filter bar automatically",this);
	mHideFilterBarAct->setCheckable(true);

	mFilterNamesAct = new QAction("Filter authority names only",this);
	mFilterNamesAct->setCheckable(true);
	mFilterNamesAct->setChecked(true);
}

AuthorityView::~AuthorityView()
{
	mInfoWorkerThread.interrupt();
	mPropertyWorkerThread.interrupt();

	mInfoWorkerThread.join();
	mPropertyWorkerThread.join();

	delete mControl;
}

QWidget* AuthorityView::createPartControl()
{
	mUI=new UI(this);
	QMenu* menu = getViewMenu();
	QAction* action = menu->addAction("Show hidden authorities");
	action->setCheckable(true);
	action->setChecked(mShowHidden);
	connect(action, SIGNAL(toggled(bool)), this, SLOT(onShowHidden(bool)));
	QAction* action2 = menu->addAction("Show relative channel names");
	action2->setCheckable(true);
	action2->setChecked(mShowRelative);
	connect(action2, SIGNAL(toggled(bool)), this, SLOT(onShowRelative(bool)));
	QAction* action3 = menu->addAction("Show internal channel names");
	action3->setCheckable(true);
	action3->setChecked(mShowInternal);
	connect(action3, SIGNAL(toggled(bool)), this, SLOT(onShowInternal(bool)));

	menu->addAction(mHideFilterBarAct);
	menu->addAction(mFilterNamesAct);

	QAction* showFilterAction = new QAction(this);
	showFilterAction->setShortcutContext(Qt::WidgetWithChildrenShortcut);
	showFilterAction->setShortcut(QString("Ctrl+F"));
	addAction(showFilterAction);
	connect(showFilterAction, SIGNAL(triggered(bool)), mUI->mFilter, SLOT(showFilter()));

	connect(mHideFilterBarAct, SIGNAL(toggled(bool)), mUI->mFilter, SLOT(setAutoHide(bool)));
	connect(mFilterNamesAct, SIGNAL(toggled(bool)), mUI->mFilter, SLOT(useFilterRole(bool)));
	mUI->mFilter->setAutoHide(mHideFilterBarAct->isChecked());
	mUI->mFilter->useFilterRole(mFilterNamesAct->isChecked());

	mUI->tree->installEventFilter(this);

	startTimer(2000);
	updateAuthorityTree();

	mControl->setWindowTitle(this->windowTitle());

	mInfoWorkerThread = boost::thread(boost::bind(&AuthorityView::updateInfoWorker, this));
	mPropertyWorkerThread = boost::thread(boost::bind(&AuthorityView::updatePropertyWorker, this));

	return mUI;
}

bool AuthorityView::eventFilter(QObject* obj, QEvent* event)
{
	if (obj == mUI->tree && event)
	{
		if (event->type() == PROPERTY_CLEAR_EVENT_ID)
		{
			getPropertyControl()->clearPropertyNodes();
		} else
		if (event->type() == PROPERTY_UPDATE_EVENT_ID)
		{
			PropertyUpdateEvent* updateEvent = static_cast<PropertyUpdateEvent*>(event);
			getPropertyControl()->clearPropertyNodes();
			foreach(auto n, updateEvent->nodes)
				getPropertyControl()->addPropertyNode(n);
		} else
		if (event->type() == QEvent::ToolTip)
		{
			QHelpEvent* helpEvent = static_cast<QHelpEvent*>(event);
			QTreeWidgetItem* item = mUI->tree->itemAt(mUI->tree->viewport()->mapFromGlobal(helpEvent->globalPos()));
			if (item && !item->text(1).isEmpty())
			{
				bool isLocal = true;
				try
				{
					AuthorityTreeWidget::Item* a = dynamic_cast<AuthorityTreeWidget::Item*>(item);
					if (a != NULL && !a->authorityID.empty()) {
						boost::recursive_mutex::scoped_lock lock(mInfoMutex);
						AuthorityInfoMap::const_iterator iter =	mAuthorityInfo.find(a->authorityID);
						if (iter != mAuthorityInfo.end())
							isLocal = iter->second.isLocal;
					}
				}
				catch(...) {}
				if (isLocal)
					QToolTip::showText(helpEvent->globalPos(), item->text(0) + " : " + item->text(1));
				else
					QToolTip::showText(helpEvent->globalPos(), item->text(0) + " (remote) : " + item->text(1));
			}
			else
				QToolTip::hideText();
		}
	}
	return QObject::eventFilter(obj, event);
}

Object* AuthorityView::getAdapter(const Class& adapter)
{
	if(adapter == PropertyViewPage::CLASS())
		return mControl;

	return ViewPart::getAdapter(adapter);
}

void AuthorityView::updateInfoWorker()
{
	ThreadMonitor::instance().addThisThread("#AuthorityView InfoThread");
	while (!boost::this_thread::interruption_requested())
	{
		std::set<std::string> authorities = MIRA_FW.getAuthorityManager().getAuthorities();
		// first erase all authorities that do not longer exist
		{
			boost::recursive_mutex::scoped_lock lock(mInfoMutex);
			auto it = mAuthorityInfo.begin();
			for (; it != mAuthorityInfo.end();)
			{
				if (authorities.find(it->second.id) == authorities.end())
				{
					auto tmp = it;
					++it;
					mAuthorityInfo.erase(tmp);
				}
				else
					++it;
			}
		}

		foreach(const std::string& a, authorities)
		{
			if (a.find(MIRA_HIDDEN_AUTHORITY_PREFIX) != std::string::npos && !mShowHidden)
				continue;

			boost::recursive_mutex::scoped_lock lock(mInfoMutex);
			if (mAuthorityInfo.find(a) != mAuthorityInfo.end())
				continue;

			AuthorityItemInfo info;
			info.id = a;
			info.isStarted = AuthorityItemInfo::RunningState::UNKNOWN;
			info.isLocal = MIRA_FW.getAuthorityManager().isLocal(a);
			info.hasUnit = MIRA_FW.getUnitManager()->getUnit(a) != NULL;
			mAuthorityInfo[a] = info;
		}

		// update all authorities
		foreach(const std::string& a, authorities)
		{
			AuthorityItemInfo info;
			if (a.find(MIRA_HIDDEN_AUTHORITY_PREFIX) == std::string::npos || mShowHidden)
			{
				info.id = a;
				try
				{
					SharedAuthority sa = MIRA_FW.getAuthorityManager().getAuthority(a);
					info.status = sa.authority()->getStatusMap();
					try {
						if (sa.authority()->isStarted())
							info.isStarted = AuthorityItemInfo::RunningState::STARTED;
						else
							info.isStarted = AuthorityItemInfo::RunningState::PAUSED;
					}
					catch (XIO& ex) {
						info.isStarted = AuthorityItemInfo::RunningState::UNKNOWN;
					}
					info.isLocal = MIRA_FW.getAuthorityManager().isLocal(a);
					info.hasUnrecoverableFailure = sa.authority()->hasUnrecoverableFailure();
					info.publishedMapping = sa.authority()->getPublishedChannelNames();
					info.subscribedMapping = sa.authority()->getSubscribedChannelNames();
					try {
						info.channelStatus = sa.authority()->getSubscribedChannelStatus();
					}
					catch(XRPC&) {}
					info.interfaces = sa.authority()->getServiceInterfaces();
					info.hasUnit = MIRA_FW.getUnitManager()->getUnit(a) != NULL;
					info.complete = true;
				}
				catch(boost::thread_interrupted&) {
					ThreadMonitor::instance().removeThisThread();
					return;
				}
				catch(...) {
					// If there was an exception, set the internalError flag.
					info.internalError = true;
				}
				// copy back to global map
				boost::recursive_mutex::scoped_lock lock(mInfoMutex);
				mAuthorityInfo[a] = info;
			}
			else
			{
				// not showing authority -> remove if in global map
				boost::recursive_mutex::scoped_lock lock(mInfoMutex);
				mAuthorityInfo.erase(a);
			}
		}
		MIRA_SLEEP(2000);
	}
	if(!ThreadMonitor::isDestroyed())
		ThreadMonitor::instance().removeThisThread();
}

void AuthorityView::updatePropertyWorker()
{
	ThreadMonitor::instance().addThisThread("#AuthorityView PropertyUpdate");
	while (!boost::this_thread::interruption_requested())
	{
		// If there nothing to do, wait and continue
		if (mPropertyAuthorityIDs.empty()) {
			boost::mutex::scoped_lock lock(mPropertyAuthorityIDMutex);

			// Wait max 100ms to ensure, that the thread can be interrupted
			boost::system_time tEndTime =
					boost::get_system_time() + Duration::milliseconds(100);
			if (!mPropertyAuthorityIDCondition.timed_wait(lock, tEndTime))
				continue;
		}
		if (boost::this_thread::interruption_requested())
			break;

		// Copy the worker data
		std::vector<std::string> idVector = mPropertyAuthorityIDs;
		mPropertyAuthorityIDs.clear();

		// Reset the interrupt flag
		mInterruptPropertiesUpdateFlag = false;

		// Collect the properties for all selected authorities
		PropertyUpdateEvent* updateEvent = new PropertyUpdateEvent();
		foreach(const std::string& id, idVector)
		{
			try {
				boost::shared_ptr<PropertyNode> node =
					MIRA_FW.getAuthorityManager().getProperties(id);
				updateEvent->nodes.push_back(node);
				if (mInterruptPropertiesUpdateFlag)
					break;
			} catch(std::exception& ex) {
				MIRA_LOG(ERROR) << "Exception in AuthorityView: " << ex.what() << std::endl;
				// TODO: Anyone who did the threading stuff here: Should we
				//       set the internal error flag as in updateInfoWorker ??
			}
		}

		// send the update event to the Qt thread if there is no pending stop flag
		if (!mInterruptPropertiesUpdateFlag)
			QCoreApplication::postEvent(mUI->tree, updateEvent);
		else
			delete updateEvent;
	}
	if(!ThreadMonitor::isDestroyed())
		ThreadMonitor::instance().removeThisThread();
}

void AuthorityView::onShowHidden(bool show)
{
	mShowHidden = show;
	if (mUI!=NULL)
		updateAuthorityTree();
}

void AuthorityView::onShowRelative(bool show)
{
	mShowRelative = show;
	if (mUI!=NULL)
		updateAuthorityTree();
}

void AuthorityView::onShowInternal(bool show)
{
	mShowInternal = show;
	if (mUI!=NULL)
		updateAuthorityTree();
}

void AuthorityView::timerEvent(QTimerEvent *e)
{
	updateAuthorityTree();
}

void AuthorityView::removeItem(QTreeWidgetItem* item)
{
	QTreeWidgetItem* p = item->parent();
	// if item has no parent (top level item) remove it from tree
	if (p == NULL)
		mUI->tree->takeTopLevelItem(mUI->tree->indexOfTopLevelItem(item));
	else
	{
		// remove it from its parent
		p->takeChild(p->indexOfChild(item));
		// if p is namespace node check if namespace contains other authorities
		// and remove empty namespace items recursively
		AuthorityTreeWidget::Item* nsItem = dynamic_cast<AuthorityTreeWidget::Item*>(p);
		if (nsItem != NULL && nsItem->authorityID.empty() && p->childCount() == 0)
			removeItem(p);
	}
	delete item;
}

void AuthorityView::updateAuthorityTree()
{
	assert(mUI!=NULL);

	boost::recursive_mutex::scoped_lock lock(mInfoMutex);

	// test for all authorities in our tree view if they still exist or should still be shown
	auto i = mAuthorityItems.begin();
	while (i != mAuthorityItems.end())
	{
		if (mAuthorityInfo.count(i->first) > 0 &&
			(i->first.find(MIRA_HIDDEN_AUTHORITY_PREFIX) == std::string::npos || mShowHidden))
		{
			// authority does exist continue
			++i;
			continue;
		}
		// remove item from tree
		removeItem(i->second);
		i = mAuthorityItems.erase(i);
	}

	foreach(const auto& a, mAuthorityInfo)
	{
		if(mAuthorityItems.count(a.first)==0) // new authority?
		{
			if (a.first.find(MIRA_HIDDEN_AUTHORITY_PREFIX) == std::string::npos || mShowHidden)
				insertAuthority(a.second);
		}
		else
			updateAuthority(a.second);
	}
	// update selected one (if only one is selected)
	auto items = mUI->tree->selectedItems();
	if (items.size()!=1)
		return;
	mUI->updateItem(*items.begin());
}

void AuthorityView::insertAuthority(const AuthorityItemInfo& authority)
{
	std::vector<std::string> s;
	boost::split(s, authority.id, boost::is_from_range('/','/'));

	if(s.size()<=1)
		return;

	// search for existing toplevel item where we need to add the
	// authority
	QTreeWidgetItem* root = NULL;
	for(int i=0; i<mUI->tree->topLevelItemCount(); ++i)
	{
		QTreeWidgetItem* it= mUI->tree->topLevelItem(i);
		if(it->text(0).toLocal8Bit().data() == s[1]) {
			// we can only use the matching item if we are still at namespace level
			// and the item is a namespace (not authority)
			// (otherwise we create hierarchical authority items, which will mess up removal later)
			AuthorityTreeWidget::Item* aitem = dynamic_cast<AuthorityTreeWidget::Item*>(it);
			assert(aitem!=NULL);
			if ((s.size() > 2) && (aitem->authorityID.empty())) {
				root = it;
				break;
			}
		}
	}

	if(root==NULL) { // nothing was found -> create new top level item
		root = new AuthorityTreeWidget::Item(mUI->tree,s[1]);
		mUI->tree->insertTopLevelItem(0, root);
		root->setExpanded(true);
		// set italic font per default
		QFont font = root->font(0);
		font.setItalic(true);
		root->setFont(0,font);
		root->setIcon(0, QIcon(":/icons/Namespace.ico"));
	}

	// now search the sub-items recursively
	QTreeWidgetItem* parent = root;
	for(size_t i=2; i<s.size(); ++i)
	{
		// check if we already have a item with the name s[i]
		QTreeWidgetItem* item = NULL;
		for(int j=0; j<parent->childCount(); ++j)
		{
			QTreeWidgetItem* it = parent->child(j);
			if(it->text(0).toLocal8Bit().data() == s[i]) {
				// we can only use the matching item if we are still at namespace level
				// and the item is a namespace (not authority)
				// (otherwise we create hierarchical authority items, which will mess up removal later)
				AuthorityTreeWidget::Item* aitem = dynamic_cast<AuthorityTreeWidget::Item*>(it);
				assert(aitem!=NULL);
				if ((s.size()-1 > i) && (aitem->authorityID.empty())) {
					item = it;
					break;
				}
			}
		}

		if(item==NULL) { // not found -> create new item
			item =  new AuthorityTreeWidget::Item(parent,s[i]);
			parent->addChild(item);
			// set italic font per default
			QFont font = item->font(0);
			item->setExpanded(true);
			font.setItalic(true);
			item->setFont(0,font);
			item->setIcon(0, QIcon(":/icons/Namespace.ico"));
		}

		parent = item; // prepare next round
	}

	// when we reach here parent contains the leaf-node for our id
	AuthorityTreeWidget::Item* item = dynamic_cast<AuthorityTreeWidget::Item*>(parent);
	assert(item!=NULL);

	mAuthorityItems.insert(std::make_pair(authority.id, item));

	QFont font = item->font(0);
	font.setItalic(false);
	item->setFont(0,font);
	item->setExpanded(false);
	item->authorityID = authority.id;

	updateAuthority(authority);
}

AuthorityItemInfo AuthorityView::getAuthorityInfo(const std::string authorityID)
{
	boost::recursive_mutex::scoped_lock lock(mInfoMutex);
	AuthorityInfoMap::const_iterator iter =	mAuthorityInfo.find(authorityID);
	if (iter != mAuthorityInfo.end())
		return iter->second;
	else
		return AuthorityItemInfo();
}

StatusManager::StatusMap::const_iterator findNextDMGroup(StatusManager::StatusMap::const_iterator begin,
                                                         StatusManager::StatusMap::const_iterator end)
{
	StatusManager::StatusMap::const_iterator i;
	for(i = begin; i!=end && i->first == begin->first; ++i) {}
	return i;
}

QColor getColorForStatus(Status::StatusMode status)
{
	if (status == Status::BOOTUP)
		return QColor(191, 191, 255);
	if (status == Status::WARNING)
		return QColor(255, 255, 127);
	if (status == Status::ERROR)
		return QColor(255, 191, 191);
	if (status == Status::RECOVER)
		return QColor(255, 191, 255);
	return QColor(191, 255, 191);
}

void setBackground(QTreeWidgetItem* item, const QColor& color)
{
	item->setBackground(0, QBrush(color));
	item->setBackground(1, QBrush(color));
}

void AuthorityView::updateAuthority(const AuthorityItemInfo& authority)
{
	AuthorityTreeWidget::Item* authorityItem = dynamic_cast<AuthorityTreeWidget::Item*>(mAuthorityItems[authority.id]);
	if (authorityItem->infoItem != NULL)
	{
		removeItem(authorityItem->infoItem);
		authorityItem->infoItem = NULL;
	}

	if (!authority.complete) {
		authorityItem->setIcon(0, QIcon());
		setBackground(authorityItem, QColor(230, 230, 230));

		QString state = "Waiting for authority info...";
		authorityItem->setText(1, state);
		return;
	}

	// Internal error
	if (authority.internalError) {
		//authorityItem->setIcon(0, QIcon(":/icons/StatusError.png"));
		authorityItem->setIcon(0, QIcon());
		setBackground(authorityItem, QColor(191, 191, 191));

		QString state = "Internal error. Incompatible version?";
		authorityItem->setText(1, state);
		return;
	}

	// check channels
	// if we have no subscribed channels remove child
	if (authority.subscribedMapping.size() == 0 && authorityItem->subscribedChannelItem != NULL)
	{
		removeItem(authorityItem->subscribedChannelItem);
		authorityItem->subscribedChannelItem = NULL;
		// clear the map, otherwise they will be deleted again below...
		authorityItem->subscribedItems.clear();
	}
	// if we have no published channels remove child
	if (authority.publishedMapping.size() == 0 && authorityItem->publishedChannelItem != NULL)
	{
		removeItem(authorityItem->publishedChannelItem);
		authorityItem->publishedChannelItem = NULL;
		// clear the map, otherwise they will be deleted again below...
		authorityItem->publishedItems.clear();
	}
	// if we have no service interfaces remove child
	if (authority.interfaces.size() == 0 && authorityItem->serviceInterfaceItem != NULL)
	{
		removeItem(authorityItem->serviceInterfaceItem);
		authorityItem->serviceInterfaceItem = NULL;
		// clear the map, otherwise they will be deleted again below...
		authorityItem->interfaceItems.clear();
	}
	// first remove all channels and service interfaces that are no longer published or subscribed
	std::map<std::string, QTreeWidgetItem*>::iterator sit = authorityItem->subscribedItems.begin();
	for(; sit != authorityItem->subscribedItems.end();)
	{
		if (authority.subscribedMapping.count(sit->first) == 0)
		{
			std::map<std::string, QTreeWidgetItem*>::iterator tmp = sit;
			++sit;
			removeItem(tmp->second);
			authorityItem->subscribedItems.erase(tmp);
			continue;
		}
		++sit;
	}
	std::map<std::string, QTreeWidgetItem*>::iterator pit = authorityItem->publishedItems.begin();
	for(; pit != authorityItem->publishedItems.end();)
	{
		if (authority.publishedMapping.count(pit->first) == 0)
		{
			std::map<std::string, QTreeWidgetItem*>::iterator tmp = pit;
			++pit;
			removeItem(tmp->second);
			authorityItem->publishedItems.erase(tmp);
			continue;
		}
		++pit;
	}
	std::map<std::string, QTreeWidgetItem*>::iterator svit = authorityItem->interfaceItems.begin();
	for(; svit != authorityItem->interfaceItems.end();)
	{
		if (authority.interfaces.count(svit->first) == 0)
		{
			std::map<std::string, QTreeWidgetItem*>::iterator tmp = svit;
			++svit;
			removeItem(tmp->second);
			authorityItem->interfaceItems.erase(tmp);
			continue;
		}
		++svit;
	}
	// now add all channels and interfaces that are not yet added and update them
	if (authority.subscribedMapping.size() > 0)
	{
		if (authorityItem->subscribedChannelItem == NULL)
		{
			// add a child node for the subscribed channels info
			authorityItem->subscribedChannelItem = new QTreeWidgetItem();
			authorityItem->subscribedChannelItem->setText(0, "Subscribed");
			authorityItem->addChild(authorityItem->subscribedChannelItem);
		}
		Status::StatusMode channelStatus = Status::OK;
		foreach(const auto& channel, authority.subscribedMapping)
		{
			if (authorityItem->subscribedItems.count((channel.first)) == 0)
			{
				QTreeWidgetItem* channelItem = new QTreeWidgetItem();
				authorityItem->subscribedChannelItem->addChild(channelItem);
				authorityItem->subscribedItems.insert(std::make_pair(channel.first, channelItem));
			}
			QTreeWidgetItem* channelItem = authorityItem->subscribedItems[channel.first];
			setBackground(channelItem, getColorForStatus(Status::OK));
			channelItem->setText(0, channel.first.c_str());
			if (mShowInternal) {
				channelItem->setText(0, channel.second.c_str());
			} else if (mShowRelative) {
				try {
					channelItem->setText(0, relativizePath(channel.first, Path(authority.id).parent_path()).c_str());
				} catch (...) {}
			}

			try {
				Status status = authority.channelStatus.at(channel.first);

				channelItem->setText(1, QString::fromStdString(status.category));
				setBackground(channelItem, getColorForStatus(status.mode));
				channelItem->setToolTip(1, QString::fromStdString(status.message));

				channelStatus = std::max(channelStatus, status.mode);
			}
			catch(std::out_of_range&) {
				channelItem->setText(1, "UNKNOWN");
				setBackground(channelItem, QColor(191, 191, 191));
				channelItem->setToolTip(1, "");
			}
		}
		setBackground(authorityItem->subscribedChannelItem, getColorForStatus(channelStatus));
	}
	if (authority.publishedMapping.size() > 0)
	{
		if (authorityItem->publishedChannelItem == NULL)
		{
			authorityItem->publishedChannelItem = new QTreeWidgetItem();
			authorityItem->publishedChannelItem->setText(0, "Published");
			authorityItem->addChild(authorityItem->publishedChannelItem);
		}
		foreach(const auto& channel, authority.publishedMapping)
		{
			if (authorityItem->publishedItems.count((channel.first)) == 0)
			{
				QTreeWidgetItem* channelItem = new QTreeWidgetItem();
				setBackground(channelItem, getColorForStatus(Status::OK));
				authorityItem->publishedChannelItem->addChild(channelItem);
				authorityItem->publishedItems.insert(std::make_pair(channel.first, channelItem));
			}
			QTreeWidgetItem* channelItem = authorityItem->publishedItems[channel.first];
			channelItem->setText(0, channel.first.c_str());
			if (mShowInternal) {
				channelItem->setText(0, channel.second.c_str());
			} else if (mShowRelative) {
				try {
					channelItem->setText(0, relativizePath(channel.first, Path(authority.id).parent_path()).c_str());
				} catch (...) {}
			}
		}
		setBackground(authorityItem->publishedChannelItem, getColorForStatus(Status::OK));
	}
	if (authority.interfaces.size() > 0)
	{
		if (authorityItem->serviceInterfaceItem == NULL)
		{
			authorityItem->serviceInterfaceItem = new QTreeWidgetItem();
			authorityItem->serviceInterfaceItem->setText(0, "RPC Interfaces");
			authorityItem->addChild(authorityItem->serviceInterfaceItem);
		}
		foreach(const std::string& interface, authority.interfaces)
		{
			if (authorityItem->interfaceItems.count((interface)) == 0)
			{
				QTreeWidgetItem* interfaceItem = new QTreeWidgetItem();
				interfaceItem->setText(0, interface.c_str());
				setBackground(interfaceItem, getColorForStatus(Status::OK));
				authorityItem->serviceInterfaceItem->addChild(interfaceItem);
				authorityItem->interfaceItems.insert(std::make_pair(interface, interfaceItem));
			}
		}
		setBackground(authorityItem->serviceInterfaceItem, getColorForStatus(Status::OK));
	}

	Status::StatusMode overallStatus = StatusManager::getOverallStatus(authority.status.begin(),
	                                                                   authority.status.end());
	setBackground(authorityItem, getColorForStatus(overallStatus));

	// everything is fine
	if (overallStatus == Status::OK) {
		authorityItem->setIcon(0, QIcon(":/icons/StatusOK.png"));

		QString state;
		switch (authority.isStarted) {
			case AuthorityItemInfo::RunningState::STARTED: state = "OK"; break;
			case AuthorityItemInfo::RunningState::PAUSED : state = "PAUSED"; break;
			case AuthorityItemInfo::RunningState::UNKNOWN: state = "UNKNOWN"; break;
			default                                      : state = "INVALID"; break;
		}

		authorityItem->setText(1, state);
		if (!authorityItem->userExpanded)
			authorityItem->setExpanded(false);
		return;
	}
	// if we reach here the status is not OK
	// add a child node for the status info
	authorityItem->infoItem = new QTreeWidgetItem();
	authorityItem->infoItem->setText(0, "Info");
	setBackground(authorityItem->infoItem, getColorForStatus(overallStatus));
	authorityItem->addChild(authorityItem->infoItem);
	// expand authority to show the error
	authorityItem->autoExpanded = true;
	authorityItem->setExpanded(true);
	QString overAllStatusTxt;
	if (overallStatus == Status::ERROR)
	{
		authorityItem->setIcon(0, QIcon(":/icons/StatusError.png"));
		overAllStatusTxt = "ERROR";
	}
	else if (overallStatus == Status::WARNING)
	{
		authorityItem->setIcon(0, QIcon(":/icons/StatusWarning.png"));
		overAllStatusTxt = "WARNING";
	}
	else if (overallStatus == Status::RECOVER)
	{
		authorityItem->setIcon(0, QIcon(":/icons/StatusRecover.png"));
		overAllStatusTxt = "RECOVER";
	}
	else
	{
		authorityItem->setIcon(0, QIcon(":/icons/StatusBooting.png"));
		overAllStatusTxt = "BOOTING UP";
	}

	// iterate over all modules and their status
	auto firstOfDMGroup = authority.status.begin();

	while (firstOfDMGroup != authority.status.end())
	{
		auto lastOfDMGroup = findNextDMGroup(firstOfDMGroup, authority.status.end());
		QTreeWidgetItem* diagnosticItem = new QTreeWidgetItem();
		diagnosticItem->setText(0, QString::fromUtf8(firstOfDMGroup->first.c_str()));
		authorityItem->infoItem->addChild(diagnosticItem);
		Status::StatusMode mode = StatusManager::getOverallStatus(firstOfDMGroup, lastOfDMGroup);
		setBackground(diagnosticItem, getColorForStatus(mode));
		if (mode == Status::OK) {
			diagnosticItem->setText(1, "OK");
			continue;
		}
		QString statusTxt;
		if (mode == Status::ERROR)
			statusTxt = "ERROR";
		else if (mode == Status::WARNING)
			statusTxt = "WARNING";
		else if (mode == Status::RECOVER)
			statusTxt = "RECOVER";
		else
			statusTxt = "BOOTING UP";
		for(auto i=firstOfDMGroup; i!=lastOfDMGroup; ++i)
		{
			QTreeWidgetItem* item = new QTreeWidgetItem();
			item->setText(0, QString::fromUtf8(i->second.category.c_str()));
			setBackground(item, getColorForStatus(i->second.mode));
			if (i->second.trText.empty())
				item->setText(1, QString::fromUtf8(i->second.message.c_str()));
			else
				item->setText(1, QString::fromUtf8((i->second.trText+": "+i->second.message).c_str()));
			item->setExpanded(true);
			diagnosticItem->addChild(item);
		}
		diagnosticItem->setText(1, statusTxt);
		diagnosticItem->setExpanded(true);
		firstOfDMGroup = lastOfDMGroup;
	}
	authorityItem->infoItem->setExpanded(true);
	if (authority.hasUnrecoverableFailure)
		overAllStatusTxt = "UNRECOVERABLE ERROR";
	authorityItem->setText(1, overAllStatusTxt);
}

void AuthorityView::triggerUpdateProperties(const std::vector<std::string>& ids)
{
	boost::mutex::scoped_lock lock(mPropertyAuthorityIDMutex);
	mPropertyAuthorityIDs = ids;
	mPropertyAuthorityIDCondition.notify_one();
}

void AuthorityView::stopUpdateProperties()
{
	mInterruptPropertiesUpdateFlag = true;
}

///////////////////////////////////////////////////////////////////////////////

AuthorityView::UI::UI(AuthorityView* parent) :
	QWidget(parent),
	mParent(parent)
{
	setupUi();
}

void AuthorityView::UI::setupUi()
{
	QVBoxLayout* vLayout = new QVBoxLayout(this);
	tree = new AuthorityTreeWidget(this);

	mFilter = new TreeViewFilter(tree,this);
	mFilter->setAlwaysActive(true);
	mFilter->setExpandChildren(false);
	mFilter->useFilterRole(true);

	vLayout->setSpacing(0);
	vLayout->setContentsMargins(0,3,0,0);
	vLayout->addWidget(mFilter);
	vLayout->addWidget(tree);

	QHBoxLayout* hLayout = new QHBoxLayout();

	startBtn = new QToolButton(this);
	startBtn->setIcon(QIcon(":/icons/PowerOn.png"));
	startBtn->setAutoRaise(true);
	startBtn->setToolTip("Resume the selected authority");
	hLayout->addWidget(startBtn);

	stopBtn = new QToolButton(this);
	stopBtn->setIcon(QIcon(":/icons/PowerOff.png"));
	stopBtn->setAutoRaise(true);
	stopBtn->setToolTip("Stop the selected authority");
	hLayout->addWidget(stopBtn);

	hLayout->addStretch();

	saveBtn = new QToolButton(this);
	saveBtn->setIcon(QIcon(":/icons/FileSave.png"));
	saveBtn->setToolTip("Save the configuration of the selected authority");
	saveBtn->setAutoRaise(true);
	saveBtn->setEnabled(false);
	hLayout->addWidget(saveBtn);

	docBtn = new QToolButton(this);
	docBtn->setIcon(QIcon(":/icons/Dictionary.png"));
	docBtn->setToolTip("Generate documentation of the selected authority");
	docBtn->setAutoRaise(true);
	docBtn->setEnabled(false);
	hLayout->addWidget(docBtn);

	vLayout->addLayout(hLayout);

	connect(startBtn, SIGNAL(clicked()), this, SLOT(onStart()));
	connect(stopBtn, SIGNAL(clicked()), this, SLOT(onStop()));
	connect(saveBtn, SIGNAL(clicked()), this, SLOT(onSave()));
	connect(docBtn, SIGNAL(clicked()), this, SLOT(onGenerateDocumentation()));
	connect(tree, SIGNAL(currentItemChanged(QTreeWidgetItem*, QTreeWidgetItem*)),
	        this, SLOT(itemChanged(QTreeWidgetItem*, QTreeWidgetItem*)));
	connect(tree, SIGNAL(itemExpanded(QTreeWidgetItem*)),
	        this, SLOT(itemExpanded(QTreeWidgetItem*)));
	connect(tree, SIGNAL(itemCollapsed(QTreeWidgetItem*)),
	        this, SLOT(itemCollapsed(QTreeWidgetItem*)));
}

void AuthorityView::UI::collectAuthorityIDs(QTreeWidgetItem* iItem,
                                            std::vector<std::string>& oIDVector)
{
	if (iItem == NULL)
		return;

	AuthorityTreeWidget::Item* item = dynamic_cast<AuthorityTreeWidget::Item*>(iItem);
	while(item == NULL)
	{
		iItem = iItem->parent();
		item = dynamic_cast<AuthorityTreeWidget::Item*>(iItem);
	}
	assert(item!=NULL);

	try
	{
		if (item->authorityID.empty())
		{
			for (int i=0; i<item->childCount(); ++i)
			{
				AuthorityTreeWidget::Item* subItem = dynamic_cast<AuthorityTreeWidget::Item*>(item->child(i));
				assert(subItem!=NULL);
				if (subItem->authorityID.empty())
				{
					collectAuthorityIDs(subItem, oIDVector);
					continue;
				}
				oIDVector.push_back(subItem->authorityID);
			}
		}
		else
			oIDVector.push_back(item->authorityID);
	}
	catch(...)
	{
	}
}

void AuthorityView::UI::itemChanged(QTreeWidgetItem* current, QTreeWidgetItem* previous)
{
	mParent->setFocus();
	mParent->stopUpdateProperties();
	QCoreApplication::postEvent(tree, new QEvent(PROPERTY_CLEAR_EVENT_ID));

	// update the item itself
	updateItem(current);

	// collect the IDs for all selected authorities
	std::vector<std::string> authorityIDs;
	collectAuthorityIDs(current, authorityIDs);

	mParent->triggerUpdateProperties(authorityIDs);
}

void AuthorityView::UI::itemCollapsed(QTreeWidgetItem* item)
{
	AuthorityTreeWidget::Item* authorityItem = dynamic_cast<AuthorityTreeWidget::Item*>(item);
	if (authorityItem == NULL || authorityItem->authorityID.empty())
		return;
	authorityItem->userExpanded = false;
	authorityItem->autoExpanded = false;
}

void AuthorityView::UI::itemExpanded(QTreeWidgetItem* item)
{
	AuthorityTreeWidget::Item* authorityItem = dynamic_cast<AuthorityTreeWidget::Item*>(item);
	if (authorityItem == NULL || authorityItem->authorityID.empty())
		return;
	if (!authorityItem->autoExpanded)
		authorityItem->userExpanded = true;
}

void AuthorityView::UI::onSave()
{
	auto items = tree->selectedItems();
	if (items.size()!=1)
		return;
	std::string a = getAuthorityFromItem(*items.begin());
	if (a.empty())
		return;
	MicroUnitPtr m = MIRA_FW.getUnitManager()->getUnit(a);
	if (!m)
		return;

	static QString saveDir = ".";
	QString name = QtUtils::getSaveFileName(this, tr("Save configuration"), saveDir,
	                                        tr("Config (*.xml)"),
	                                        NULL, QFileDialog::DontUseNativeDialog, QStringList("xml"));
	if(name.isNull() || name.isEmpty())
		return;
	XMLDom xml;
	XMLDom::iterator i = xml.root();
	XMLSerializer s(i);
	s.serialize("unit", m, "");
	i.find("unit").add_attribute("id", m->getID());
	xml.saveToFile(name.toStdString());

	// memorize the selected location
	saveDir = QFileInfo(name).path();
}

void AuthorityView::UI::onGenerateDocumentation()
{
	auto items = tree->selectedItems();
	if (items.size()!=1)
		return;
	std::string a = getAuthorityFromItem(*items.begin());
	if (a.empty())
		return;
	MicroUnitPtr m = MIRA_FW.getUnitManager()->getUnit(a);
	if (!m)
		return;

	QString name = QtUtils::getSaveFileName(this, tr("Save documentation"), QString(),
	                                        tr("Documentation (*.dox)"),
	                                        NULL, QFileDialog::DontUseNativeDialog, QStringList("dox"));
	if(name.isNull() || name.isEmpty())
		return;

	std::ofstream doc(name.toStdString().c_str());
	AbstractAuthority::ChannelNameMapping published = m->getPublishedChannelNames();
	AbstractAuthority::ChannelNameMapping subscribed = m->getSubscribedChannelNames();

	if (!published.empty() || !subscribed.empty())
	{
		doc << "\\subsection Channels" << std::endl;
		if (!published.empty())
		{
			doc << "\\subsubsection Published" << std::endl;
			foreach(const auto& p, published)
			{
				doc << "- \\channel{" << p.second << ","
						<< MIRA_FW.getChannelManager().getTypename(p.first) << "}" << std::endl;
				doc << "  Description" << std::endl;
			}
		}
		if (!subscribed.empty())
		{
			doc << "\\subsubsection Subscribed" << std::endl;
			foreach(const auto& s, subscribed)
			{
				doc << "- \\channel{" << s.second << ","
						<< MIRA_FW.getChannelManager().getTypename(s.first) << "}" << std::endl;
				doc << "  Description" << std::endl;
			}
		}
	}

	MetaTypeDatabase db;
	MetaSerializer ms(db);
	TypeMetaPtr meta = ms.addMeta(m);
	MetaTypeDatabase::const_iterator i = db.find(m->getClass().getTypename());
	if (i != db.end())
	{
		CompoundMetaPtr mp = i->second;
		if (!mp->interfaces.empty() || !mp->methods.empty())
		{
			doc << std::endl;
			doc << "\\subsection Services" << std::endl;
			doc << "\\subsubsection Published" << std::endl;
			foreach(const std::string& i, mp->interfaces)
				doc << "- \\service{" << i << "}" << std::endl;
			foreach(MethodMetaPtr mmp, mp->methods)
			{
				doc << "  " << mmp->returnType->toString() << " " << mmp->name << "(";
				for(auto p = mmp->parameters.begin(); p!=mmp->parameters.end(); ++p)
				{
					if(p!=mmp->parameters.begin())
						doc << ",";
					doc << p->type->toString();
					if (!p->name.empty())
						doc << " " << p->name;
				}
				doc << ")" << std::endl;
				doc << "    " << mmp->comment << std::endl;
				for(auto p = mmp->parameters.begin(); p!=mmp->parameters.end(); ++p)
				{
					if (!p->name.empty() && !p->description.empty())
						doc << "    " + p->name << " \t: " << p->description << std::endl;
				}
			}
		}
		if (!mp->members.empty())
		{
			doc << std::endl;
			doc << "\\subsection Parameters" << std::endl;
			foreach(const auto& member, mp->members)
			{
				if (member.name.rfind("@", 0) == 0) // first char is a '@'? -> internal reflection stuff
					continue;
					
				// TODO distinguish between member and property
				doc << "- \\member{" + member.name << "," << member.type->toString() << "}" << std::endl;
				doc << "  " << member.comment << std::endl;
			}
		}
	}
	doc.close();
}

void AuthorityView::UI::onStart()
{
	auto items = tree->selectedItems();
	if (items.size()!=1)
		return;
	std::string a = getAuthorityFromItem(*items.begin());
	if (a.empty())
		return;

	try
	{
		MIRA_FW.getAuthorityManager().start(a);
	}
	catch(...)
	{
	}
}

void AuthorityView::UI::onStop()
{
	auto items = tree->selectedItems();
	if (items.size()!=1)
		return;
	std::string a = getAuthorityFromItem(*items.begin());
	if (a.empty())
		return;

	try
	{
		MIRA_FW.getAuthorityManager().stop(a);
	}
	catch(...)
	{
	}
}

void AuthorityView::UI::updateItem(QTreeWidgetItem* item)
{
	std::string a = getAuthorityFromItem(item);
	AuthorityItemInfo info = mParent->getAuthorityInfo(a);
	if (a.empty()) {
		startBtn->setEnabled(false);
		stopBtn->setEnabled(false);
		saveBtn->setEnabled(false);
		docBtn->setEnabled(false);
	}
	else
	{
		if (info.hasUnit)
		{
			saveBtn->setEnabled(true);
			docBtn->setEnabled(true);
		}
		else
		{
			saveBtn->setEnabled(false);
			docBtn->setEnabled(false);
		}

		if (info.hasUnrecoverableFailure)
		{
			startBtn->setEnabled(false);
			stopBtn->setEnabled(false);
		}
		else
		{
			switch(info.isStarted) {
			case AuthorityItemInfo::RunningState::STARTED:
				startBtn->setEnabled(false);
				stopBtn->setEnabled(true);
				break;
			case AuthorityItemInfo::RunningState::PAUSED:
				startBtn->setEnabled(true);
				stopBtn->setEnabled(false);
				break;
			default: // UNKNOWN or undefined --> leave them enabled, we can always try starting/stopping
				startBtn->setEnabled(true);
				stopBtn->setEnabled(true);
			}
		}
	}
}

std::string AuthorityView::UI::getAuthorityFromItem(QTreeWidgetItem* item)
{
	AuthorityTreeWidget::Item* a = dynamic_cast<AuthorityTreeWidget::Item*>(item);
	if (a == NULL || a->authorityID.empty())
		return "";
	return a->authorityID;
}

///////////////////////////////////////////////////////////////////////////////

}
