/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file WrenchVisualization.C
 *    3D visualization plugin for miracenter to visualize wrench data.
 *
 * @author Steffen Mueller
 * @date   2022/02/07
 */


#include <visualization/Visualization3DBasic.h>

#include <limits>
#include <vector>
#include <boost/shared_ptr.hpp>
#include <boost/math/constants/constants.hpp>

#include <robot/Wrench.h>
#include <math/YawPitchRoll.h>
#include <widgets/OgreUtils.h>
#include <serialization/adapters/OGRE/OgreColourValue.h>
#include <visualization/3d/MeshObject.h>
#include <serialization/DefaultInitializer.h>

using namespace mira;
using namespace mira::robot;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

template<typename ScalarType>
class WrenchVisualization : public Visualization3DBasic<mira::robot::Wrench3<ScalarType>>
{
private:

	typedef Visualization3DBasic<mira::robot::Wrench3<ScalarType>> Base;
	typedef boost::shared_ptr<MeshObject> ConeObjectPtr;

public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		mira::serialization::VersionType v = MIRA_REFLECT_VERSION(r, 1);
		MIRA_REFLECT_BASE_IF_VERSION_ATLEAST(r, Base, v, 1);

		r.property("Scale", mScale, "Scale factor for rendering arrows", 1.);
		r.property("WidthForce",mFWidth,"width of the force arrow",0.02f);
		r.property("WidthTorque",mTWidth,"width of the torque arrow",0.02f);

		r.property("ForceColor",mFColor,"Color of the velocity vector.",Ogre::ColourValue(1.f,0.f,1.f));
		r.property("TorqueColor",mTColor,"Color of the velocity vector.",Ogre::ColourValue(1.f,1.f,0.f));
	}

private:

	bool mEnabled;
	ConeObjectPtr mForceCone;
	ConeObjectPtr mForceShaft;
	ConeObjectPtr mTorqueCone;
	ConeObjectPtr mTorqueShaft;

	float mFWidth;
	float mTWidth;
	float mScale;
	Ogre::ColourValue mFColor;
	Ogre::ColourValue mTColor;

	robot::Wrench3<ScalarType> mWrench;
	std::string mWrenchFrame;

	///////////////////////////////////////////////////////////////////////////////
public:
	WrenchVisualization() : Base("Wrench"),
	                        mEnabled(false)
	{
		MIRA_INITIALIZE_THIS;
	}

	virtual void setupScene(Ogre::SceneManager* mgr, Ogre::SceneNode* node)
	{
		mForceCone=ConeObjectPtr(new MeshObject("Cone.mesh", mgr, node));
		mForceShaft=ConeObjectPtr(new MeshObject("Cylinder.mesh", mgr, node));
		mTorqueCone=ConeObjectPtr(new MeshObject("Cone.mesh", mgr, node));
		mTorqueShaft=ConeObjectPtr(new MeshObject("Cylinder.mesh", mgr, node));
	}

	virtual void setEnabled(bool enabled)
	{
		Visualization3D::setEnabled(enabled);
		mEnabled = enabled;

		mForceCone->setVisible(mEnabled);
		mForceShaft->setVisible(mEnabled);
		mTorqueCone->setVisible(mEnabled);
		mTorqueShaft->setVisible(mEnabled);
	}

protected:

	virtual void dataChanged(ChannelRead<mira::robot::Wrench3<ScalarType>> data)
	{
		mWrench = data->value();
		mWrenchFrame = data->frameID;
		draw();
	}

	virtual void draw()
	{
		setCone(mForceCone,mForceShaft,mWrench.force(0), mWrench.force(1), mWrench.force(2), mFColor,mFWidth);
		mForceCone->setVisible(mEnabled);
		mForceShaft->setVisible(mEnabled);
		setCone(mTorqueCone,mTorqueShaft, mWrench.torque(0), mWrench.torque(1), mWrench.torque(2), mTColor,mTWidth);
		mForceCone->setVisible(mEnabled);
		mForceShaft->setVisible(mEnabled);
	}

	virtual void setCone(ConeObjectPtr cone,ConeObjectPtr shaft,
	                     float x,float y,float z,
	                     const Ogre::ColourValue& color,float width)
	{
		Ogre::Vector3 position(0.f,0.f,0.f);
		Point3f endpoint=mScale*Point3f(x,y,z);
		Ogre::Vector3 positionCone(endpoint.x(), endpoint.y(), endpoint.z());
		Eigen::Quaternionf rot = Eigen::Quaternionf::FromTwoVectors(Eigen::Vector3f::Unit(1),endpoint.normalized());
		Ogre::Quaternion ogreRot(rot.w(), rot.x(), rot.y(), rot.z());
		Ogre::Quaternion rotOffset;
		cone->setScale(Ogre::Vector3(2 * width,
		                             3 * width,
		                             2 * width));
		cone->setPosition(positionCone);
		cone->setOrientation(ogreRot);
		cone->setColor(color);
		shaft->setScale(Ogre::Vector3(width,
		                              endpoint.norm(),
		                              width));
		shaft->setPosition(position);
		shaft->setOrientation(ogreRot);
		shaft->setColor(color);
	}
};

///////////////////////////////////////////////////////////////////////////////

class Wrench3fVisualization : public WrenchVisualization<float>
{
MIRA_META_OBJECT(Wrench3fVisualization,
                ("Name", "Wrench3f")
                ("Category", "RobotDataTypes")
                ("Description", "Visualization of a float wrench."));
};

///////////////////////////////////////////////////////////////////////////////

class Wrench3dVisualization : public WrenchVisualization<double>
{
MIRA_META_OBJECT(Wrench3dVisualization,
                ("Name", "Wrench3d")
                ("Category", "RobotDataTypes")
                ("Description", "Visualization of a double wrench."));
};

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::Wrench3fVisualization, mira::Visualization3D );
MIRA_CLASS_SERIALIZATION(mira::Wrench3dVisualization, mira::Visualization3D );
