/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RobotModel.h
 *    Base class for robot models.
 *
 * @author Erik Einhorn
 * @date   2011/05/27
 */

#include <transform/Pose.h>
#include <transform/Velocity.h>

#include <model/RigidModel.h>

#include <robot/Trajectory.h>

#ifndef _MIRA_ROBOTMODEL_H_
#define _MIRA_ROBOTMODEL_H_

namespace mira { namespace robot {

///////////////////////////////////////////////////////////////////////////////

/**
 * Base class for all robot models, such as DifferentialRobotModel.
 * Each robot model is a RigidModel that provides geometric information
 * for collision checks and visualization. Moreover, most robot models
 * additionally provide kinematic information like max. velocities, etc.
 */
class RobotModel : public mira::model::RigidModel
{
	MIRA_ABSTRACT_OBJECT(RobotModel)

public:

	/**
	 * Computes the local movement of the robot if it moves with the specified
	 * velocity v for the specified time dt. The movement is returned as
	 * relative pose in relation to the robots movement center, which is
	 * assumed to be (0,0,0).
	 * @param[in] v  The measured velocity.
	 * @param[in] dt The time interval in [s]. It must be >= 0.0f.
	 * @return The delta pose.
	 */
	virtual Pose2 localKinematics(const Velocity2& v, float dt) const = 0;

	/**
	 * Computes the movement of the robot if it moves with the specified
	 * velocity v for the specified time dt.
	 * The movement is returned as pose in relation to the specified pose p0.
	 * @param[in] v  The velocity of the robot.
	 * @param[in] dt The delta time.
	 * @param[in] p0 The original pose.
	 * @return The movement as a pose in relative to pose p0.
	 */
	Pose2 globalKinematics(const Velocity2& v, float dt, const Pose2& p0) const
	{
		Pose2 dp = localKinematics(v, dt); // delta pose
		float sinphi0=sin(p0.phi());
		float cosphi0=cos(p0.phi());
		return Pose2(cosphi0*dp.x() - sinphi0*dp.y() + p0.x(),
		             sinphi0*dp.x() + cosphi0*dp.y() + p0.y(),
		             dp.phi()+p0.phi());
	}

	/**
	 * Calculate the distance that is needed for braking to stand still, if
	 * the robot moves with the specified velocity v.
	 * @param[in] v The current velocity.
	 * @return The estimated stopping distance.
	 */
	virtual float predictStandstillDistance(const Velocity2& v) const = 0;

	/**
	 * Calculate the rotation that is needed for braking to stand still, if
	 * the robot moves with the specified velocity v.
	 * @param[in] v The current velocity.
	 * @return The estimated stopping rotation.
	 */
	virtual float predictStandstillRotation(const Velocity2& v) const = 0;

	/**
	 * Generates a trajectory by accelerating/decelerating starting at velocity v
	 * for lookAheadTime time up to the velocity targetV.
	 * The resulting trajectory will contain samples+1 trajectory samples starting
	 * at position 0,0 with velocity v.
	 * It uses the acceleration limits defined by the robot model.
	 */
	virtual PoseVelocityTrajectory generateTrajectory(Velocity2 v, const Velocity2& targetV,
	                                                  float lookAheadTime, int samples) = 0;
};

/// A typedef for a shared pointer to a RobotModel
typedef boost::shared_ptr<RobotModel> RobotModelPtr;

///////////////////////////////////////////////////////////////////////////////

} } // namespace

#endif
