/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ProbabilisticMotionModel.h
 *    Base class for probabilistic motion models.
 *
 * @author Tim Langner
 * @date   2013/02/04
 */

#ifndef _MIRA_PROBABILISTICMOTIONMODEL_H_
#define _MIRA_PROBABILISTICMOTIONMODEL_H_

#include <transform/Pose.h>

namespace mira { namespace robot {

///////////////////////////////////////////////////////////////////////////////

/**
 * Base class for probabilistic motion models.
 */
class ProbabilisticMotionModel
{
public:

	virtual ~ProbabilisticMotionModel() {}

	/**
	 * @param[in] startPose The starting position for applying the sampled motion
	 * @param[in] p0        The last observed pose
	 * @param[in] p1        The current observed pose
	 * @return Starting position updated by the sampled motion
	 */
	virtual Pose2 sampleMotion(const Pose2& startPose, const Pose2& p0, const Pose2& p1) const = 0;

	/**
	 * Like the above sampleMotion but samples multiple poses at once.
	 * Default implementation uses a naive approach by calling the above sampleMotion
	 * for each start pose. However you should re-implement this method in a derived class
	 * to speed up the sampling process.
	 * @param[in] startPoses The starting positions for applying the sampled motion
	 * @param[in] p0    The last observed pose
	 * @param[in] p1    The current observed pose
	 * @param[out] sampledPoses Starting positions updated by the sampled motion
	 *
	 */
	virtual void sampleMotion(const std::vector<Pose2>& startPoses,
	                          const Pose2& p0, const Pose2& p1, std::vector<Pose2>& sampledPoses) const
	{
		sampledPoses.resize(startPoses.size());
		for(std::size_t i=0; i<startPoses.size(); ++i)
			sampledPoses[i] = sampleMotion(startPoses[i], p0, p1);
	}
};

typedef boost::shared_ptr<ProbabilisticMotionModel> ProbabilisticMotionModelPtr;

///////////////////////////////////////////////////////////////////////////////

}} // namespace

#endif
