/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationPlotView.C
 *    .
 *
 * @author Tim Langner
 * @date   2011/07/26
 */

#include <views/VisualizationPlotView.h>

#include <QMenu>
#include <QPen>
#include <QPainter>

#include <qwt_legend.h>
#include <qwt_plot_canvas.h>
#include <qwt_plot_layout.h>
#include <qwt_plot_panner.h>
#include <qwt_plot_magnifier.h>

#include <widgets/QtUtils.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

VisualizationPlotView::VisualizationPlotView() :
	mLegendEnabled(true),
	mAcceptingUpdates(true),
	mSaveContentMinPictureSize(2048),
	ui(NULL)
{
	mAutoScaleY[0] = true;
	mAutoScaleY[1] = true;
	mYMax[0] = 1.0;
	mYMax[1] = 1.0;
	mYMin[0] = 0.0;
	mYMin[1] = 0.0;

	setUpdateInterval(40);
}

QwtPlot* VisualizationPlotView::getPlot()
{
	return ui;
}

void VisualizationPlotView::setupVisualizations()
{
	// initialize all existing visualizations
	foreach(Visualization* vis, getVisualizations())
		vis->init(this);
}

void VisualizationPlotView::updateAxes()
{
	bool leftUsed = false;
	bool rightUsed = false;

	foreach(Visualization* vis, getVisualizations()) {
		VisualizationPlot* v = dynamic_cast<VisualizationPlot*>(vis);
		assert(v!=NULL);

		if (v->getYAxis() == QwtPlot::yLeft) {
			leftUsed = true;
		} else if (v->getYAxis() == QwtPlot::yRight) {
			rightUsed = true;
		}
	}

	ui->enableAxis(QwtPlot::yLeft, leftUsed);
	ui->enableAxis(QwtPlot::yRight, rightUsed);

	foreach(Visualization* vis, getVisualizations()) {
		VisualizationPlot* v = dynamic_cast<VisualizationPlot*>(vis);
		assert(v!=NULL);

		foreach (auto item, v->getPlotItems())
		{
			if (!item)
				continue;

			if (leftUsed && rightUsed) {
				QString lr = (v->getYAxis() == QwtPlot::yLeft) ? " (l)" : " (r)";
				item->setTitle(v->getPlotTitle() + lr);
			} else {
				item->setTitle(v->getPlotTitle());
			}
		}
	}

	ui->replot(); // there seems to be no simpler way before Qwt6 to redraw the legend?
}


void VisualizationPlotView::addVisualization(Visualization* vis)
{
	VisualizationPlot* v = dynamic_cast<VisualizationPlot*>(vis);
	assert(v!=NULL);
	v->init(this);
	VisualizationView::addVisualization(vis);
	updateAxes();
}

void VisualizationPlotView::removeVisualization(Visualization* vis)
{
	VisualizationView::removeVisualization(vis);
	updateAxes();
}

void VisualizationPlotView::moveUpVisualization(Visualization* vis) {
	VisualizationViewTransformable::moveUpVisualization(vis);
	reattachVisualizationItems();
}

void VisualizationPlotView::moveDownVisualization(Visualization* vis)
{
	VisualizationViewTransformable::moveDownVisualization(vis);
	reattachVisualizationItems();
}

void VisualizationPlotView::reattachVisualizationItems()
{
	std::list<Visualization*> visualizations = getVisualizations();
	if (visualizations.size() < 2)
		return;

	foreach (Visualization* v, visualizations) {
		VisualizationPlot* plotvis = dynamic_cast<VisualizationPlot*>(v);
		assert (plotvis!=NULL);

		foreach (auto item, plotvis->getPlotItems())
		{
			assert (item!=NULL);

			item->detach();
			item->attach(getPlot());
		}
	}

	// recreate legend
	if (legendEnabled())
		enableLegend(true);

	updateAxes();
}

void VisualizationPlotView::update(Duration dt)
{
	assert(ui!=NULL);

	// this is a workaround
	// if the legend and plot items are created at the same time
	// the legend stays empty on some versions of qwt (e.g. 6.1)
	if ( mLegendEnabled && !ui->legend())
		enableLegend( mLegendEnabled );
}

QWidget* VisualizationPlotView::createVisualizationPart()
{
	ui = createUI();
	updateScale(QwtPlot::yLeft);
	updateScale(QwtPlot::yRight);
	setupVisualizations();
	updateAxes();

	if(getVisualizations().empty())
		populateDefaultVisualizations();

	return ui;
}

void VisualizationPlotView::saveContentToFile()
{
	QwtPlot* plot = getPlot();
	const int w = plot->width();
	const int h = plot->height();
	const float ar = static_cast<float>(w) / static_cast<float>(h);

	uint targetW = w;
	uint targetH = h;
	if(mSaveContentMinPictureSize > std::min<uint>(targetW, targetH))
	{
		if(w >= h)
		{
			targetH = mSaveContentMinPictureSize;
			targetW = floor(targetH * ar + 0.5f);
		}
		else
		{
			targetW = mSaveContentMinPictureSize;
			targetH = floor(targetW / ar + 0.5f);
		}
	}

	QImage image(targetW, targetH, QImage::Format_RGB32);
	QPainter painter(&image);
	painter.scale((qreal)targetW/(qreal)w, (qreal)targetH/(qreal)h);
	painter.setRenderHint(QPainter::Antialiasing);
	plot->render(&painter);

	QStringList extList;
	extList << "png" << "bmp" << "jpg" << "jpeg";
	static QString saveDir = ".";
	QString filename = QtUtils::getSaveFileName(NULL, tr("Save content as image"),
	                                            saveDir, tr("Images")+" (*.png *.bmp *.jpg *.jpeg)",
	                                            NULL, QFileDialog::DontUseNativeDialog, extList);
	if(filename.isEmpty())
		return;

	image.save(filename);

	// memorize the selected location
	saveDir = QFileInfo(filename).path();
}

VisualizationPlotView::UI* VisualizationPlotView::createUI()
{
	return new UI(this);
}

void VisualizationPlotView::setAutoScaleY(int axis, bool enabled) {
	mAutoScaleY[axis] = enabled;
	updateScale(axis);
}

void VisualizationPlotView::setYMin(int axis, float y) {
	mYMin[axis] = y;
	updateScale(axis);
}

void VisualizationPlotView::setYMax(int axis, float y) {
	mYMax[axis] = y;
	updateScale(axis);
}

void VisualizationPlotView::updateScale(int axis) {
	if (ui) {
		if (mAutoScaleY[axis])
			ui->setAxisAutoScale(axis);
		else
			ui->setAxisScale(axis, mYMin[axis], mYMax[axis]);
		ui->updateAxes();
	}
}

void VisualizationPlotView::enableLegend( bool enabled )
{
	mLegendEnabled = enabled;
	if (ui) {
		delete ui->legend();
		if ( enabled ) {
			QwtLegend *legend = new QwtLegend();
			#if (QWT_VERSION >= 0x060100)
				legend->setDefaultItemMode(QwtLegendData::Checkable);
			#else
				legend->setItemMode(QwtLegend::CheckableItem);
			#endif
			ui->insertLegend(legend, QwtPlot::TopLegend);
		}
		ui->updateLayout();
	}
}

bool VisualizationPlotView::legendEnabled() const
{
	return mLegendEnabled;
}

void VisualizationPlotView::createDefaultVisualization(const std::string& className,
                                                       const std::string& displayName)
{
	try {
		if ( ClassFactory::instance().isClassRegistered(className) ) {
			ClassProxy c = ClassFactory::instance().getClassByIdentifier(className);
			VisualizationPlot* vis = c.newInstance<VisualizationPlot>();
			vis->setName(displayName);
			addVisualization(vis);
		}
	} catch(...) {} // ignore errors
}

//////////////////////////////////////////////////////////////////////////////

class InvertedMagnifier : public QwtPlotMagnifier
{
public:
	InvertedMagnifier(QWidget* canvas) : QwtPlotMagnifier(canvas) {}
protected:
	void rescale(double factor) { QwtPlotMagnifier::rescale(1.0/factor); }
};

VisualizationPlotView::UI::UI(VisualizationPlotView* parent) :
	QwtPlot(parent), vis(parent)
{
	setAcceptDrops(true);
	setCanvasBackground(QColor(255,255,255));
	QFont axisFont("Arial", 10);
	setAxisFont(QwtPlot::yLeft, axisFont);
	setAxisFont(QwtPlot::yRight, axisFont);
	setAxisFont(QwtPlot::xBottom, axisFont);

	plotLayout()->setAlignCanvasToScales(true);

	QwtPlotMagnifier* plotMagnifier = new InvertedMagnifier(canvas());
	plotMagnifier->setMouseButton(Qt::NoButton);
	QwtPlotPanner* panner = new QwtPlotPanner(canvas());
	panner->setMouseButton(Qt::LeftButton);

	this->setContextMenuPolicy(Qt::CustomContextMenu);
	connect(this,SIGNAL(customContextMenuRequested(const QPoint& )),this,SLOT(popupMenu(const QPoint&)));

	vis->startUpdateTimer();
}

void VisualizationPlotView::UI::popupMenu(const QPoint& pos)
{
	QMenu menu;
	menu.addAction("Clear");

	QAction* freezeAction = menu.addAction("Freeze");
	freezeAction->setCheckable(true);
	freezeAction->setChecked(!vis->mAcceptingUpdates);

	populatePopupMenu(menu);
	QAction* a = menu.exec(QCursor::pos());
	if (a)
	{
		if (a->text() == "Clear")
		{
			auto& vs = vis->getVisualizations();
			foreach(auto& v, vs)
			{
				VisualizationPlot* vp = dynamic_cast<VisualizationPlot*>(v);
				vp->clear();
			}
			vis->updateAxes();
		}
		else
			onPopupMenuAction(a);
	}

	freeze(freezeAction->isChecked());
}

void VisualizationPlotView::UI::freeze(bool freeze)
{
	vis->mAcceptingUpdates = !freeze;
	canvas()->setStyleSheet(QString("background-color:") + (freeze ? "#DBEBF8;" : "white;"));
	replot();
}

///////////////////////////////////////////////////////////////////////////////

}
