/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RestrictAreaTask.h
 *    Restrict permitted movement to an area.
 *
 * @author Tom Mehner
 * @date   Jan 17, 2023
 */


#ifndef _MIRA_RESTRICTAREATASK_H_
#define _MIRA_RESTRICTAREATASK_H_

#include <geometry/Polygon.h>
#include <navigation/Task.h>

namespace mira { namespace navigation {

///////////////////////////////////////////////////////////////////////////////

/**
 * Task for restricting the permitted movement to an area.
 */
class RestrictAreaTask : public navigation::SubTask
{
	MIRA_OBJECT(RestrictAreaTask);

public:
	/// Default constructor.
	RestrictAreaTask() :
		frameID("/GlobalFrame")
	{}

	/**
	 * Creates a task that limits the movement to a polygon.
	 * @param[in] area The restricting area
	 * @param[in] f    The frame ID.
	 */
	RestrictAreaTask(Polygon2f area, const std::string& f = "/GlobalFrame") :
		restrictingAreas{std::move(area)},
		frameID(f)
	{}

	/**
	 * Creates a task that limits the movement to a union of a vector of polygons.
	 * @param[in] areas The restricting area
	 * @param[in] f     The frame ID.
	 */
	RestrictAreaTask(std::vector<Polygon2f> areas, const std::string& f = "/GlobalFrame") :
		restrictingAreas(std::move(areas)),
		frameID(f)
	{}

	/// The reflect method.
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, SubTask);
		r.property("RestrictingAreas", restrictingAreas, "The areas the robot is restricted to.");
		r.property("FrameID", frameID, "The frame the position is related to.", "/GlobalFrame");
	}

public:
	/// The restricted area.
	std::vector<Polygon2f> restrictingAreas;

	/// The frame the restricted area is related to
	std::string frameID;
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace mira::navigation

#endif
