/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Task.h
 *    Base classes and interfaces for navigation tasks.
 *
 * @author Tim Langner
 * @date   2011/04/18
 */

#ifndef _MIRA_TASK_H_
#define _MIRA_TASK_H_

#include <typeinfo>

#include <factory/Factory.h>
#include <serialization/adapters/boost/shared_ptr.hpp>
#include <serialization/adapters/std/list>

namespace mira { namespace navigation {

///////////////////////////////////////////////////////////////////////////////

/**
 * Interface for sub tasks to be added to a navigation task.
 */
class SubTask : public Object
{
	MIRA_OBJECT(SubTask)
public:
	/// The destructor.
	virtual ~SubTask() {}

	/// The reflect method.
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.itemName(this->getClass().getName());
	}
};

/// A subtask pointer
typedef boost::shared_ptr<SubTask> SubTaskPtr;

///////////////////////////////////////////////////////////////////////////////

/**
 * A navigation task that consists of one or multiple sub tasks.
 */
class Task
{
public:
	/// The reflect method.
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("SubTasks", subTasks, "Sub tasks");
	}

public:
	/**
	 * Add a subtask to this task. In principle multiple subtasks of the same
	 * type are allowed. Normally getSubTask is used to retrieve a subtask
	 * of a certain type. getSubTask returns the first item in the list that
	 * matches the requested type. In this case all other subtasks of the same 
	 * type will be ignored.
	 * @param[in] task The new subtask.
	 */
	void addSubTask(SubTaskPtr task)
	{
		subTasks.push_back(task);
	}

	/**
	 * Get a subtask of type T.
	 * @param allowDerived Specifies whether the method should also return tasks
	 *                     that are derived from T (see below).
	 * @return pointer to subtask or null pointer if no subtask of type T exists
	 *
	 * @note To understand the allowDerived parameter, imagine the following
	 *       example: Let Foo and Bar be two task classes, where Bar is derived
	 *       from Foo. <TT>getSubTask<Foo>(true)</TT> now also returns sub
	 *       tasks of the type Bar, while <TT>getSubTask<Foo>(false)</TT>
	 *       won't. <TT>getSubTask<Foo>(false)</TT> will only return sub tasks
	 *       of the type Foo and not of derived classes.
	 */
	template<class T>
	boost::shared_ptr<T> getSubTask(bool allowDerived=true)
	{
		foreach(auto task, subTasks)
		{
			auto r = boost::dynamic_pointer_cast<T>(task);
			if ( r != 0 && (allowDerived || typeid(*r) == typeid(T)))
				return r;
		}
		return boost::shared_ptr<T>();
	}


	/**
	 * Get a subtask of type T.
	 * @param allowDerived Specifies whether the method should also return tasks
	 *                     that are derived from T (see below).
	 * @return pointer to subtask or null pointer if no subtask of type T exists
	 *
	 * @note To understand the allowDerived parameter, imagine the following
	 *       example: Let Foo and Bar be two task classes, where Bar is derived
	 *       from Foo. <TT>getSubTask<Foo>(true)</TT> now also returns sub
	 *       tasks of the type Bar, while <TT>getSubTask<Foo>(false)</TT>
	 *       won't. <TT>getSubTask<Foo>(false)</TT> will only return sub tasks
	 *       of the type Foo and not of derived classes.
	 */
	template<class T>
	boost::shared_ptr<const T> getSubTask(bool allowDerived=true) const
	{
		foreach(auto task, subTasks)
		{
			auto r = boost::dynamic_pointer_cast<const T>(task);
			if ( r != 0 && (allowDerived || typeid(*r) == typeid(T)))
				return r;
		}
		return boost::shared_ptr<const T>();
	}

public:

	/// list of subtasks
	std::list<SubTaskPtr> subTasks;
};

/// pointer to task
typedef boost::shared_ptr<Task> TaskPtr;

///////////////////////////////////////////////////////////////////////////////

}}

#endif
