/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationGraphView.C
 *    .
 *
 * @author Erik Einhorn, Ronny Stricker
 * @date   2014/03/12
 */

#include <views/GraphView.h>

#include <QFileDialog>
#include <QScrollBar>
#include <QGraphicsView>

#undef USE_OPENGL
#ifdef USE_OPENGL // collides with our 3D view
#include <QtOpenGL/QGLWidget>
#endif

#include <QtSvg/QSvgRenderer>
#include <QtSvg/QGraphicsSvgItem>

#include <error/Logging.h>

#include <widgets/QtUtils.h>

namespace mira {

GraphView::GraphView(QWidget* parent) : QGraphicsView(parent)
{
	mPan=false;
	mAutoScale = true;
	mScene = new QGraphicsScene(this);
	mGraphItem = NULL;
	setScene(mScene);
//	showText("No data ...");

	mSvgRenderer = new QSvgRenderer(this);


#ifdef USE_OPENGL
	setViewport(new QGLWidget()); // enables OpenGL
#endif
	// setRenderHints(QPainter::Antialiasing); // looks good, but is damn slow
	connect( this, SIGNAL(autoScaleChanged(bool)), this, SLOT(updateGraph()));

}

GraphView::~GraphView()
{
}

void GraphView::mouseMoveEvent(QMouseEvent *event)
{
	if (mPan)
	{
		horizontalScrollBar()->setValue(horizontalScrollBar()->value() - (event->x() - mPanStartX));
		verticalScrollBar()->setValue(verticalScrollBar()->value() - (event->y() - mPanStartY));
		mPanStartX = event->x();
		mPanStartY = event->y();
		event->accept();
		setAutoScale( false );
		return;
	}
	QGraphicsView::mouseMoveEvent(event);
}

void GraphView::mousePressEvent(QMouseEvent *event)
{
	if (event->button() == Qt::RightButton)
	{
		mPan = true;
		mPanStartX = event->x();
		mPanStartY = event->y();
		setCursor(Qt::ClosedHandCursor);
		event->accept();
		return;
	}
	QGraphicsView::mousePressEvent(event);
}

void GraphView::mouseReleaseEvent(QMouseEvent *event)
{
	if (event->button() == Qt::RightButton)
	{
		mPan = false;
		setCursor(Qt::ArrowCursor);
		event->accept();
		return;
	}
	QGraphicsView::mouseReleaseEvent(event);
}


void GraphView::wheelEvent ( QWheelEvent * event )
{
	double angle = event->delta();
	double factor = std::pow(1.0015, angle);
	scale(factor, factor);
	setAutoScale( false );
}

void GraphView::updateGraph()
{
	// display the svg

	mScene->clear();

	QGraphicsSvgItem* item = new QGraphicsSvgItem;
	mGraphItem = item;
	mGraphItem->setFlags(QGraphicsItem::ItemClipsToShape);
	mGraphItem->setCacheMode(QGraphicsItem::NoCache);

	delete mSvgRenderer;
	QXmlStreamReader reader( mGraph );
	mSvgRenderer = new QSvgRenderer();
	if ( !mSvgRenderer->load( &reader ) ) {
		MIRA_LOG(WARNING) << "GraphView: Failed to render graph!";
	}
	item->setSharedRenderer(mSvgRenderer);

	mScene->addItem(mGraphItem);
	mScene->setSceneRect(mGraphItem->boundingRect());

	// adjust scale
	if( mAutoScale ) {
		fitInView(mGraphItem,Qt::KeepAspectRatio);
		// Reset the view scale if automatic scale zoomed in
#if QT_VERSION > QT_VERSION_CHECK(5, 12, 0)
		QRectF unity = transform().mapRect(QRectF(0, 0, 1, 1));
#else
		QRectF unity = matrix().mapRect(QRectF(0, 0, 1, 1));
#endif
		if (!unity.isEmpty() && unity.width()>1.0 )
			scale(1 / unity.width(), 1 / unity.height());
	}
}

void GraphView::updateGraph( QString const& graph )
{
	mGraph = graph;
	updateGraph();
}

void GraphView::saveGraph()
{
	QString name = QtUtils::getSaveFileName(this, tr("Save graph"), QString(),
	                                        tr("Graphviz dot files (*.dot)"));
	if(name.isNull() || name.isEmpty())
		return;

	std::ofstream os(name.toStdString());
	os << mGraph.toStdString();
	os.close();
}

void GraphView::showText( QString const& text)
{
//	std::cout << "scene: " << mScene << std::endl;
	mScene->clear();
	mTextItem = new QGraphicsSimpleTextItem(text);
	mScene->addItem(mTextItem);
}

void GraphView::setAutoScale( bool autoscale )
{
	if ( mAutoScale != autoscale ) {
		mAutoScale = autoscale;
		emit autoScaleChanged( mAutoScale );
	}
}

///////////////////////////////////////////////////////////////////////////////

}
