/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ColorPropertyDelegate.C
 *    A delegate for color properties.
 *
 * @author Tim Langner
 * @date   2011/01/05
 */

#include <boost/format.hpp>

#include <QColorDialog>

#include <QPixmap>
#include <QIcon>
#include <QPainter>

#include <widgets/PropertyEditor.h>

#include <image/Color.h>

#include <serialization/adapters/Qt/QColor>
#include <serialization/adapters/OGRE/OgreColourValue.h>
#include <serialization/PropertySerializer.h>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

static QPixmap brushValuePixmap(const QBrush &b)
{
	QImage img(16, 16, QImage::Format_ARGB32_Premultiplied);
	img.fill(0);

	QPainter painter(&img);
	painter.setCompositionMode(QPainter::CompositionMode_Source);
	painter.fillRect(0, 0, img.width(), img.height(), b);
	QColor color = b.color();
	if (color.alpha() != 255) { // indicate alpha by an inset
		QBrush  opaqueBrush = b;
		color.setAlpha(255);
		opaqueBrush.setColor(color);
		painter.fillRect(img.width() / 4, img.height() / 4,
		                 img.width() / 2, img.height() / 2, opaqueBrush);
	}
	painter.end();
	return QPixmap::fromImage(img);
}

static QIcon brushValueIcon(const QBrush &b)
{
	return QIcon(brushValuePixmap(b));
}

///////////////////////////////////////////////////////////////////////////////

template <typename Color>
class PropertyDelegateColorBase : public PropertyEditor::Delegate
{
public:

	virtual QColor toQColor(const Color& color) = 0;
	virtual Color fromQColor(const QColor& color) = 0;

public:
	virtual SupportedTypes supportedTypes() const {
		return makeSupportedType<Color>(SupportedType::TEXT |
		                                SupportedType::DIALOG |
		                                SupportedType::ICON);
	}

	virtual std::string getText(const PropertyNode* property) {
		const TypedPropertyNode<Color>* p = property->toTyped<Color>();
		assert(p!=NULL);
		QColor c = toQColor(p->get());
		return (boost::format("[%.2f, %.2f, %.2f] (%.1f)")
		                      % c.redF() % c.greenF() % c.blueF()
		                      % c.alphaF()).str();
	}

	virtual QIcon getIcon(const PropertyNode* property) {
		const TypedPropertyNode<Color>* p = property->toTyped<Color>();
		assert(p!=NULL);
		QColor c = toQColor(p->get());
		QBrush br(c);
		return brushValueIcon(br);
	}

	virtual bool execDialog(PropertyNode* property, QWidget* parent)
	{
		TypedPropertyNode<Color>* p = property->toTyped<Color>();
		assert(p!=NULL);

		QColor c = toQColor(p->get());
		QColor qc = QColorDialog::getColor(c, parent, "Select color",
		                                   QColorDialog::ShowAlphaChannel |
		                                   QColorDialog::DontUseNativeDialog);

		if (qc.isValid()) {
			p->set(fromQColor(qc));
			return true;
		} else
			return false;
	}
};

///////////////////////////////////////////////////////////////////////////////

class PropertyDelegate_QColor : public PropertyDelegateColorBase<QColor>
{
	MIRA_OBJECT(PropertyDelegate_QColor)
public:

	virtual QColor toQColor(const QColor& color) { return color; }
	virtual QColor fromQColor(const QColor& color) { return color; }
};

///////////////////////////////////////////////////////////////////////////////

class PropertyDelegate_OgreColor : public PropertyDelegateColorBase<Ogre::ColourValue>
{
	MIRA_OBJECT(PropertyDelegate_OgreColor)
public:

	virtual QColor toQColor(const Ogre::ColourValue& c) {
		return QColor::fromRgbF(c.r,c.g,c.b,c.a);
	}

	virtual Ogre::ColourValue fromQColor(const QColor& c) {
		return Ogre::ColourValue(c.redF(), c.greenF(), c.blueF(), c.alphaF());
	}
};

///////////////////////////////////////////////////////////////////////////////

class PropertyDelegate_ColorRGB : public PropertyDelegateColorBase<Color::RGB>
{
	MIRA_OBJECT(PropertyDelegate_ColorRGB)
public:

	virtual QColor toQColor(const Color::RGB& c) {
		return QColor::fromRgbF(c.r,c.g,c.b,1.0f);
	}

	virtual Color::RGB fromQColor(const QColor& c) {
		return Color::RGB(c.redF(), c.greenF(), c.blueF());
	}
};

///////////////////////////////////////////////////////////////////////////////

class PropertyDelegate_ColorRGBA : public PropertyDelegateColorBase<Color::RGBA>
{
	MIRA_OBJECT(PropertyDelegate_ColorRGBA)
public:

	virtual QColor toQColor(const Color::RGBA& c) {
		return QColor::fromRgbF(c.r,c.g,c.b,c.a);
	}

	virtual Color::RGBA fromQColor(const QColor& c) {
		return Color::RGBA(c.redF(), c.greenF(), c.blueF(), c.alphaF());
	}
};

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_REGISTER(mira::gui::PropertyDelegate_QColor,
                    mira::PropertyEditor::Delegate);
MIRA_CLASS_REGISTER(mira::gui::PropertyDelegate_OgreColor,
                    mira::PropertyEditor::Delegate);
MIRA_CLASS_REGISTER(mira::gui::PropertyDelegate_ColorRGB,
                    mira::PropertyEditor::Delegate);
MIRA_CLASS_REGISTER(mira::gui::PropertyDelegate_ColorRGBA,
                    mira::PropertyEditor::Delegate);
