/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PoseVisualization.C
 *    Visualization of 3D pose.
 *
 * @author Erik Einhorn
 * @date   2010/12/30
 */

#include <deque>

#include <serialization/Serialization.h>
#include <serialization/GetterSetter.h>

#include <transform/Pose.h>
#include <math/EigenFormat.h>

#include <visualization/3d/AxesVisualization.h>
#include <visualization/3d/AxesCovObject.h>
#include <visualization/3d/TextObject.h>
#include <visualization/3d/MeshObject.h>

#include "private/PoseCovVisualization.h"

#include <widgets/OgreUtils.h>

#include <OGRE/OgreSceneNode.h>
#include <OGRE/OgreSceneManager.h>

#include <Eigen/SVD>

#include <visualization/3d/LineStripObject.h>
#include <image/Colormap.h>
#include <visualization/ColormapProperty.h>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

enum TrailType : uint32 {
	NONE = 0,
	AXIS,
	PATH,
	BOTH
};

struct TrailSettings
{
	uint32 length;
	float  interval;
	TrailType mTrailVisualization;

	TrailSettings() : length(0), interval(0.2f),  mTrailVisualization(PATH) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Length", length, "The trail length (0 to disable)",
		           0, PropertyHints::minimum<uint32>(0));
		r.property("Interval", interval,
		           "Interval between two trail samples (in seconds)",
		           0.2f, PropertyHints::minimum<float>(0.0f));
		r.property("TrailVisualization", mTrailVisualization,
		           "Visualize trail as path, axis or both",
		           PATH, PropertyHints::enumeration("NONE;AXIS;PATH;BOTH"));
	}
};

struct PathSettings
{
	float lineWidth;
	float minVelocity;
	float maxVelocity;
	ColormapProperty colorMap;

	PathSettings() : lineWidth(0.1f), minVelocity(0.0), maxVelocity(3.0), colorMap("mira::RedBlueColormap") {
		colorMap.setFilter(ColormapProperty::filterContinuous);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("LineWidth", lineWidth,
		           "The line width of the path visualization",
		           0.1f, PropertyHints::minimum<float>(0.0f));
		r.property("MinVelocity", minVelocity, "Velocity which gets minimum color of palette", 0.0, PropertyHints::minimum(0.0) );
		r.property("MaxVelocity", maxVelocity, "Velocity which gets maximum color of palette", 3.0, PropertyHints::minimum(0.0) );
		r.property("Colormap", colorMap, "The color palette",
		           ColormapProperty("mira::GrayscaleColormap"));
	}
};

///////////////////////////////////////////////////////////////////////////////

template <typename Pose>
class PoseVisualizationBase : public AxesVisualization
{
public:
	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

public:

	/** @name Constructor, destructor and reflect */
	//@{

	/// The constructor.
	PoseVisualizationBase() :
		mType(""),
		mText(NULL),
		mShowName(true),
		mShowYAxis(false),
		mPathObject(NULL)
	{
		mAxisLength = 0.1f; // set initial default axis length to 0.1
		mAxisRadius = 0.01f;
		mTrailSampleInterval = 0.0f;
		mTransformType = CovVisTrait<Pose>::type;
	}

	/// The destructor.
	virtual ~PoseVisualizationBase()
	{
		delete mText;
		delete mPathObject;
	}

	/// The reflect method.
	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.roproperty("Type", mType, "Indicates whether the shown pose is a transform or a plain pose");

		MIRA_REFLECT_BASE(r, AxesVisualization);
		channelProperty(r, "Pose", mPoseChannel, "");
		r.property("Show Name", mShowName,
		           setter<bool>(&PoseVisualizationBase::setShowName, this),
		           "Should the name be shown", true);

		r.property("ShowYAxis", mShowYAxis, "Should Y axis be visualized (for 2d poses)", false);


		TransformType defaultType = CovVisTrait<Pose>::type;
		r.property("TransformType", mTransformType,
		           setter<TransformType>(&PoseVisualizationBase::setTransformType, this),
		           "Type of the transform that is rendered",
		           defaultType, PropertyHints::enumeration("Pose2;PoseCov2;Pose3;PoseCov3"));

		r.property("Sigma",
		           getter<float>(&CovVisualization::getSigma, &mCovVisualization),
		           setter<float>(&CovVisualization::setSigma, &mCovVisualization),
		           "The sigma of the shown surface", 0.5f);

		r.property("Trail", mTrailSettings,
		           "Settings for the trail", TrailSettings());
		r.property("Path", mPathSettings,
		           "Settings for the path", PathSettings());
		r.roproperty("Position", mPosition,
		             "The position relative to the set target frame");
	}

	//@}

public:
	/** @name Public implementation of Visualization3D */
	//@{

	virtual DataConnection getDataConnection()
	{
		return DataConnection(mPoseChannel);
	}

	virtual void setupScene(IVisualization3DSite* site)
	{
		AxesVisualization::setupScene(site);
		mText = new TextObject("Unknown", site->getSceneManager(), mAxes->getNode());
		mText->setTextAlignment(TextObject::H_LEFT, TextObject::V_BELOW);
		mText->setCharacterHeight(0.1f);
		mText->setColor(Ogre::ColourValue::Black);
		mText->setVisible(isEnabled() && mShowName);
		mCovVisualization.setupScene(site->getSceneManager(),mAxes);
		mCovVisualization.setVisible(isEnabled() &&
		                             (mTransformType == POSE2COV || mTransformType == POSE3COV));

		Ogre::SceneManager* mgr = site->getSceneManager();
		mNode = mgr->getRootSceneNode()->createChildSceneNode();
		mPathObject = new LineStripObject(mgr,mNode);
		mNode->setVisible(isEnabled());
	}

	//@}

public:
	/** @name Setters */
	//@{

	void setShowName(bool show)
	{
		mShowName = show;
		if(mText)
			mText->setVisible(isEnabled() && mShowName);
	}

	virtual void setEnabled(bool enabled)
	{
		AxesVisualization::setEnabled(enabled);
		if(mText)
			mText->setVisible(isEnabled() && mShowName);

		for(uint32 i=0; i<mTrail.size(); ++i)
			if(i<mTrailSettings.length)
				mTrail[i].axesObject->setVisible(isEnabled());
			else
				mTrail[i].axesObject->setVisible(false);

		mCovVisualization.setVisible(isEnabled() &&
		                             (mTransformType == POSE2COV || mTransformType == POSE3COV));

		if(mNode != NULL)
			mNode->setVisible(isEnabled());

		if(mPathObject)
			mPathObject->setVisible(isEnabled());
	}

	void setTransformType(TransformType type)
	{
		mTransformType = type;
		mCovVisualization.setVisible(isEnabled() &&
		                             (mTransformType == POSE2COV || mTransformType == POSE3COV));
	}

	//@}

protected:
	/** @name Protected implementation of Visualization3D */
	//@{

	virtual void update(Duration dt)
	{
		switch(mTransformType)
		{
		case POSE2:
			updateFrame<Pose2>(dt);
			break;
		case POSE2COV:
			updateFrame<PoseCov2>(dt);
			break;
		case POSE3:
			updateFrame<Pose3>(dt);
			break;
		case POSE3COV:
			updateFrame<PoseCov3>(dt);
			break;
		case INVALID : break;
		}

		if(!mPoseChannel.isValid() || mPoseChannel.getDataUpdateCount() == 0)
			return;

		std::string frameID = mPoseChannel.getID();
		if(frameID.empty()){
			MIRA_THROW(TransformerBase::XTransform,
			           "Data has no frame for transforming it into the view's camera frame");
		};

		mPathSettings.minVelocity = std::min(mPathSettings.minVelocity, mPathSettings.maxVelocity);
		mPathSettings.maxVelocity = std::max(mPathSettings.minVelocity, mPathSettings.maxVelocity);
	}

	template <typename P>
	void updateFrame(Duration dt)
	{
		bool showZAxis = std::min<int>(P::Dim,Pose::Dim)>2;

		mAxes->showYAxis(mShowYAxis || showZAxis);
		mAxes->showZAxis(showZAxis);
		mText->setCaption(mPoseChannel.getID());


		auto poseRead = mPoseChannel.getChannel().read();
		Time poseTime = poseRead->timestamp;

		P pose;

		// determine if we have a real transform or a plain pose
		auto tfnode = MIRA_FW.getTransformer()->getNode(mPoseChannel.getID());
		if(tfnode==NULL) {
			// plain pose
			mType = "Pose";
			// get transform to parent pose
			try {
				pose = getAuthority().template getTransform<P>(
										poseRead->frameID, poseTime,
										getSite()->getCameraFrame(), Time::now(),
										getSite()->getFixedFrame(), LinearInterpolatorNearestNeighbourExtrapolator());
			}
			catch(Exception& ex) {	// fall back to non interpolating versions
				pose = getAuthority().template getTransform<P>(
										poseRead->frameID, poseTime,
										getSite()->getCameraFrame(), Time::now(),
										getSite()->getFixedFrame());
			}
			pose = pose * transform_cast<P>(poseRead->value());
			mPosition = poseRead->value();

		} else {
			// real transform
			mType = "Transform";

			TransformDesc td = prepareTransform(mPoseChannel.getID(), getSite()->getCameraFrame());
			// get latest timestamp in whole transform chain
			for (auto tr = td.chain1.forward.begin(); tr != td.chain1.forward.end(); ++tr) {
				Time ts = MIRA_FW.getChannelManager().getLastSlotTime((*tr)->getID());
				if(ts.isValid() && ts > poseTime)
					poseTime = ts;
			}

			try {
				pose = getAuthority().template getTransform<P>(
								mPoseChannel.getID(), poseTime,
								getSite()->getCameraFrame(), Time::now(),
								getSite()->getFixedFrame(), LinearInterpolatorNearestNeighbourExtrapolator());
			}
			catch(Exception& ex) {	// fall back to non interpolating versions
				pose = getAuthority().template getTransform<P>(mPoseChannel.getID(), poseTime,
				                                               getSite()->getCameraFrame(), Time::now(),
				                                               getSite()->getFixedFrame());
			}
			mPosition = getAuthority().template getTransform<Pose>(mPoseChannel.getID(), getSite()->getCameraFrame());
		}

		mAxes->setTransform(pose);
		mCovVisualization.update(pose);

		if(mTrailSettings.length>0) {
			mTrailSampleInterval += dt.totalMilliseconds() * 0.001f;

			if(mTrailSampleInterval > mTrailSettings.interval) {
				mTrailSampleInterval = 0.0f;

				AxesObjectWithTime p;

				if(mTrail.size()>=mTrailSettings.length) {
					p = mTrail.back();
					mTrail.pop_back();
				} else {
					Ogre::SceneManager* sceneManager = this->getSite()->getSceneManager();
					p.axesObject.reset(new AxesObject(sceneManager, sceneManager->getRootSceneNode()));
				}

				p.axesObject->showYAxis(mShowYAxis || showZAxis);
				p.axesObject->showZAxis(showZAxis);
				p.axesObject->setTransform(pose);
				p.axesObject->setLength(mAxes->getLength());
				p.axesObject->setRadius(mAxes->getRadius());
				p.time = poseTime;
				mTrail.push_front(p);
			}
		} else
			mTrailSampleInterval = 0.0f;

		for(uint32 i=0; i<mTrail.size(); ++i)
		{
			if(i<mTrailSettings.length) {
				mTrail[i].axesObject->setVisible(isEnabled() &&
				                                 (mTrailSettings.mTrailVisualization == AXIS || mTrailSettings.mTrailVisualization == BOTH));
				// fade out visible trail alpha from axes alpha to '0'
				float alpha = mAxes->getAlpha() * (1.f - (float)i / (float)mTrailSettings.length);
				mTrail[i].axesObject->setAlpha(alpha);
			} else
				mTrail[i].axesObject->setVisible(false);
		}

		mPathObject->clear();
		mPathObject->setLineWidth(mPathSettings.lineWidth);

		if(mTrailSettings.mTrailVisualization == PATH || mTrailSettings.mTrailVisualization == BOTH) {
			mPathObject->begin();
			const ContinuousColormap& colormap = dynamic_cast<const ContinuousColormap&>(mPathSettings.colorMap.getColormap());
			uint32 l = std::min<uint32>(mTrail.size(), mTrailSettings.length);
			for(uint32 i=0; i + 1 < l; ++i)
			{
				double seconds = (mTrail[i].time - mTrail[i+1].time).totalMicroseconds() / 1000000.0;
				Ogre::Vector3 src = mTrail[i].axesObject->getNode()->getPosition();
				Ogre::Vector3 dst = mTrail[i+1].axesObject->getNode()->getPosition();
				double velocity = (Point3f(src.x, src.y, src.z) - Point3f(dst.x,dst.y,dst.z)).norm() / seconds;

				Color::RGB miraColor =  colormap.getf((velocity-mPathSettings.minVelocity)/(mPathSettings.maxVelocity-mPathSettings.minVelocity));
				Ogre::ColourValue color(miraColor.r, miraColor.g, miraColor.b, 1.0);
				mPathObject->point(src, color);
			}
			mPathObject->end();
		};
	}

	//@}

struct AxesObjectWithTime{
	AxesObjectWithTime(){

	}

	AxesObjectWithTime(boost::shared_ptr<AxesObject>& iAxesObject, Time& iTime):axesObject(iAxesObject), time(iTime){
	}

	boost::shared_ptr<AxesObject> axesObject;
	Time time;
};

private:
	ChannelProperty<Pose> mPoseChannel;
	std::string mType;
	TextObject* mText;
	Pose mPosition;
	bool mShowName;
	bool mShowYAxis;

	TrailSettings mTrailSettings;
	std::deque<AxesObjectWithTime> mTrail;
	float mTrailSampleInterval;

	PathSettings mPathSettings;
	LineStripObject* mPathObject;
	Ogre::SceneNode* mNode;

	TransformType mTransformType;
	CovVisualization mCovVisualization;
};

///////////////////////////////////////////////////////////////////////////////

class Pose3Visualization3D : public PoseVisualizationBase<Pose3>
{
	MIRA_META_OBJECT(Pose3Visualization3D,
		("Name", "Pose 3D")
		("Description", "Visualization of 3D poses")
		("Category", "Pose"))
};

class PoseCov3Visualization3D : public PoseVisualizationBase<PoseCov3>
{
	MIRA_META_OBJECT(PoseCov3Visualization3D,
		("Name", "Pose 3D")
		("Description", "Visualization of 3D poses with covariance")
		("Category", "Pose"))
};

class Pose2Visualization3D : public PoseVisualizationBase<Pose2>
{
	MIRA_META_OBJECT(Pose2Visualization3D,
		("Name", "Pose 2D")
		("Description", "Visualization of 2D poses")
		("Category", "Pose"))
};

class PoseCov2Visualization3D : public PoseVisualizationBase<PoseCov2>
{
	MIRA_META_OBJECT(PoseCov2Visualization3D,
		("Name", "Pose 2D")
		("Description", "Visualization of 2D poses with covariance")
		("Category", "Pose"))
};

//////////////////////////////////////////////////////////////////////////////

}} // namespace

MIRA_CLASS_SERIALIZATION(mira::gui::Pose3Visualization3D,
                         mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::PoseCov3Visualization3D,
                         mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Pose2Visualization3D,
                         mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::PoseCov2Visualization3D,
                         mira::Visualization3D);
