/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */
/**
 * @file PointsVisualization.C
 *    Visualizations for point types and vectors of point types
 *
 * @author Erik Einhorn, Christof Schröter, Tim van der Grinten
 * @date   2021/11/08
 */

#include <boost/scoped_ptr.hpp>

#include <serialization/Serialization.h>
#include <serialization/SetterNotify.h>

#include <math/Saturate.h>

#include <transform/Pose.h>
#include <geometry/Point.h>

#include <visualization/Visualization3DBasic.h>
#include <serialization/adapters/OGRE/OgreColourValue.h>
#include <visualization/3d/DynamicRenderable.h>

#include <widgets/OgreUtils.h>
#include <OGRE/Ogre.h>

using namespace Eigen;
using namespace std;

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

template <typename PointType>
struct PointTypeDimensionTrait { static const int dims = -1; };

template <> struct PointTypeDimensionTrait<Point2i> { static const int dims = 2; };
template <> struct PointTypeDimensionTrait<Point2f> { static const int dims = 2; };
template <> struct PointTypeDimensionTrait<Point2d> { static const int dims = 2; };
template <> struct PointTypeDimensionTrait<Eigen::Vector2i> { static const int dims = 2; };
template <> struct PointTypeDimensionTrait<Eigen::Vector2f> { static const int dims = 2; };
template <> struct PointTypeDimensionTrait<Eigen::Vector2d> { static const int dims = 2; };
template <> struct PointTypeDimensionTrait<Point3i> { static const int dims = 3; };
template <> struct PointTypeDimensionTrait<Point3f> { static const int dims = 3; };
template <> struct PointTypeDimensionTrait<Point3d> { static const int dims = 3; };
template <> struct PointTypeDimensionTrait<Eigen::Vector3i> { static const int dims = 3; };
template <> struct PointTypeDimensionTrait<Eigen::Vector3f> { static const int dims = 3; };
template <> struct PointTypeDimensionTrait<Eigen::Vector3d> { static const int dims = 3; };

///////////////////////////////////////////////////////////////////////////////

static float gPointVertices[1 * 2] = {0.0f,0.0f};

static float gSquareBillboardVertices[6 * 2] =
{
	-0.5f, 0.5f,
	-0.5f, -0.5f,
	0.5f, 0.5f,
	0.5f, 0.5f,
	-0.5f, -0.5f,
	0.5f, -0.5f,
};

static float gTriangleBillboardVertices[3 * 2] =
{
	 0.0f,  1.0,
	-0.866f, -0.5f,
	 0.866f, -0.5f,
};

//cos(120) = -0.5
//sin(120) = 0,866025403784



struct Settings
{
	enum DrawStyle
	{
		Points = 0,
		Squares,
		Triangles,
		Spheres,
	};

	enum ColorMode
	{
		Solid = 0,
		XYZ
	};

	Settings() : drawStyle(Points), size(0.1f),
		color(Ogre::ColourValue::Blue) ,
		colorMode(Solid){}
	DrawStyle drawStyle;
	float size;
	Ogre::ColourValue color;
	ColorMode colorMode;
};


float trianglewave(float x, float a)
{
	float xa = x / a;
	return std::abs(2.0f*(xa-std::floor(xa+0.5f)));
}

template<typename PointType>
class PointVectorRenderer : public DynamicRenderable
{

public:
	PointVectorRenderer(Settings& settings) : mSettings(settings)
	{
		Ogre::VertexDeclaration decl;
		decl.addElement(0, 0, Ogre::VET_FLOAT3, Ogre::VES_POSITION);
		std::size_t offset = Ogre::VertexElement::getTypeSize(Ogre::VET_FLOAT3);
		decl.addElement(0, offset, Ogre::VET_FLOAT2, Ogre::VES_TEXTURE_COORDINATES, 0);
		offset += Ogre::VertexElement::getTypeSize(Ogre::VET_FLOAT2);
		decl.addElement(0, offset, Ogre::VET_COLOUR, Ogre::VES_DIFFUSE);

		initialize(Ogre::RenderOperation::OT_POINT_LIST, decl, false);
		dummyBuffer();

		// create a copy of the 'NoLight' material so it can be manipulated
		// (e.g. change point size) without affecting other objects

		Ogre::MaterialManager* manager = Ogre::MaterialManager::getSingletonPtr();
		mPointMaterial = manager->create(
			"PointVectorRenderer"+toString(this)+"/Material", "MIRAMaterials");
		Ogre::MaterialPtr mBaseMaterial = manager->getByName("NoLight");
		if (!mBaseMaterial.isNull())
			mBaseMaterial->copyDetailsTo(mPointMaterial);
	}

	~PointVectorRenderer()
	{
		if (!mPointMaterial.isNull())
			Ogre::MaterialManager::getSingletonPtr()->remove(mPointMaterial->getName());
	}

	void setPoints(const std::vector<PointType>& data)
	{
		fillBuffers(data);
	}

	void setSize(float s)
	{
		mSettings.size = s;
		Ogre::Vector4 size(mSettings.size*0.1f, mSettings.size*0.1f,
		                   mSettings.size*0.1f, 0.0f);
		setCustomParameter(0, size);
		this->getMaterial()->setPointSize(mSettings.size);
	}

private:

	void fillBuffers(const std::vector<PointType>& data)
	{
		::uint32 vertexCount;
		float* vertices;

		switch(mSettings.drawStyle)
		{
		case Settings::Points:
			mRenderOp.operationType = Ogre::RenderOperation::OT_POINT_LIST;
			this->setMaterial(mPointMaterial->getName());
			vertices = gPointVertices;
			vertexCount = 1;
			break;
		case Settings::Squares:
			mRenderOp.operationType = Ogre::RenderOperation::OT_TRIANGLE_LIST;
			setMaterial("Billboard");
			vertices = gSquareBillboardVertices;
			vertexCount = 6;
			break;
		case Settings::Triangles:
			mRenderOp.operationType = Ogre::RenderOperation::OT_TRIANGLE_LIST;
			setMaterial("Billboard");
			vertices = gTriangleBillboardVertices;
			vertexCount = 3;
			break;
		case Settings::Spheres:
			mRenderOp.operationType = Ogre::RenderOperation::OT_TRIANGLE_LIST;
			setMaterial("BillboardSphere");
			vertices = gTriangleBillboardVertices;
			vertexCount = 3;
			break;
		}

		setSize(mSettings.size);

		prepareBuffers(data.size()*vertexCount);
		mBox=Ogre::AxisAlignedBox(); // start with empty bbox

		Ogre::HardwareVertexBufferSharedPtr vbuf = getVertexBuffer();
		Ogre::RenderSystem* rs = Ogre::Root::getSingleton().getRenderSystem();

		Ogre::Real* prPos = static_cast<Ogre::Real*> (vbuf->lock(Ogre::HardwareBuffer::HBL_DISCARD));

		size_t pointsAdded = 0;
		// loop through each point in the cloud
		for(auto it=data.begin(); it!=data.end(); ++it)
		{
			float x=0.0f, y=0.0f, z=0.0f;

			const int dim = PointTypeDimensionTrait<PointType>::dims;

			x = (*it)[0];
			y = (*it)[1];
			if(dim >= 3)
				z = (*it)[2];

			if (std::isnan(x) || std::isnan(y) || std::isnan(z))
				continue;

			// use specified color for modes: Auto and Solid
			Ogre::ColourValue oc = mSettings.color;
			if(mSettings.colorMode == Settings::XYZ) {
				oc.r = 1.0f - trianglewave(x, 7.0f);
				oc.g = 1.0f - trianglewave(y, 7.0f);
				oc.b = trianglewave(z, 7.0f);
			}

			::uint32 color;
			rs->convertColourValue(oc, &color);

			mBox.merge(Ogre::Vector3(x,y,z));
			for(::uint32 j=0; j<vertexCount; ++j)
			{
				// position
				*prPos++ = x;
				*prPos++ = y;
				*prPos++ = z;
				// billboard coords as tex coords
				*prPos++ = vertices[j*2];
				*prPos++ = vertices[j*2+1];
				::uint32* colorPtr = (::uint32*)(prPos++);
				// color
				*colorPtr = color;
			}
			++pointsAdded;
		}
		vbuf->unlock();
		this->updateVertexCount(pointsAdded*vertexCount);
	}

	void dummyBuffer()
	{
		prepareBuffers(1, 0);
		mBox.setExtents(0,0,0,0,0,0); // start with empty bbox
		Ogre::HardwareVertexBufferSharedPtr vbuf = getVertexBuffer();
		Ogre::Real* prPos = static_cast<Ogre::Real*> (vbuf->lock(Ogre::HardwareBuffer::HBL_DISCARD));
		for(int i=0; i<6; ++i)
			*prPos++ = 0;
		vbuf->unlock();
	}

private:
	Settings& mSettings;
	Ogre::MaterialPtr mPointMaterial;
};

///////////////////////////////////////////////////////////////////////////////

template <typename PointType, typename DataType>
class PointVectorVisualizationBase :  public Visualization3DBasic<DataType>
{
	typedef Visualization3DBasic<DataType> Base;

public:

	PointVectorVisualizationBase() :
		Base("PointData"),
		mNrPoints(0) {}

public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Base);
		r.roproperty("Points", mNrPoints, "The number of points in the container");
		r.property("DrawStyle", mSettings.drawStyle,
		           setterNotify(mSettings.drawStyle, &PointVectorVisualizationBase::drawLast, this),
		           "Visualization style for points", Settings::Points, PropertyHints::enumeration("Points;Squares;Triangles;Spheres"));
		r.property("Size", mSettings.size,
		           setter<float>(&PointVectorVisualizationBase::setSize, this),
		           "Size of the vertices", 0.1f);
		r.property("Color", mSettings.color,
		           setterNotify(mSettings.color, &PointVectorVisualizationBase::drawLast, this),
		           "The color", Ogre::ColourValue::Blue);
		r.property("Color Mode", mSettings.colorMode,
		           setterNotify(mSettings.colorMode, &PointVectorVisualizationBase::drawLast, this),
		           "The color mode", Settings::Solid, PropertyHints::enumeration("Solid;XYZ"));
	}

public: // implementation of Visualization3D and Visualization

	virtual void setupScene(Ogre::SceneManager* mgr, Ogre::SceneNode* node)
	{
		mPointVectorRenderer.reset(new PointVectorRenderer<PointType>(mSettings));
		node->attachObject(mPointVectorRenderer.get());
	}

private:

	void setSize(float s)
	{
		mSettings.size = s;
		if(mPointVectorRenderer)
			mPointVectorRenderer->setSize(s);

		drawLast();
	}

	void drawLast()
	{
		try
		{
			if (!this->mDataChannel.isValid())
				return;
			this->dataChanged(this->mDataChannel.getChannel().read());
		}
		catch(Exception&)
		{
		}
	}

protected:
	boost::scoped_ptr<PointVectorRenderer<PointType>> mPointVectorRenderer;
	Settings mSettings;
	std::size_t mNrPoints;
};

///////////////////////////////////////////////////////////////////////////////

template <typename PointType>
class PointVisualization : public PointVectorVisualizationBase<PointType, PointType>
{
private:

	virtual void dataChanged(ChannelRead<PointType> data)
	{
		assert(this->mPointVectorRenderer);
		std::vector<PointType> vec = {data->value()};
		this->mPointVectorRenderer->setPoints(vec);
		this->mNrPoints = 1;
		this->mNode->needUpdate();
	}
};

template <typename PointType>
class PointVectorVisualization : public PointVectorVisualizationBase<PointType, std::vector<PointType>>
{
private:

	virtual void dataChanged(ChannelRead<std::vector<PointType>> data)
	{
		assert(this->mPointVectorRenderer);
		this->mPointVectorRenderer->setPoints(data->value());
		this->mNrPoints = data->size();
		this->mNode->needUpdate();
	}
};

///////////////////////////////////////////////////////////////////////////////

#define CREATE_VISUALIZATION(Class, BaseClass, MetaName, MetaDescription) \
	class Class : public BaseClass                                        \
	{                                                                     \
		MIRA_META_OBJECT(Class,                                           \
			("Name", MetaName)                                            \
			("Description", "Visualization of " MetaDescription)          \
			("Category", "Geometry"))                                     \
	};                                                                    \

CREATE_VISUALIZATION(Point2iVisualization3D, PointVisualization<Point2i>, "Point2i", "a Point2i")
CREATE_VISUALIZATION(Point3iVisualization3D, PointVisualization<Point3i>, "Point2i", "a Point3i")
CREATE_VISUALIZATION(Point2fVisualization3D, PointVisualization<Point2f>, "Point2f", "a Point2f")
CREATE_VISUALIZATION(Point3fVisualization3D, PointVisualization<Point3f>, "Point3f", "a Point3f")
CREATE_VISUALIZATION(Point2dVisualization3D, PointVisualization<Point2d>, "Point2d", "a Point2d")
CREATE_VISUALIZATION(Point3dVisualization3D, PointVisualization<Point3d>, "Point3d", "a Point3d")
CREATE_VISUALIZATION(EigenVector2iVisualization3D, PointVisualization<Eigen::Vector2i>, "EigenVector2i", "an Eigen::Vector2i")
CREATE_VISUALIZATION(EigenVector3iVisualization3D, PointVisualization<Eigen::Vector3i>, "EigenVector3i", "an Eigen::Vector3i")
CREATE_VISUALIZATION(EigenVector2fVisualization3D, PointVisualization<Eigen::Vector2f>, "EigenVector2f", "an Eigen::Vector2f")
CREATE_VISUALIZATION(EigenVector3fVisualization3D, PointVisualization<Eigen::Vector3f>, "EigenVector3f", "an Eigen::Vector3f")
CREATE_VISUALIZATION(EigenVector2dVisualization3D, PointVisualization<Eigen::Vector2d>, "EigenVector2d", "an Eigen::Vector2d")
CREATE_VISUALIZATION(EigenVector3dVisualization3D, PointVisualization<Eigen::Vector3d>, "EigenVector3d", "an Eigen::Vector3d")

CREATE_VISUALIZATION(Point2iVectorVisualization3D, PointVectorVisualization<Point2i>, "Point2iVector", "a vector of Point2i")
CREATE_VISUALIZATION(Point3iVectorVisualization3D, PointVectorVisualization<Point3i>, "Point3iVector", "a vector of Point3i")
CREATE_VISUALIZATION(Point2fVectorVisualization3D, PointVectorVisualization<Point2f>, "Point2fVector", "a vector of Point2f")
CREATE_VISUALIZATION(Point3fVectorVisualization3D, PointVectorVisualization<Point3f>, "Point3fVector", "a vector of Point3f")
CREATE_VISUALIZATION(Point2dVectorVisualization3D, PointVectorVisualization<Point2d>, "Point2dVector", "a vector of Point2d")
CREATE_VISUALIZATION(Point3dVectorVisualization3D, PointVectorVisualization<Point3d>, "Point3dVector", "a vector of Point3d")
CREATE_VISUALIZATION(EigenVector2iVectorVisualization3D, PointVectorVisualization<Eigen::Vector2i>, "EigenVector2iVector", "a vector of Eigen::Vector2i")
CREATE_VISUALIZATION(EigenVector3iVectorVisualization3D, PointVectorVisualization<Eigen::Vector3i>, "EigenVector3iVector", "a vector of Eigen::Vector3i")
CREATE_VISUALIZATION(EigenVector2fVectorVisualization3D, PointVectorVisualization<Eigen::Vector2f>, "EigenVector2fVector", "a vector of Eigen::Vector2f")
CREATE_VISUALIZATION(EigenVector3fVectorVisualization3D, PointVectorVisualization<Eigen::Vector3f>, "EigenVector3fVector", "a vector of Eigen::Vector3f")
CREATE_VISUALIZATION(EigenVector2dVectorVisualization3D, PointVectorVisualization<Eigen::Vector2d>, "EigenVector2dVector", "a vector of Eigen::Vector2d")
CREATE_VISUALIZATION(EigenVector3dVectorVisualization3D, PointVectorVisualization<Eigen::Vector3d>, "EigenVector3dVector", "a vector of Eigen::Vector3d")

//////////////////////////////////////////////////////////////////////////////

}} // namespace

MIRA_CLASS_SERIALIZATION(mira::gui::Point2iVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Point3iVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Point2fVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Point3fVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Point2dVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Point3dVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::EigenVector2iVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::EigenVector3iVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::EigenVector2fVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::EigenVector3fVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::EigenVector2dVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::EigenVector3dVisualization3D, mira::Visualization3D);

MIRA_CLASS_SERIALIZATION(mira::gui::Point2iVectorVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Point3iVectorVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Point2fVectorVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Point3fVectorVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Point2dVectorVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Point3dVectorVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::EigenVector2iVectorVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::EigenVector3iVectorVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::EigenVector2fVectorVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::EigenVector3fVectorVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::EigenVector2dVectorVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::EigenVector3dVectorVisualization3D, mira::Visualization3D);

