/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ImgVisualization.C
 *    A 2D visualization for different image types.
 *
 * @author Erik Einhorn
 * @date   2011/01/15
 */

#include <serialization/Serialization.h>

#include <image/Img.h>

#include <widgets/QtUtils.h>
#include <visualization/Visualization2D.h>
#include <visualization/ColormapProperty.h>

#include <QGraphicsScene>
#include <QGraphicsItem>
#include <QGraphicsPixmapItem>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

namespace Private {

///////////////////////////////////////////////////////////////////////////////

// img param is deliberately NOT const&, makes it possible to avoid copying for certain formats
// see e.g. use of asQImage() in overloads below
// (img is guaranteed to exist as long as we are using the QImage)
template <typename ImageType>
QImage toQImage(ImageType& img, float min, float max)
{
	if (img.getMat().elemSize1() > 1)
		return QtUtils::makeQImage(img, (double)min, (double)max);

	// handle cases where we can share memory instead of new allocation
	// convertFrom() will just wrap a new typed image around img's internal cv::Mat
	if (img.depth() == CV_8U) {
		if (img.depth() == 1) {
			Img8U1 img8u1 = Img8U1::convertFrom(img);
			return toQImage(img8u1, min, max);
		} else if (img.depth() == 4) {
			Img8U4 img8u4 = Img8U4::convertFrom(img);
			return toQImage(img8u4, min, max);
		}
	}

	return QtUtils::makeQImage(img);
}

// overload for Img8U1, ignores min/max
// resulting QImage can potentially share img's memory, avoiding allocation
QImage toQImage(Img8U1& img, float min, float max)
{
	if (img.step() % 4 == 0)
		return QtUtils::asQImage(img);
	else
		return QtUtils::makeQImage(img);
}

// overload for Img8U3, ignores min/max
QImage toQImage(Img8U3& img, float min, float max)
{
	return QtUtils::makeQImage(img);
}

// overload for Img8U4, ignores min/max
// resulting QImage shares img's memory, avoiding allocation
QImage toQImage(Img8U4& img, float min, float max)
{
	return QtUtils::asQImage(img);
}

// overload for Img16U1
QImage toQImage(Img16U1& img, float min, float max)
{
	return QtUtils::makeQImage(img, (uint16)floor(min), (uint16)ceil(max));
}

// overload for Img32F1
QImage toQImage(Img32F1& img, float min, float max)
{
	return QtUtils::makeQImage(img, min, max);
}

// overload for Img64F1
QImage toQImage(Img64F1& img, float min, float max)
{
	return QtUtils::makeQImage(img, (double)min, (double)max);
}

///////////////////////////////////////////////////////////////////////////////

}

/// A 2D visualization for different image types.
template <typename Image>
class ImgVisualizationBase :  public Visualization2D
{
public:
	/** @name Constructor, destructor and reflect */
	//@{

	/// The constructor.
	ImgVisualizationBase() :
		mImageItem(NULL),
		mColormap("mira::GrayscaleColormap")
	{
		mImageChannel.setDataChangedCallback(
			boost::bind(&ImgVisualizationBase::dataChanged, this, _1));
		mImageChannel.setChannelChangedCallback(
			boost::bind(&ImgVisualizationBase::channelChanged, this));
		mMin = 0.0f;
		mMax = 255.0f;
		mAlpha = 1.0f;
		mFlip = false;
	 	mFlop = false;
		mImageSize = Size2i(0,0);
	}

	/// The destructor.
	virtual ~ImgVisualizationBase()
	{
		delete mImageItem;
	}

	/// The reflect method.
	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Visualization2D);
		channelProperty(r, "Image", mImageChannel,
		                "The channel with the image to display");

		r.roproperty("Size", mImageSize, "The size of the image");

		r.property("Colormap", mColormap,
		           "The color palette (only valid for single channel images)",
		           ColormapProperty("mira::GrayscaleColormap"));
		r.property("Min", mMin,
		           setter(&ImgVisualizationBase::setMin, this),
		           "Min. value in the image, mapped to 0 (ignored for 8bit images)",
		           0.f);
		r.property("Max", mMax,
		           setter(&ImgVisualizationBase::setMax, this),
		           "Max. value in the image, mapped to 255 (ignored for 8bit images)",
		           255.f);
		r.property("Alpha", mAlpha, "Opacity 0.0-1.0, default is 1.0", 1.0f);
		r.property("Flip", mFlip,
		           "Mirror around horizontal axis (change top/bottom orientation)",
		           false);
		r.property("Flop", mFlop,
		           "Mirror around vertical axis (change left/right orientation)",
		           false);
	}

	//@}

public:
	/** @name Public implementation of Visualization2D */
	//@{

	virtual void setupScene(IVisualization2DSite* site)
	{
		QGraphicsScene* mgr = site->getSceneManager();
		mImageItem = new QGraphicsPixmapItem();
		mImageItem->setOpacity(mAlpha);

		// here we care about facing the correct way.
#if QT_VERSION > QT_VERSION_CHECK(5, 12, 0)
		mImageItem->setTransform(QTransform::fromScale(1.0, -1.0), true);
#else
		mImageItem->scale(1.0, -1.0);
#endif

		mgr->addItem(mImageItem);
	}

	virtual QGraphicsItem* getItem()
	{
		return mImageItem;
	}

	virtual void setEnabled(bool enabled)
	{
		Visualization2D::setEnabled(enabled);
		mImageItem->setVisible(enabled);
	}

	//@}

public:
	/** @name Overridden methods of Visualization */
	//@{

	virtual DataConnection getDataConnection()
	{
		return DataConnection(mImageChannel);
	}

	//@}

private:
	void dataChanged(ChannelRead<Image> img)
	{
		QImage qimg;
		try {
			// const cast enables some optimization, we are not manipulating the img content
			qimg = Private::toQImage(const_cast<Image&>(img->value()), mMin, mMax);
		}
		catch(XNotImplemented& ex) {
			MIRA_LOG_EXCEPTION(ERROR, ex);
			error("Type", "Unsupported image format");
			return;
		}

		if(qimg.format()==QImage::Format_Indexed8) {
			if(mColormap.isValid())
				qimg.setColorTable(mColormap.getColorTable());
		}
		if(mFlip || mFlop) 
			qimg = qimg.mirrored(mFlop, mFlip);
		mImageSize = img->value().size();
		mImageItem->setPixmap(QPixmap::fromImage(qimg));
		mImageItem->setOpacity(mAlpha);
		ok("Type");
	}

	void channelChanged()
	{
	}

	void setMin(float min)
	{
		if (min < mMax)
			mMin = min;
	}

	void setMax(float max)
	{
		if (max > mMin)
			mMax = max;
	}

private:
	ChannelProperty<Image> mImageChannel;

	QGraphicsPixmapItem* mImageItem;
	Size2i mImageSize;

	ColormapProperty mColormap;
	float mMin, mMax;
	float mAlpha;
	bool mFlip, mFlop;
};

///////////////////////////////////////////////////////////////////////////////

#define CREATE_IMG_VISUALIZATION(type,channels,name)           \
class ImgVisualization2D_##type##channels :                    \
    public ImgVisualizationBase<Img<type,channels>>            \
{                                                              \
    MIRA_META_OBJECT(ImgVisualization2D_##type##channels,      \
        ("Category", "Images")                                 \
        ("Name", name)                                         \
        ("Description", "Displays images"))                    \
};

CREATE_IMG_VISUALIZATION(void,  1,"Img<>")
CREATE_IMG_VISUALIZATION(uint8, 1,"Img<uint8,1>")
CREATE_IMG_VISUALIZATION(uint8, 3,"Img<uint8,3>")
CREATE_IMG_VISUALIZATION(uint8, 4,"Img<uint8,4>")
CREATE_IMG_VISUALIZATION(uint16,1,"Img<uint16,1>")
CREATE_IMG_VISUALIZATION(float, 1,"Img<float,1>")
CREATE_IMG_VISUALIZATION(double,1,"Img<double,1>")

///////////////////////////////////////////////////////////////////////////////

}} // namespace

///////////////////////////////////////////////////////////////////////////////

MIRA_CLASS_SERIALIZATION(mira::gui::ImgVisualization2D_void1,
                         mira::Visualization2D);
MIRA_CLASS_SERIALIZATION(mira::gui::ImgVisualization2D_uint81,
                         mira::Visualization2D);
MIRA_CLASS_SERIALIZATION(mira::gui::ImgVisualization2D_uint83,
                         mira::Visualization2D);
MIRA_CLASS_SERIALIZATION(mira::gui::ImgVisualization2D_uint84,
                         mira::Visualization2D);
MIRA_CLASS_SERIALIZATION(mira::gui::ImgVisualization2D_uint161,
                         mira::Visualization2D);
MIRA_CLASS_SERIALIZATION(mira::gui::ImgVisualization2D_float1,
                         mira::Visualization2D);
MIRA_CLASS_SERIALIZATION(mira::gui::ImgVisualization2D_double1,
                         mira::Visualization2D);

///////////////////////////////////////////////////////////////////////////////
