/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DepthCameraVisualization.C
 *    3d visualization for a generic depth camera image
 *
 * @author Tim Langner, Erik Einhorn, Tim van der Grinten
 * @date   2011/08/28
 */

#include <OGRE/OgreFrustum.h>
#include <OGRE/OgreManualObject.h>
#include <OGRE/OgreMaterialManager.h>
#include <OGRE/OgreHardwarePixelBuffer.h>
#include <serialization/adapters/OGRE/OgreColourValue.h>
#include <image/Img.h>

#include <serialization/SetterNotify.h>

#include <visualization/3d/ImageObject.h>
#include <visualization/Visualization3DBasic.h>
#include <visualization/ColormapProperty.h>
#include <cameraparameters/DepthCameraIntrinsic.h>

namespace mira { namespace camera {

///////////////////////////////////////////////////////////////////////////////

class DepthCameraVisualization : public Visualization3DBasic<Img<>>
{
	MIRA_META_OBJECT(DepthCameraVisualization,
			("Name", "DepthCamera")
			("Category", "Images")
			("Description", "Visualizes a depth camera frustum with image."))

	typedef Visualization3DBasic<Img<>> Base;
public:

	DepthCameraVisualization() :
		Base("Image"), mColormap("mira::GrayscaleColormap")
	{
		mColormap.setFilter(ColormapProperty::filterContinuous);

		mCameraView = false;
		mCamera = NULL;
		mAlpha = 1.0f;

		mFar = 1.0f;
		mMin =  0.0;
		mMax = -1.0;
		mImageSize = Size2i(0,0);

		mFrustum = NULL;
		mFrustumColor = Ogre::ColourValue(0.0f,0.0f,0.0f,0.5f);
		mLineColor = Ogre::ColourValue::Black;

		mImageObject = NULL;
		mDepthIntrinsicParamsChannel.setDataChangedCallback(boost::bind(&DepthCameraVisualization::onDepthIntrinsicChanged, this, _1));
	}

	virtual ~DepthCameraVisualization() {
		releaseCamera();

		if(mFrustum!=NULL)
			getSite()->getSceneManager()->destroyManualObject(mFrustum);

		delete mImageObject;

		if (!mFrustumMaterial.isNull())
			Ogre::MaterialManager::getSingletonPtr()->remove(mFrustumMaterial->getName());
	}

public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		//MIRA_REFLECT_BASE(r, Base); // bypass Visualization3DBasic to add the channel by ourself
		MIRA_REFLECT_BASE(r, Visualization3D); // but call reflect of our grandparent class

		channelProperty(r, "Image", mDataChannel, "The camera image to be visualized", NOT_REQUIRED);
		channelProperty(r, "DepthIntrinsic", mDepthIntrinsicParamsChannel, "Optional channel that"
		                " contains the normalized intrinsic depth camera parameters, if not specified"
		                " here, the parameters must be included in the image info", NOT_REQUIRED);

		r.roproperty("Size", mImageSize, "The size of the image");

		r.property("Far Plane", mFar, setterNotify(mFar, &DepthCameraVisualization::updateFrustum, this),
		           "The distance of camera far plane, i.e. the size of the visualized frustum", 1.0f);

		r.property("Camera View", mCameraView, setter(&DepthCameraVisualization::enableCameraView,this), "Enable 'onboard' camera view",false);
		r.property("Alpha", mAlpha, setter(&DepthCameraVisualization::setAlpha,this), "The alpha value of the image",1.0f, PropertyHints::limits<float>(0.0f,1.0f));

		r.property("Colormap", mColormap, "The color palette",
		           ColormapProperty("mira::GrayscaleColormap"));
		r.property("Min", mMin,
		           "Min. value in the image (only valid for float and uint16 images)",
		           0.0f);
		r.property("Max", mMax,
		           "Max. value in the image (only valid for float and uint16 images)",
		           -1.0f);

		r.property("Line Color", mLineColor, setterNotify(mLineColor, &DepthCameraVisualization::updateFrustum, this),
		           "The color of the frustum lines. Set alpha to 0 to hide the lines.", Ogre::ColourValue::Black);
		r.property("Frustum Color", mFrustumColor,  setterNotify(mFrustumColor, &DepthCameraVisualization::updateFrustum, this),
		           "The color of the frustum planes. Set alpha to 0 to hide the planes.", Ogre::ColourValue(0.0f,0.0f,0.0f,0.5f));
	}


public:

	virtual void setupScene(Ogre::SceneManager* mgr, Ogre::SceneNode* node)
	{

		mFrustumMaterial = Ogre::MaterialManager::getSingleton().create("Frustum"+toString(this)+"/Material", "MIRAMaterials");
		mFrustumMaterial->setReceiveShadows(false);
		mFrustumMaterial->setLightingEnabled(false);
		mFrustumMaterial->setCullingMode(Ogre::CULL_CLOCKWISE);
		mFrustumMaterial->setSceneBlending(Ogre::SBT_TRANSPARENT_ALPHA);
		mFrustumMaterial->setDepthWriteEnabled(false);

		// create manual object for frustum
		mFrustum = mgr->createManualObject("Frustum" + toString(this));
		node->attachObject(mFrustum);

		mImageObject = new ImageObject(false, mgr, node);
		mImageObject->setAlpha(mAlpha);

		updateFrustum();

		if(mCameraView)
			acquireCamera();
	}

	void onDataChanged(ChannelRead<Img<>> read)
	{
		const ContinuousColormap& colormap = dynamic_cast<const ContinuousColormap&>(mColormap.getColormap());
		mImageSize = read->value().size();
		mImageObject->setImage(read->value(),mMin, mMax, colormap);
	}

	void onDepthIntrinsicChanged(ChannelRead<DepthCameraIntrinsicNormalized> read)
	{
		mDepthIntrinsicParams = read->value();
		updateFrustum();
	}

	virtual void setEnabled(bool enabled)
	{
		Base::setEnabled(enabled);
		if (mImageObject != NULL)
			mImageObject->setVisible(enabled);
	}


	void updateFrustum()
	{

		const DepthCameraIntrinsicNormalized& p = mDepthIntrinsicParams; // for abbrev.

		// compute coords of frustum
		float width = 1/p.fx * mFar;
		float left = -p.cx * width;
		float right = width+left;

		float height = 1/p.fy * mFar;
		float bottom = -p.cy * height;
		float top = height+bottom;

		if(mFrustum) {
			//mFrustum->setRenderQueueGroup(Ogre::RENDER_QUEUE_MAX);
			mFrustum->clear();

			if(mFrustumColor.a>0.0f) {
				mFrustum->begin(mFrustumMaterial->getName(), Ogre::RenderOperation::OT_TRIANGLE_LIST);

				mFrustum->position(0.0f,0.0f,0.0f); mFrustum->colour(mFrustumColor);
				mFrustum->position(right, top,    mFar); mFrustum->colour(mFrustumColor);
				mFrustum->position(right, bottom, mFar); mFrustum->colour(mFrustumColor);

				mFrustum->position(0.0f,0.0f,0.0f); mFrustum->colour(mFrustumColor);
				mFrustum->position(right, bottom, mFar); mFrustum->colour(mFrustumColor);
				mFrustum->position(left,  bottom, mFar); mFrustum->colour(mFrustumColor);

				mFrustum->position(0.0f,0.0f,0.0f); mFrustum->colour(mFrustumColor);
				mFrustum->position(left, bottom,  mFar); mFrustum->colour(mFrustumColor);
				mFrustum->position(left, top,  mFar); mFrustum->colour(mFrustumColor);

				mFrustum->position(0.0f,0.0f,0.0f); mFrustum->colour(mFrustumColor);
				mFrustum->position(left,  top,     mFar); mFrustum->colour(mFrustumColor);
				mFrustum->position(right, top,     mFar); mFrustum->colour(mFrustumColor);

				mFrustum->end();
			}

			if(mLineColor.a>0.0f) {
				mFrustum->begin(mFrustumMaterial->getName(), Ogre::RenderOperation::OT_LINE_LIST);

				mFrustum->position(right, top,    mFar); mFrustum->colour(mLineColor);
				mFrustum->position(right, bottom, mFar); mFrustum->colour(mLineColor);

				mFrustum->position(right, bottom, mFar); mFrustum->colour(mLineColor);
				mFrustum->position(left,  bottom, mFar); mFrustum->colour(mLineColor);

				mFrustum->position(left,  bottom, mFar); mFrustum->colour(mLineColor);
				mFrustum->position(left,  top,    mFar); mFrustum->colour(mLineColor);

				mFrustum->position(left,  top,    mFar); mFrustum->colour(mLineColor);
				mFrustum->position(right, top,    mFar); mFrustum->colour(mLineColor);


				mFrustum->position(0.0f,0.0f,0.0f); mFrustum->colour(mLineColor);
				mFrustum->position(right, top,    mFar); mFrustum->colour(mLineColor);

				mFrustum->position(0.0f,0.0f,0.0f); mFrustum->colour(mLineColor);
				mFrustum->position(right, bottom, mFar); mFrustum->colour(mLineColor);

				mFrustum->position(0.0f,0.0f,0.0f); mFrustum->colour(mLineColor);
				mFrustum->position(left, bottom,  mFar); mFrustum->colour(mLineColor);

				mFrustum->position(0.0f,0.0f,0.0f); mFrustum->colour(mLineColor);
				mFrustum->position(left, top,     mFar); mFrustum->colour(mLineColor);

				mFrustum->end();
			}
		}

		if(mImageObject) {
			// prepare image object
			Ogre::ManualObject* imageObj = mImageObject->getPlane();
			imageObj->clear();
			imageObj->begin(mImageObject->getMaterial()->getName(), Ogre::RenderOperation::OT_TRIANGLE_FAN);

			imageObj->position(right, bottom, mFar);
			imageObj->textureCoord(1.0f, 0.0f);
			imageObj->position(left,  bottom, mFar);
			imageObj->textureCoord(0.0f, 0.0f);
			imageObj->position(left,  top,    mFar);
			imageObj->textureCoord(0.0f, 1.0f);
			imageObj->position(right, top,    mFar);
			imageObj->textureCoord(1.0f, 1.0f);

			imageObj->end();
		}

	}

	void enableCameraView(bool enable)
	{
		if(enable==mCameraView) // do nothing, if unchanged
			return;

		mCameraView = enable;

		if(getSite()!=NULL) {
			if(mCameraView) acquireCamera();
			else            releaseCamera();
		}
	}

	void setAlpha(float alpha) {
		mAlpha = alpha;
		if(mImageObject)
			mImageObject->setAlpha(mAlpha);
	}

protected:

	void cameraViewDisabled() {
		releaseCamera();
		mCameraView = false;
		mCamera = NULL;
	}

	void acquireCamera() {
		if(mCamera)
			return;
		mCamera = getSite()->acquireCamera(boost::bind(&DepthCameraVisualization::cameraViewDisabled,this));


		float tFar=mCamera->getFarClipDistance();
		float tNear=mCamera->getNearClipDistance();

		const PinholeCameraIntrinsicNormalized& p = mDepthIntrinsicParams; // for abbrev.


		// compute coords of frustum
		float width = 1/p.fx * tNear;
		float left = -p.cx * width;
		float right = width+left;

		float height = 1/p.fy * tNear;
		float bottom = -(1-p.cy) * height;
		float top = height+bottom;

		// for A,B,C,D see http://www.opengl.org/sdk/docs/man/xhtml/glFrustum.xml
		float A = (right+left)/(right-left);
		float B = (top+bottom)/(top-bottom);
		float C = -(tFar+tNear)/(tFar-tNear);
		float D = -(2*tFar*tNear)/(tFar-tNear);

		Ogre::Matrix4 m(2*tNear/(right-left), 0, A, 0,
		                0, 2*tNear/(top-bottom), B, 0,
		                0,                   0, C, D,
		                0,                   0,-1, 0);

		mCamera->setCustomProjectionMatrix(true, m);

		mFrustum->setVisible(false);
	}

	void releaseCamera() {
		if(!mCamera)
			return;
		mCamera->setCustomProjectionMatrix(false);
		getSite()->releaseCamera();
		mCamera = NULL;

		mFrustum->setVisible(true);
	}


protected:

	virtual void update(Duration dt) {


		if(mCameraView) {
			Ogre::Vector3 p;
			Ogre::Quaternion q;
			try {
				Base::update(dt);
				RigidTransform3f t = getAuthority().getTransform<RigidTransform3f>(
							mDataFrameID, getSite()->getCameraFrame(), Time::now());

				if(boost::math::isnan(t.x()) || boost::math::isnan(t.y())  || boost::math::isnan(t.z()) ||
				   boost::math::isnan(t.r.w()) || boost::math::isnan(t.r.x()) || boost::math::isnan(t.r.y()) || boost::math::isnan(t.r.z()))
					MIRA_THROW(XLogical, "Transform contains NaN values");

				p = Ogre::Vector3(t.x(),t.y(),t.z());
				q = Ogre::Quaternion(t.r.w(), t.r.x(), t.r.y(), t.r.z());
				Ogre::Quaternion r(Ogre::Vector3::UNIT_X, -Ogre::Vector3::UNIT_Y, -Ogre::Vector3::UNIT_Z);
				mCamera->setPosition(p);
				mCamera->setOrientation(q*r);
			} catch(...) {
				p = mNode->getPosition();
				q = mNode->getOrientation();
				Ogre::Quaternion r(Ogre::Vector3::UNIT_X, -Ogre::Vector3::UNIT_Y, -Ogre::Vector3::UNIT_Z);
				mCamera->setPosition(p);
				mCamera->setOrientation(q*r);
				throw;
			}
		} else
			Base::update(dt);
	}

protected:
	bool mCameraView;
	Ogre::Camera* mCamera; // the camera for the onboard camera view

	float mFar;
	float mAlpha;

	ColormapProperty mColormap;
	float mMin, mMax;

	Ogre::MaterialPtr mFrustumMaterial;
	Ogre::ManualObject* mFrustum;
	ImageObject* mImageObject;
	Size2i mImageSize;

	Ogre::ColourValue mLineColor;
	Ogre::ColourValue mFrustumColor;

	ChannelProperty<DepthCameraIntrinsicNormalized> mDepthIntrinsicParamsChannel;
	DepthCameraIntrinsicNormalized mDepthIntrinsicParams;
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

MIRA_CLASS_SERIALIZATION(mira::camera::DepthCameraVisualization, mira::Visualization3D);
