/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DepthCameraIntrinsic.C
 *    Implementation of DepthCameraIntrinsic.h.
 *
 * @author Erik Einhorn
 * @date   13/08/16
 */

#include <cameraparameters/DepthCameraIntrinsic.h>

#include <serialization/BinarySerializer.h>
#include <stream/GzStream.h>


namespace mira { namespace camera {

///////////////////////////////////////////////////////////////////////////////

DepthCameraCorrectionFactors::DepthCameraCorrectionFactors() :
		depthMin(0.0f), depthResolution(1.0f)
{
}

DepthCameraCorrectionFactors::DepthCameraCorrectionFactors(const std::string& file)
{
	loadFromFile(file);
}

void DepthCameraCorrectionFactors::saveToFile(const std::string& file)
{
	ogzstream ofs(file.c_str(), std::fstream::out | std::fstream::binary);
	if(ofs.fail())
		MIRA_THROW(XIO, "Failed to open file '" << file << "'");

	BinaryStlOstream bos(ofs);
	BinaryStreamSerializer serializer(bos);
	serializer.serialize(*this);
}

void DepthCameraCorrectionFactors::loadFromFile(const std::string& file)
{
	igzstream ifs(file.c_str(), std::fstream::in | std::fstream::binary);
	if(ifs.fail())
		MIRA_THROW(XIO, "Failed to open file '" << file << "'");

	BinaryStlIstream bis(ifs);
	BinaryStreamDeserializer deserializer(bis);
	deserializer.deserialize(*this);
}

///////////////////////////////////////////////////////////////////////////////

DepthCameraConversionParameters::DepthCameraConversionParameters() :
	A(1.0f), B(-1.0f), baseDistance(0.075f)
{
}

DepthCameraConversionParameters::DepthCameraConversionParameters(float a) :
	A(a), B(-1.0f), baseDistance(0.075f)
{
}

DepthCameraConversionParameters::DepthCameraConversionParameters(float a, float b) :
	A(a), B(b), baseDistance(0.075f)
{
}

///////////////////////////////////////////////////////////////////////////////

DepthCameraIntrinsicNormalized::DepthCameraIntrinsicNormalized()
{

}

DepthCameraIntrinsicNormalized::DepthCameraIntrinsicNormalized(
                   const PinholeCameraIntrinsicNormalized& normalizedIntrinsic,
                   float a, float b) :
	PinholeCameraIntrinsicNormalized(normalizedIntrinsic), DepthCameraConversionParameters(a,b)
{
}

void DepthCameraIntrinsicNormalized::setDepthCorrectionFactorsFile(const std::string& file)
{
	mDepthCorrectionFactorsFile = file;
}

void DepthCameraIntrinsicNormalized::setDepthCorrectionFactors(const DepthCameraCorrectionFactors& factors)
{
	mDepthCorrectionFactors = factors;
}

void DepthCameraIntrinsicNormalized::loadDepthCorrectionFactors(const std::string& file)
{
	mDepthCorrectionFactors = DepthCameraCorrectionFactors();
	mDepthCorrectionFactors->loadFromFile(file);
	setDepthCorrectionFactorsFile(file);
}

DepthCameraCorrectionFactors& DepthCameraIntrinsicNormalized::depthCorrectionFactors()
{
	if(!hasDepthCorrectionFactors())
		MIRA_THROW(XInvalidConfig, "No depth correction factors set");
	return *mDepthCorrectionFactors;
}

const DepthCameraCorrectionFactors& DepthCameraIntrinsicNormalized::depthCorrectionFactors() const
{
	if(!hasDepthCorrectionFactors())
		MIRA_THROW(XInvalidConfig, "No depth correction factors set");
	return *mDepthCorrectionFactors;
}


///////////////////////////////////////////////////////////////////////////////

} } // namespace

