/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PinholeCameraIntrinsic
 *    intrinsic parameters of a pinhole camera
 *
 * @author Erik Einhorn
 * @date   2010/11/26
 */

#ifndef _MIRA_CAMERA_PINHOLECAMERAINTRINSIC_H_
#define _MIRA_CAMERA_PINHOLECAMERAINTRINSIC_H_

#include <vector>

#include <math/Eigen.h>
#include <geometry/Size.h>

#include <cameraparameters/DistortionParameters.h>

#include <utils/IsCheapToCopy.h>

namespace mira { namespace camera {

//////////////////////////////////////////////////////////////////////////////

/**
 * Describes the intrinsic parameters of a pinhole camera (focal length, pincushion
 * distortion parameters, etc).
 * For each physical camera just one instance of this class must be used since
 * the intrinsic camera parameters remain constant.
 *
 * The focal length and the optical center parameters must be specified
 * relative to the image extension and NOT in pixel. E.g. if the focal
 * length fx was estimated using an 640x480 image during the calibration
 * then (fx/640) must be passed as normalized focal length parameter.
 */
class PinholeCameraIntrinsicNormalized : public DistortionParameters
{
public:

	/// Constructor.
	PinholeCameraIntrinsicNormalized();

	/**
	 * Sets all intrinsic parameters, such as the focal length Fx and Fy,
	 * the optical center Cx, Cy and the distortion parameters K1, K2, P1, P2.
	 *
	 * The focal length and the optical center parameters must be specified
	 * relative to the image extension and NOT in pixel. E.g. if the focal
	 * length fx was estimated using an 640x480 image during the calibration
	 * then (fx/640) must be passed as normalized focal length parameter.
	 */
	PinholeCameraIntrinsicNormalized(float iFx, float iFy, float iCx, float iCy,
	                                 float iK1, float iK2, float iP1, float iP2,
	                                 float iK3 = 0.f, float iK4 = 0.f,
	                                 float iK5 = 0.f, float iK6 = 0.f);

public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Fx", fx, "norm. coordinate of the focal point in x");
		r.property("Fy", fy, "norm. coordinate of the focal point in y");
		r.property("Cx", cx, "norm. coordinate of the optical center in x");
		r.property("Cy", cy, "norm. coordinate of the optical center in y");
		MIRA_REFLECT_BASE(r, DistortionParameters);
	}

public:

	/**
	 * Returns the camera calibration matrix of the rectified images
	 * with the given size
	 */
	Eigen::Matrix3f getK(const Size2i& imageSize) const;

	/**
	 * Returns the inverse camera calibration matrix of the rectified
	 * images with the given size
	 */
	Eigen::Matrix3f getK_inv(const Size2i& imageSize) const;

public:

	bool operator==(const PinholeCameraIntrinsicNormalized& other) const {
		return ((fx == other.fx) && (fy == other.fy) &&
		        (cx == other.cx) && (cy == other.cy) &&
		        ((DistortionParameters)*this == (DistortionParameters)other));
	}

public:

	// the intrinsic parameters
	float fx; ///< norm. coordinate of the focal point in x
	float fy; ///< norm. coordinate of the focal point in y
	float cx; ///< norm. coordinate of the optical center in x
	float cy; ///< norm. coordinate of the optical center in y
};

///////////////////////////////////////////////////////////////////////////////

class PinholeCameraIntrinsic : public DistortionParameters
{
public:

	/// Constructor.
	PinholeCameraIntrinsic();

	/**
	 * Sets all intrinsic parameters, such as the focal length Fx and Fy,
	 * the optical center Cx, Cy and the distortion parameters K1, K2, P1, P2.
	 */
	PinholeCameraIntrinsic(float iFx, float iFy, float iCx, float iCy,
	                       float iK1, float iK2, float iP1, float iP2,
	                       float iK3 = 0.f, float iK4 = 0.f,
	                       float iK5 = 0.f, float iK6 = 0.f);



	PinholeCameraIntrinsic(const PinholeCameraIntrinsicNormalized& normalizedIntrinsic,
	                       const Size2i& imageSize);

public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Fx", fx, "coordinate of the focal point in x");
		r.property("Fy", fy, "coordinate of the focal point in y");
		r.property("Cx", cx, "coordinate of the optical center in x");
		r.property("Cy", cy, "coordinate of the optical center in y");
		MIRA_REFLECT_BASE(r, DistortionParameters);
	}

public:

	/**
	 * Returns the camera calibration matrix of the rectified images
	 * with the given size
	 */
	Eigen::Matrix3f getK() const;

	/**
	 * Returns the inverse camera calibration matrix of the rectified
	 * images with the given size
	 */
	Eigen::Matrix3f getK_inv() const;

public:

	bool operator==(const PinholeCameraIntrinsic& other) const {
		return ((fx == other.fx) && (fy == other.fy) &&
		        (cx == other.cx) && (cy == other.cy) &&
		        ((DistortionParameters)*this == (DistortionParameters)other));
	}

public:
	// the intrinsic parameters
	float fx; ///< coordinate of the focal point in x [pixel]
	float fy; ///< coordinate of the focal point in y [pixel]
	float cx; ///< coordinate of the optical center in x [pixel]
	float cy; ///< coordinate of the optical center in y [pixel]
};

///////////////////////////////////////////////////////////////////////////////

} // namespace camera

template <>
class IsCheapToCopy<camera::PinholeCameraIntrinsic> : public std::true_type {};

template <>
class IsCheapToCopy<camera::PinholeCameraIntrinsicNormalized> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

} // namespace mira

#endif
