#!/bin/bash

# we need miracd from mirabash, so import that file
DIR=$(dirname "${BASH_SOURCE[0]}")
source $DIR/../mirabash

# exit on Ctrl+C
trap ctrl_c INT
function ctrl_c() {
	exit 2
}


function usage
{
	echo -e "\
Usage: mirasvn <command> [<package> [<package> ...]] [--auth] [options]\n\
Runs 'svn' for the specified packages or for all packages (if no package is specified).\n\n\
General Options:\n\
  --at <location>: only process packages in a certain location (directory). Can be used with absolute or relative paths (also e.g. '.')
  --auth: request username and password just ONCE and use it for all calls to svn.\n\
\n\
Available commands:\n\
    changes     - shows the changes on the server that were made since the last update\n\
    status (st) - status of working copy\n\
    update (up) - update to latest revision\n\
    commit (ci) - commit all changes (use with care!)\n\
    info        - info on working copy and repository\n\
\n\
    all other svn commands are supported too, type 'svn help' for a complete list"
}

function svnpackage
{
	miracd $1 &> /dev/null
	if [[ $? != 0 ]]; then
		echo -e "\e[01;31mPackage '$1' No such package found\e[00m"
	else
		here=`realpath $PWD`
		if [[ $here"/" != "$at"/* ]]; then
			return
		fi

		### check if we are in an svn directory
		svn info 2>&1 1>&/dev/null; ret=$?

		if [[ $ret == 0 ]]; then

			echo -e "\e[01;32mPackage '$1' \e[00;32m[$PWD]\e[00m"

			if [[ $2 = "up" ]]; then
				OUTPUT=`svn $2 $3`
				echo "$OUTPUT" | egrep --invert-match -e "Updating '.'" --invert-match -e "At revision"
			else
				svn $2 $3
			fi
		
		fi
	fi
}

if [[ $1 = "--help" ]] || [[ $1 = "-h" ]] || [[ $# -lt 1 ]]; then
	usage
	exit 0
fi

# first param is the command
command="$1"

# other params may be packages, the others are the options for svn
# find the first parameter, that does contain a -, it will be the first option to 
# pass to svn
paramcount=$#
firstoption=$((${paramcount}+1))

for (( i=2; i<=${paramcount}; i++ ))
do
	if [[ ${!i} == -* ]]; then
		firstoption=$i
		break
	fi
done

firstsvnoption=$firstoption

auth=""
at=""

while :
do
	if [[ ${!firstsvnoption} == "--auth" ]]; then
		read -p "Username: " username
		read -s -p "Password: " password
		echo ""
		firstsvnoption=$(($firstsvnoption+1))
		auth="--username $username --password $password"
	elif [[ ${!firstsvnoption} == "--at" ]]; then
		firstsvnoption=$(($firstsvnoption+1))
		at=`realpath ${!firstsvnoption}`
		firstsvnoption=$(($firstsvnoption+1))
		echo "Restrict to packages in directory" $at
	else
		break
	fi
done

svnoptions="${@:${firstsvnoption}}"

# handle additional commands
if [[ $command == "changes" ]]; then
	command="log -r BASE:HEAD"
fi

if [[ $command == "changedfiles" ]]; then
	command="status --show-updates --quiet"
fi

if [[ $command == "changeset" ]]; then
	command="diff -r BASE:HEAD"
fi

if [[ $firstoption == 2 ]]; then
	echo "Running 'svn $command $svnoptions' for all packages ..."
	echo "Reindexing packages ..." 
	$_mirafindpkg --reindex
	allpackages=($(bash $_mirafindpkg | awk -- '{print $1}'))
	for package in "${allpackages[@]}"
	do
		svnpackage $package "$command" "$svnoptions $auth" $at
	done
else
	# for each given package
	echo "Running 'svn $command $svnoptions' ..."
	for (( i=2; i<${firstoption}; i++ ))
	do
		package="${!i}"
		svnpackage $package "$command" "$svnoptions $auth" $at
	done
fi

