###############################################################################
# Copyright (C) 2012 by
#   MetraLabs GmbH (MLAB), GERMANY
# and
#   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
# All rights reserved.
#
# Contact: info@mira-project.org
#
# Commercial Usage:
#   Licensees holding valid commercial licenses may use this file in
#   accordance with the commercial license agreement provided with the
#   software or, alternatively, in accordance with the terms contained in
#   a written agreement between you and MLAB or NICR.
#
# GNU General Public License Usage:
#   Alternatively, this file may be used under the terms of the GNU
#   General Public License version 3.0 as published by the Free Software
#   Foundation and appearing in the file LICENSE.GPL3 included in the
#   packaging of this file. Please review the following information to
#   ensure the GNU General Public License version 3.0 requirements will be
#   met: http://www.gnu.org/copyleft/gpl.html.
#   Alternatively you may (at your option) use any later version of the GNU
#   General Public License if such license has been publicly approved by
#   MLAB and NICR (or its successors, if any).
#
# IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
# INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
# THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
# "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
# BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
# FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
# ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
# PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
#
###############################################################################
#
# Setting for using the GCC compiler
#
# Author: Erik Einhorn
#
###############################################################################

# minimum required version of GCC for MIRA is GCC 7.5
SET(REQUIRED_GCC_VERSION 7.5)

# check compiler version:
EXECUTE_PROCESS(COMMAND ${CMAKE_C_COMPILER} -dumpfullversion
				OUTPUT_VARIABLE GCC_VERSION)
IF(NOT GCC_VERSION VERSION_GREATER ${REQUIRED_GCC_VERSION} AND
   NOT GCC_VERSION VERSION_EQUAL ${REQUIRED_GCC_VERSION})
	MESSAGE(FATAL_ERROR "You need at least GCC version 7.5 to compile MIRA")
ENDIF()


SET(USE_GOLD_LINKER_CXX_FLAG "")
IF(NOT FORCE_USE_SYSTEM_LINKER)
	# Use gold linker if available, which is quicker and more memory efficient than default ld
	EXECUTE_PROCESS(COMMAND ${CMAKE_C_COMPILER} -fuse-ld=gold -Wl,--version
				ERROR_QUIET OUTPUT_VARIABLE LD_VERSION)
	IF("${LD_VERSION}" MATCHES "GNU gold")
		SET(USE_GOLD_LINKER_CXX_FLAG "-fuse-ld=gold")
	ELSE()
		MESSAGE(WARNING "GNU gold linker isn't available, using the default system linker.")
	ENDIF()
	UNSET(LD_VERSION)
ELSE()
	MESSAGE(WARNING "Use the default system linker since it is forced by the user.")
ENDIF()

# enable some important settings and warnings

# most of them are "-Wall" and "-Wextra" warnings but by specifying them separately
# we can be sure that we have the same behavior with different compiler versions
ADD_DEFINITIONS(
	-Waddress
	-Warray-bounds
	-Wchar-subscripts
	-Wcomment
	-Wdeprecated
	-Wdeprecated-declarations
	-Wformat
	-Wmain
	-Wmissing-braces
	-Wparentheses
	-Wreturn-type
	-Wsequence-point
	-Wsign-compare
	-Wstrict-aliasing
	-Wstrict-overflow=1
	-Wtrigraphs
	-Wunused-function
	-Wunused-label
	-Wunused-value
	-Wvolatile-register-var
	-Wclobbered
	-Wempty-body
	-Wignored-qualifiers
	-Wmissing-field-initializers
	-Wsign-compare
	-Wtype-limits
	#-Wfloat-equal
	-Wpointer-arith
	-Wredundant-decls
	#-Winline
)

SET(COVERAGE_FLAG "")
IF(DEFINED GENERATE_COVERAGE_FILES)
	SET(COVERAGE_FLAG "--coverage")
ENDIF()

SET(ADDITIONAL_CXX_FLAGS "-Wswitch -Wreorder -Wnon-virtual-dtor ${USE_GOLD_LINKER_CXX_FLAG}")

#make sure char is signed char (explicit setting necessary on ARM + GCC) 
#https://www.linuxtopia.org/online_books/an_introduction_to_gcc/gccintro_71.html
ADD_DEFINITIONS(-fsigned-char)


# compile with Stack Smashing Protector (SSP) for buffer overflow detection
ADD_DEFINITIONS(-fstack-protector-all)

SET(MIRA_SECURITY_LINKER_FLAGS "")
# hardening ELF binaries using Relocation Read-Only (RELRO)
# (https://www.redhat.com/en/blog/hardening-elf-binaries-using-relocation-read-only-relro)
SET(MIRA_SECURITY_LINKER_FLAGS "${MIRA_SECURITY_LINKER_FLAGS} -Wl,-z,relro,-z,now")
# No eXecute (NX) protection: mark the object as not requiring executable stack
SET(MIRA_SECURITY_LINKER_FLAGS "${MIRA_SECURITY_LINKER_FLAGS} -Wl,-z,noexecstack")

# use more secure linker flags for binaries and shared libraries
SET(CMAKE_EXE_LINKER_FLAGS    "${CMAKE_EXE_LINKER_FLAGS}    ${MIRA_SECURITY_LINKER_FLAGS}")
SET(CMAKE_SHARED_LINKER_FLAGS "${CMAKE_SHARED_LINKER_FLAGS} ${MIRA_SECURITY_LINKER_FLAGS}")

# enable c++17 standard, which is available since gcc-7 but is not the
# default until gcc-11 (https://gcc.gnu.org/projects/cxx-status.html)
IF(GCC_VERSION VERSION_LESS 11.0)
	# enable c++17 standard
	SET(ADDITIONAL_CXX_FLAGS "${ADDITIONAL_CXX_FLAGS} -std=gnu++17")
	SET(CMAKE_CXX_STANDARD 17)
ELSE()
	# keep the default c++ standard for newer gcc versions
	SET(CMAKE_CXX_STANDARD ${CMAKE_CXX_STANDARD_COMPUTED_DEFAULT})
ENDIF()

# define a macro for the file path within project,
# replacing '/','.' by '_' (to keep only characters valid for c++ identifiers)
SET(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} -D_FILE_NAME_='$(subst .,_,$(subst /,_,$(subst ${CMAKE_SOURCE_DIR}/,,$(abspath $<))))'")

# set additional debug flags (explicitly set optimize level to 0 and add gdb debug symbols)
SET(CMAKE_CXX_FLAGS_DEBUG   "-O0 -ggdb -DMIRA_SEVERITY_MAX_LEVEL=5 ${COVERAGE_FLAG} ${ADDITIONAL_CXX_FLAGS}")
SET(CMAKE_CXX_FLAGS_RELEASE "-O3 -DNDEBUG -DMIRA_SEVERITY_MAX_LEVEL=3 ${ADDITIONAL_CXX_FLAGS}")
SET(CMAKE_CXX_FLAGS_RELWITHDEBINFO "-O3 -ggdb -DNDEBUG -DMIRA_SEVERITY_MAX_LEVEL=5 ${ADDITIONAL_CXX_FLAGS}")
