/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file QtSerializationTest.C
 *    Tests serialization of Qt containers
 *
 * @author Erik Einhorn, Christof Schröter
 * @date   2012/11/01
 */

#include <boost/test/unit_test.hpp>

#include <serialization/adapters/Qt/QByteArray>
#include <serialization/adapters/Qt/QVector>
#include <serialization/adapters/Qt/QStringList>

#include <serialization/adapters/std/list>

#include <serialization/XMLSerializer.h>
#include <serialization/BinarySerializer.h>

using namespace mira;


template <typename T>
void testXML(T& v)
{
	XMLDom xml;
	XMLSerializer s(xml);
	s.serialize("Test", v);

	std::cout << "Output: " << xml.saveToString() << std::endl;

	XMLDeserializer d(xml);

	T copy;
	d.deserialize("Test",copy);

	BOOST_CHECK(v==copy);
}

template<typename T>
void testBinary()
{
	// serialize test class
	T c(true);
	BinaryBufferOstream::buffer_type buffer;
	BinaryBufferSerializer s(&buffer);
	s.serialize(c);

	// deserialize test class
	T c2;

	BinaryBufferDeserializer s2(&buffer);
	s2.deserialize(c2);
	c2.check();
}


BOOST_AUTO_TEST_CASE( TestQByteArray)
{
	QByteArray v;
	v.push_back(1);
	v.push_back(2);
	v.push_back(3);
	testXML(v);
}

BOOST_AUTO_TEST_CASE( TestQVector)
{
	QVector<int> v;
	v.push_back(1);
	v.push_back(2);
	v.push_back(3);
	testXML(v);
}

BOOST_AUTO_TEST_CASE( TestQList)
{
	QList<int> v;
	v.push_back(1);
	v.push_back(2);
	v.push_back(3);
	testXML(v);
}

BOOST_AUTO_TEST_CASE( TestQStringList)
{
	QStringList v;
	v.push_back("A");
	v.push_back("B");
	v.push_back("C");
	testXML(v);
}

//////////////////////////////////////////////////////////////////////////////

class QListAndPtr {
public:

	QListAndPtr() :
		ptr(NULL) {
	}

	QListAndPtr(bool) {
		container.push_back("-1-");
		container.push_back("-2-");
		container.push_back("-3-");
		container.push_back("-4-");
		container.push_back("-5-");
		ptr = &container.back();
	}

	void check() {
		BOOST_CHECK_EQUAL(container.size(), 5);

		QList<std::string>::iterator it = container.begin();
		for (int i = 0; i < container.size(); ++i, ++it) {
			BOOST_CHECK_EQUAL(*it, "-" + toString(i + 1) + "-");
		}

		BOOST_CHECK_EQUAL(ptr, &container.back());
	}

	template<typename Reflector>
	void reflect(Reflector& r) {
		r.member("container", container, "");
		r.member("ptr", ptr, "");
	}

public:

	QList<std::string> container;
	std::string* ptr;
};

//////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( TestQListPtrTracking)
{
	testBinary<QListAndPtr>();
}

