/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PropertyEditorTreeView.h
 *    For internal use by PropertyEditor only!
 *
 * @author Erik Einhorn
 * @date   2011/05/09
 */

#ifndef _MIRA_PROPERTYEDITOR_H_
#  error "This file must be included by PropertyEditor.C"
#endif

#include <QApplication>
#include <QStyledItemDelegate>
#include <QPainter>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class PropertyEditor::ItemDelegate: public QStyledItemDelegate
{
public:
	ItemDelegate(PropertyEditor* editor):
		QStyledItemDelegate(editor), mEditor(editor) {}

	virtual ~ItemDelegate() {}

public:

	QWidget* createEditor(QWidget* parent, const QStyleOptionViewItem& option,
	                      const QModelIndex& index) const;

	QSize sizeHint(const QStyleOptionViewItem& option,
	               const QModelIndex& index) const;

	void setEditorData(QWidget* editor, const QModelIndex& index) const	{}

	void paint(QPainter* painter, const QStyleOptionViewItem& option,
			const QModelIndex& index) const;

private:

	PropertyEditor* mEditor;
};

///////////////////////////////////////////////////////////////////////////////

QWidget* PropertyEditor::ItemDelegate::createEditor(QWidget* parent,
                                                    const QStyleOptionViewItem& option,
                                                    const QModelIndex& index) const
{
	PropertyNode* node = mEditor->mModel->index2node(index);
	if(node->isReadOnly())
		return NULL; // no editing for read only properties
	else {
		QWidget* editor = mEditor->createEditor(node, parent);
		if(editor!=NULL) {
			editor->setAutoFillBackground(true);
		}
		return editor;
	}

}

QSize PropertyEditor::ItemDelegate::sizeHint(const QStyleOptionViewItem& option,
                                             const QModelIndex& index) const
{
	return QStyledItemDelegate::sizeHint(option, index) + QSize(2, 2);
}

void PropertyEditor::ItemDelegate::paint(QPainter* painter,
                                         const QStyleOptionViewItem& option,
                                         const QModelIndex& index) const
{
	QStyledItemDelegate::paint(painter, option, index);

	const QColor color = static_cast<QRgb>(QApplication::style()->styleHint(QStyle::SH_Table_GridLineColor, &option));
	const QPen oldPen = painter->pen();
	painter->setPen(QPen(color));

	if(index.column()==0) {
		// draw vertical separation between left and right column
		painter->drawLine(option.rect.right(), option.rect.y(),
		                  option.rect.right(), option.rect.bottom());
	}

	// draw horizontal separator line
	painter->drawLine(option.rect.x(), option.rect.bottom(),
	                  option.rect.right(), option.rect.bottom());
	painter->setPen(oldPen);
}

///////////////////////////////////////////////////////////////////////////////

}
