/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ScrollView.C
 *    Implementation of ScrollView.h.
 *
 * @author Tim Langner
 * @date   2011/12/30
 */

#include <widgets/ScrollView.h>

#include <QPainter>
#include <QPaintEvent>
#include <QScrollBar>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

ScrollView::ScrollView(QWidget* parent) :
	QAbstractScrollArea(parent)
{
	QWidget* viewport = QAbstractScrollArea::viewport();
	viewport->setAttribute(Qt::WA_StaticContents);
	viewport->setAttribute(Qt::WA_NoSystemBackground);
	viewport->setBackgroundRole(QPalette::NoRole);
	viewport->setAutoFillBackground(false);
}

void ScrollView::setContentsPos(int x, int y)
{
	if (x < 0)
		x = 0;
	if (x != mContentsRect.x())
		QAbstractScrollArea::horizontalScrollBar()->setSliderPosition(x);

	if (y < 0)
		y = 0;
	if (y != mContentsRect.y())
		QAbstractScrollArea::verticalScrollBar()->setSliderPosition(y);
}

void ScrollView::resizeContents(int width, int height)
{
	if (width >= 0 && mContentsRect.width() != width)
		mContentsRect.setWidth(width);

	if (height >= 0 && mContentsRect.height() != height)
		mContentsRect.setHeight(height);

	updateScrollBars();
}

void ScrollView::ensureVisible(int x, int y, int marginX, int marginY)
{
	QWidget *viewport = QAbstractScrollArea::viewport();
	int w = viewport->width();
	int h = viewport->height();

	int dx = -mContentsRect.x();
	int dy = -mContentsRect.y();
	int cw = mContentsRect.width();
	int ch = mContentsRect.height();

	if (w < marginX * 2)
		marginX = w / 2;

	if (h < marginY * 2)
		marginY = h / 2;

	if (cw <= w) {
		marginX = 0;
		dx = 0;
	}

	if (ch <= h) {
		marginY = 0;
		dy = 0;
	}

	if (x < marginX - dx)
		dx = marginX - x;
	else
	if (x >= w - marginX - dx)
		dx = w - marginX - x;

	if (y < marginY - dy)
		dy = marginY - y;
	else
	if (y >= h - marginY - dy)
		dy = h - marginY - y;

	if (dx > 0)
		dx = 0;
	else if (dx < w - cw && cw > w)
		dx = w - cw;

	if (dy > 0)
		dy = 0;
	else if (dy < h - ch && ch > h)
		dy = h - ch;

	setContentsPos(-dx, -dy);
}

QPoint ScrollView::viewportToContents(const QPoint& pos) const
{
	return QPoint(pos.x() + mContentsRect.x(), pos.y() + mContentsRect.y());
}

QPoint ScrollView::contentsToViewport(const QPoint& pos) const
{
	return QPoint(pos.x() - mContentsRect.x(), pos.y() - mContentsRect.y());
}

void ScrollView::updateScrollBars()
{
	QWidget *viewport = QAbstractScrollArea::viewport();
	int w = viewport->width()  - 2;
	int h = viewport->height() - 2;

	QScrollBar* hScrollBar = QAbstractScrollArea::horizontalScrollBar();
	int cw = (mContentsRect.width() > w ? mContentsRect.width() - w : 0);
	if (hScrollBar->sliderPosition() > cw)
		hScrollBar->setSliderPosition(cw);
	hScrollBar->setRange(0, cw);
	hScrollBar->setSingleStep((w >> 4) + 1);
	hScrollBar->setPageStep(w);

	QScrollBar* vScrollBar = QAbstractScrollArea::verticalScrollBar();
	int ch = (mContentsRect.height() > h ? mContentsRect.height() - h : 0);
	if (vScrollBar->sliderPosition() > ch)
		vScrollBar->setSliderPosition(ch);
	vScrollBar->setRange(0, ch);
	vScrollBar->setSingleStep((h >> 4) + 1);
	vScrollBar->setPageStep(h);
}

void ScrollView::resizeEvent(QResizeEvent* resizeEvent)
{
	QAbstractScrollArea::resizeEvent(resizeEvent);
	updateScrollBars();
}

void ScrollView::paintEvent(QPaintEvent* paintEvent)
{
	QPainter painter(QAbstractScrollArea::viewport());
	drawContents(&painter, paintEvent->rect());
}

void ScrollView::wheelEvent(QWheelEvent* wheelEvent)
{
	if (wheelEvent->modifiers()
		& (Qt::ShiftModifier | Qt::ControlModifier)) {
		setContentsPos(
			mContentsRect.x() - wheelEvent->delta(),
			mContentsRect.y());
	}
	else QAbstractScrollArea::wheelEvent(wheelEvent);
}

void ScrollView::scrollContentsBy(int dx, int dy)
{
	int update = 0;

	if (dx) {
		mContentsRect.moveLeft(mContentsRect.x() - dx);
		++update;
	}

	if (dy) {
		mContentsRect.moveTop(mContentsRect.y() - dy);
		++update;
	}

	if (update > 0) {
		updateContents();
		emit contentsMoving(mContentsRect.x(), mContentsRect.y());
	}
}

void ScrollView::updateContents(const QRect& rect)
{
	QAbstractScrollArea::viewport()->update(
		QRect(contentsToViewport(rect.topLeft()), rect.size()));
}

void ScrollView::updateContents()
{
	QAbstractScrollArea::viewport()->update();
}

///////////////////////////////////////////////////////////////////////////////

}
