/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MeshObject.C
 *    Implementation of MeshObject.
 *
 * @author Tim Langner
 * @date   2011/01/04
 */

#include <utils/ToString.h>
#include <utils/Path.h>
#include <visualization/3d/MeshObject.h>
#include <visualization/3d/MeshFactory.h>

#include <OGRE/Ogre.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

MeshObject::MeshObject(const std::string& mesh, Ogre::SceneManager* sceneManager,
                       Ogre::SceneNode* parent) :
	VisualizationObject(sceneManager, parent),
	mEntity(NULL)
{
	create(mesh);
}

MeshObject::MeshObject(const std::string& mesh, Ogre::MaterialPtr material,
                       Ogre::SceneManager* sceneManager, Ogre::SceneNode* parent) :
	VisualizationObject(sceneManager, parent),
	mEntity(NULL),
	mMaterial(material)
{
	create(mesh);
}

MeshObject::~MeshObject()
{
	if (!mMaterial.isNull() && mOwnsMaterial)
		Ogre::MaterialManager::getSingletonPtr()->remove(mMaterial->getName());
	if (mEntity != NULL)
	{
		mNode->detachObject(mEntity);
		mSceneManager->destroyEntity(mEntity);
	}
}

void MeshObject::setColor(const Ogre::ColourValue& color)
{
	mMaterial->getTechnique(0)->setAmbient(color * 0.5);
	mMaterial->getTechnique(0)->setDiffuse(color);
	if(color.a >= 1.0f) { // completely opaque
		mMaterial->setSceneBlending(Ogre::SBT_REPLACE);
		mMaterial->getTechnique(0)->setDepthWriteEnabled(true);
	} else {
		mMaterial->setSceneBlending(Ogre::SBT_TRANSPARENT_ALPHA);
		mMaterial->getTechnique(0)->setDepthWriteEnabled(false);
	}
}

void MeshObject::create(const std::string& filename)
{
	mOwnsMaterial = false;
	std::string meshname = filename;
	if (mMaterial.isNull())
	{
		mOwnsMaterial = true;
		std::string materialName = "MeshObject"+toString(this)+"/Material";
		mMaterial = Ogre::MaterialManager::getSingleton().create(materialName, "MIRAMaterials");
		mMaterial->setReceiveShadows(false);
		mMaterial->getTechnique(0)->setLightingEnabled(true);
		setColor(Ogre::ColourValue(1,1,1,1)); // default = opaque white
	}
	Ogre::ResourceGroupManager& mgr = Ogre::ResourceGroupManager::getSingleton();
	bool resourceExists = mgr.resourceExists("MIRAMeshs", meshname);
	if(!resourceExists || Path(meshname).extension() != ".mesh")
	{
		if (resourceExists)
		{
			Ogre::FileInfoListPtr fileInfos = mgr.findResourceFileInfo("MIRAMeshs", meshname);
			Ogre::FileInfoList::iterator it = fileInfos->begin();
			if(it != fileInfos->end())
				meshname = (Path(it->archive->getName()) / Path(it->basename)).string();
		}
		meshname = resolvePath(meshname).string();
		if (!boost::filesystem::exists(meshname))
			MIRA_THROW(XInvalidConfig, "The mesh '" << meshname << "' could not be found.");
		Ogre::MeshPtr mesh = MeshFactory::instance().loadMesh(meshname);
		meshname = mesh->getName();
	}

	mEntity = mSceneManager->createEntity("MeshObject"+toString(this), meshname);
	mNode->attachObject(mEntity);

	if (mEntity != NULL)
	{
		for (uint32 i = 0; i < mEntity->getNumSubEntities(); ++i)
		{
			Ogre::SubEntity* sub = mEntity->getSubEntity(i);
			const std::string& material_name = sub->getMaterialName();
			if (material_name.empty() || material_name == "BaseWhite" || material_name == "BaseWhiteNoLighting")
				sub->setMaterialName(mMaterial->getName());
		}
	}
}

//////////////////////////////////////////////////////////////////////////////

}
