/*
 * Copyright (C) 2014 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PolygonTool2D.C
 *    Implementation of PolygonTool2D.h
 *
 * @author Christian Reuther
 * @date   2018/08/29
 */

#include <visualization/2d/PolygonTool2D.h>

#include <QMouseEvent>
#include <QGraphicsView>
#include <QDebug>
#include <qmath.h>

#include <serialization/Serialization.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

PathPointItem::PathPointItem(qreal radius, QGraphicsItem* parent) : QAbstractGraphicsShapeItem(parent), RADIUS(radius) {
	setFlag(ItemIsMovable);
	setFlag(ItemIgnoresTransformations);
	setZValue(100.0);
}

PathPointItem::~PathPointItem() {
	// Empty
}

QRectF PathPointItem::boundingRect() const {
	return QRectF(-RADIUS, -RADIUS, 2 * RADIUS, 2 * RADIUS);
}

void PathPointItem::paint(QPainter* painter, const QStyleOptionGraphicsItem*, QWidget*) {
	painter->setPen(pen());
	painter->setBrush(brush());
	painter->setOpacity(opacity());

	painter->drawEllipse(boundingRect());
}

///////////////////////////////////////////////////////////////////////////////

PolygonTool2D::PolygonTool2D() {
	mItem = NULL;
	mPolygonItem = NULL;
}

PolygonTool2D::~PolygonTool2D() {
	auto site = this->getSite();
	if(!site || !site->getSceneManager())
		return;

	reset();

	if(mPolygonItem) {
		mItem->removeFromGroup(mPolygonItem);
		delete mPolygonItem;
		mPolygonItem = NULL;
	}

	site->getSceneManager()->removeItem(mItem);

	// Do not delete mItem as the base class destructor does this.
}

Polygon2f PolygonTool2D::polygon() const {
	Polygon2f poly;
	foreach(const QPointF& p, mPathPoints)
		poly.push_back(Point2f(p.x(), p.y()));
	return poly;
}

QPen PolygonTool2D::pen() const {
	return mPen;
}

QBrush PolygonTool2D::brush() const {
	return mBrush;
}

void PolygonTool2D::setPen(QPen pen) {
	mPen = pen;
	updatePenAndBrush();
}

void PolygonTool2D::setBrush(QBrush brush) {
	mBrush = brush;
	updatePenAndBrush();
}

///////////////////////////////////////////////////////////////////////////////

void PolygonTool2D::onMousePressed(QMouseEvent* e) {
	// The right mouse button disables/deactivates the tool
	if(e->button() == Qt::RightButton) {
		Polygon2f poly;
		foreach(const QPointF& p, mPathPoints)
			poly.push_back(Point2f(p.x(), p.y()));
		onNewPolygon(poly);

		deactivate();
		return;
	}

	if(e->button() != Qt::LeftButton)
		return;

	mItem->setVisible(true);
	e->accept();
}

void PolygonTool2D::onMouseReleased(QMouseEvent* e) {
	if(e->button() != Qt::LeftButton)
		return;

	const QPointF pos = getSite()->getViewManager()->mapToScene(e->pos());
	const QPointF fixedPos = QPointF(pos.x(), pos.y());
	mPathPoints.append(fixedPos);

	QPolygonF poly(mPathPoints);
	poly.append(mPathPoints.front());
	mPolygonItem->setPolygon(poly);

	PathPointItem* item = new PathPointItem(2.0);
	item->setPos(pos);
	item->setPen(mPen);
	item->setBrush(QBrush(mPen.color(), Qt::SolidPattern));
	mPointItems.append(item);
	mItem->addToGroup(item);

	e->accept();
}

void PolygonTool2D::activate() {
	reset();
	VisualizationTool2D::activate();
	mItem->setVisible(false); // Will be activated by left click
}

void PolygonTool2D::deactivate() {
	VisualizationTool2D::deactivate();
	reset();
}

void PolygonTool2D::reset() {
	mPathPoints.clear();
	mPolygonItem->setPolygon(QPolygonF());

	foreach(PathPointItem* item, mPointItems) {
		mItem->removeFromGroup(item);
		delete item;
	}
	mPointItems.clear();
}

QIcon PolygonTool2D::getIcon() {
	QPixmap pixmap(":/icons/ToolPolygon.png");
	return QIcon(pixmap);
}

///////////////////////////////////////////////////////////////////////////////

void PolygonTool2D::init(IVisualizationSite* site) {
	VisualizationTool2D::init(site);

	const float baseZ = getSite()->getToolBaseZValue();
	mItem->setZValue(baseZ + 0.0f);
}

QGraphicsItem* PolygonTool2D::setupScene(QGraphicsScene* mgr) {
	mPen.setColor(QColor(172, 0, 0, 200));
	mPen.setCosmetic(true);

	mBrush.setColor(QColor(255, 255, 255, 64));
	mBrush.setStyle(Qt::SolidPattern);

	mPolygonItem = new QGraphicsPolygonItem();
	mPolygonItem->setPen(mPen);
	mPolygonItem->setBrush(mBrush);

	mItem = new QGraphicsItemGroup();
	mItem->addToGroup(mPolygonItem);
	mItem->setVisible(false);
	mgr->addItem(mItem);

	return mItem;
}

void PolygonTool2D::updatePenAndBrush() {
	if(mPolygonItem) {
		mPolygonItem->setPen(mPen);
		mPolygonItem->setBrush(mBrush);
	}
	foreach(PathPointItem* item, mPointItems) {
		item->setPen(mPen);
		item->setBrush(QBrush(mPen.color(), Qt::SolidPattern));
	}
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::PolygonTool2D, mira::VisualizationTool2D)
