/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationControlPage.C
 *    Implementation of VisualizationControlPage.h.
 *
 * @author Erik Einhorn
 * @date   2010/12/30
 */

#include <views/VisualizationControlPage.h>

#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QPushButton>
#include <QInputDialog>
#include <QAction>

#include <serialization/PropertySerializer.h>

#include <views/VisualizationView.h>
#include <widgets/PropertyEditor.h>
#include <widgets/SingleSelectionListDialog.h>
#include <visualization/Visualization.h>
#include <visualization/ChannelProperty.h>

///////////////////////////////////////////////////////////////////////////////

MIRA_CLASS_REGISTER(mira::VisualizationControlPage, mira::Object)


namespace mira {

///////////////////////////////////////////////////////////////////////////////

VisualizationControlPage::VisualizationControlPage(VisualizationView* view) :
	QWidget(NULL), mView(view)
{
	QVBoxLayout* verticalLayout = new QVBoxLayout(this);
	//verticalLayout->setMargin(0);
	verticalLayout->setSpacing(0);
	verticalLayout->setContentsMargins(0,3,0,0);

	mEditor = new PropertyEditor(this);
	mEditor->installFilterShortCut(this);
	verticalLayout->addWidget(mEditor);


	QHBoxLayout* horizontalLayout = new QHBoxLayout();
	mBtVisualizationAdd = new QPushButton(this);
	mBtVisualizationAdd->setIcon(QIcon(":/icons/Plus.png"));
	mBtVisualizationAdd->setToolTip("Add a new visualization");
	horizontalLayout->addWidget(mBtVisualizationAdd);

	mBtVisualizationRemove = new QPushButton(this);
	mBtVisualizationRemove->setIcon(QIcon(":/icons/Minus.png"));
	mBtVisualizationRemove->setToolTip("Remove the selected visualization");
	horizontalLayout->addWidget(mBtVisualizationRemove);

	horizontalLayout->addStretch();

	mBtVisualizationUp = new QPushButton(this);
	mBtVisualizationUp->setIcon(QIcon(":/icons/ArrowUp.png"));
	mBtVisualizationUp->setToolTip("Move visualization upwards");
	mBtVisualizationUp->setEnabled(true);
	horizontalLayout->addWidget(mBtVisualizationUp);

	mBtVisualizationDown = new QPushButton(this);
	mBtVisualizationDown->setIcon(QIcon(":/icons/ArrowDown.png"));
	mBtVisualizationDown->setToolTip("Move visualization downwards");
	mBtVisualizationDown->setEnabled(true);
	horizontalLayout->addWidget(mBtVisualizationDown);


	verticalLayout->addLayout(horizontalLayout);

	connect(mBtVisualizationAdd, SIGNAL(clicked()),
	        this, SLOT(addVisualizationClicked()));
	connect(mBtVisualizationRemove, SIGNAL(clicked()),
	        this, SLOT(removeVisualizationClicked()));
	connect(mBtVisualizationUp, SIGNAL(clicked()),
	        this, SLOT(moveUpVisualizationClicked()));
	connect(mBtVisualizationDown, SIGNAL(clicked()),
	        this, SLOT(moveDownVisualizationClicked()));

	// Add shortcut for removeVisualizationClicked()
	QAction* removeVisualizationAction = new QAction(this);
	removeVisualizationAction->setShortcutContext(Qt::WidgetWithChildrenShortcut);
	removeVisualizationAction->setShortcut(QKeySequence(Qt::Key_Delete));
	this->addAction(removeVisualizationAction);
	connect(removeVisualizationAction, SIGNAL(triggered(bool)),
	        this, SLOT(removeVisualizationClicked()));


	// add properties of view as General properties:
	PropertySerializer s;
	PropertyNode* p = s.reflectProperties("General", view);
	mEditor->addProperty(p);
	mEditor->setColor(p, PropertyEditor::lightBlue);

	// add all existing visualizations to the property editor
	foreach(Visualization* vis, mView->getVisualizations())
		addVisualizationToEditor(vis);
}

void VisualizationControlPage::addVisualizationClicked()
{
	SingleSelectionListDialog<ClassProxy> d(this);
	d.setWindowTitle(tr("Choose a visualization"));
	d.setHeaderLabels(QStringList() << "Visualization" << "Description");
	d.resize(600,400);

	typedef std::map<std::string, ClassProxy > ClassMap;
	ClassMap classes = mView->supportedVisualizationClass().getDerivedClasses();

	foreach(ClassMap::value_type i, classes)
	{
		if(i.second.isAbstract())
			continue;

		std::string name = i.second.getMetaInfo("Name");
		if(name.empty())
			name = i.first; // use class name, if no meta name was specified

		std::string category = i.second.getMetaInfo("Category");
		if(category.empty())
			category = "General"; // use default category

		QStringList labels;
		labels << name.c_str() << i.second.getMetaInfo("Description").c_str();
		d.addItem(category.c_str(), labels, i.second);
	}

	if(d.exec()==0)
		return;

	if(!d.hasSelection())
		return; // nothing chosen

	auto selection = d.selectedItem();

	// instantiate a new visualization
	ClassProxy p = selection.second;
	Visualization* vis = p.newInstance<Visualization>();
	vis->setName(selection.first.toLocal8Bit().data());

	// and add it to the view
	mView->addVisualization(vis);
	addVisualizationToEditor(vis);
}

void VisualizationControlPage::removeVisualizationClicked()
{
	PropertyNode* current = mEditor->currentProperty();
	if(current==NULL)
		return;

	Visualization* vis = (Visualization*)mEditor->getUser(current).value<void*>();
	if(vis!=NULL) {
		mView->removeVisualization(vis);
		mEditor->removeProperty(mEditor->currentProperty());
		delete vis; // delete the visualization since it is not longer maintained by the view
	}
}

void VisualizationControlPage::moveUpVisualizationClicked()
{
	PropertyNode* current = mEditor->currentProperty();
	if(current==NULL)
		return;

	Visualization* vis = (Visualization*)mEditor->getUser(current).value<void*>();
	if(vis!=NULL) {
		mView->moveUpVisualization(vis);
		mEditor->moveUpProperty(current);
	}
}

void VisualizationControlPage::moveDownVisualizationClicked()
{
	PropertyNode* current = mEditor->currentProperty();
	if(current==NULL)
		return;

	Visualization* vis = (Visualization*)mEditor->getUser(current).value<void*>();
	if(vis!=NULL) {
		mView->moveDownVisualization(vis);
		mEditor->moveDownProperty(current);
	}
}

void VisualizationControlPage::addVisualizationToEditor(Visualization* vis)
{
	PropertySerializer s;
	PropertyNode* p = s.reflectProperties(vis->getName(), vis);
	mEditor->addProperty(p, QVariant::fromValue((void*)vis));
	mEditor->setColor(p, PropertyEditor::lightGreen);
}

///////////////////////////////////////////////////////////////////////////////

}
