
/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ServiceSelectorBox.C
 *    Implementation of ServiceSelectorBox.h.
 *
 * @author Christof Schröter
 * @date   2024/12/18
 */

#include <fw/ServiceSelectorBox.h>

#include <fw/Framework.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

ServiceSelectorBox::ServiceSelectorBox(QWidget* parent) : QComboBox(parent)
{
	setAutoHide(true);
	connect(this, SIGNAL(currentIndexChanged(int)),
	        this, SLOT(selected(int)));
}

void ServiceSelectorBox::setAutoHide(bool hide)
{
	mAutoHide = hide;
	if (hide)
		QComboBox::setVisible(QComboBox::count() > 1);
	else
		QComboBox::setVisible(true);
}

bool ServiceSelectorBox::updateServices(bool localFirst)
{
	RPCManager& rpcman = MIRA_FW.getRPCManager();
	std::set<std::string> localServices = rpcman.getLocalServices();

	std::list<std::string> matchingServices;

	foreach(const auto& ifc, mInterfaces)
	{
		auto services = rpcman.queryServicesForInterface(ifc);
		services.sort();
		matchingServices.merge(services);
		matchingServices.unique();
	}

	QString current = QComboBox::currentText(); // remember previous selection

	blockSignals(true);
	QComboBox::clear();
	mServices.clear();
	mServices.reserve(matchingServices.size());

	int countLocal = 0;
	foreach(const std::string& s, matchingServices) {
		bool isLocal = (localServices.find(s) != localServices.end());
		QString item = QString::fromStdString((isLocal ? "local  : " : "remote : ") + s);
		if (localFirst && isLocal) {
			mServices.insert(mServices.begin()+countLocal, s);
			QComboBox::insertItem(countLocal, item);
			++countLocal;
		} else {
			mServices.push_back(s);
			QComboBox::addItem(item);
		}

		if (s == mPreferredService) {
			current = item;
			mPreferredService.clear(); // only prefer it when it first appears
		}
	}

	// keep selection, if still valid
//	mServiceBox->setCurrentText(current); // works in Qt5, but not Qt4
	int index = QComboBox::findText(current);
	if (index >= 0)
		QComboBox::setCurrentIndex(index);

	if (mAutoHide)
		QComboBox::setVisible(QComboBox::count() > 1);

	blockSignals(false);

	// the previously selected index is not found anymore
	if (index < 0) {
		emit selected(mServices[currentIndex()]);
		return true;
	}

	return false;
}

std::string ServiceSelectorBox::getSelectedService()
{
	if (mServices.empty())
		return std::string();

	return mServices[currentIndex()];
}

///////////////////////////////////////////////////////////////////////////////

}
