/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file QtUtils.h
 *    Declaration of helper functions related to Qt.
 *
 * @author Tim Langner, Christof Schröter
 * @date   2011/02/06
 */

#ifndef _MIRA_QTUTILS_H_
#define _MIRA_QTUTILS_H_

#include <image/Img.h>
#include <transform/RigidTransform.h>

#include <QString>
#include <QFileDialog>

class QImage;
class QGraphicsItem;
class QFileDialog;
class QWidget;

namespace mira { namespace QtUtils {

///////////////////////////////////////////////////////////////////////////////

/// Create QImage from 8bit unsigned image.
QImage makeQImage(const Img8U1& img);

/// Create QImage from 8bit signed image.
QImage makeQImage(const Img<int8, 1>& img);

/// Create QImage from 3-channel 8bit unsigned image.
QImage makeQImage(const Img8U3& img);

/// Create QImage from 4-channel 8bit unsigned image.
QImage makeQImage(const Img8U4& img);

/**
 * Create QImage from 16bit unsigned image.
 * Pixel values are mapped from (minVal,maxVal) interval to (0,255) (and saturated).
 * @throw XInvalidParameter if maxVal <= minVal.
 */
QImage makeQImage(const Img16U1& img, uint16_t minVal, uint16_t maxVal);

/**
 * Create QImage from 16bit signed image.
 * Pixel values are mapped from (minVal,maxVal) interval to (0,255) (and saturated).
 * @throw XInvalidParameter if maxVal <= minVal.
 */
QImage makeQImage(const Img16S1& img, int16_t minVal, int16_t maxVal);

/**
 * Create QImage from float image.
 * Pixel values are mapped from (minVal,maxVal) interval to (0,255) (and saturated).
 * @throw XInvalidParameter if maxVal <= minVal.
 */
QImage makeQImage(const Img32F1& img, float minVal, float maxVal);

/**
 * Create QImage from double image.
 * Pixel values are mapped from (minVal,maxVal) interval to (0,255) (and saturated).
 * @throw XInvalidParameter if maxVal <= minVal.
 */
QImage makeQImage(const Img64F1& img, double minVal, double maxVal);

/**
 * Create QImage from generic image, checks actual format and calls
 * the respective specific overload for handling.
 * @throw XNotImplemented if conversion is not implemented for image pixel format.
 * @throw XInvalidParameter if image pixel format is other than 8bit (single or multi channel).
 */
QImage makeQImage(const Img<>& img);

/**
 * Create QImage from generic image, checks actual format and calls
 * the respective specific overload for handling.
 * Pixel values are mapped from (minVal,maxVal) interval to (0,255) (and saturated).
 * @throw XNotImplemented if conversion is not implemented for image pixel format.
 * @throw XInvalidParameter if image pixel format is 8bit (single or multi channel).
 * @throw XInvalidParameter if maxVal <= minVal.
 */
QImage makeQImage(const Img<>& img, double minVal, double maxVal);

/**
 * Create QImage wrapper for 8bit unsigned image (sharing img's memory).
 * QImage requires all image data to be 32bit-aligned, so this throws
 * XInvalidParameter if @ref Img8U1::step() "img.step()" does not return a multiple of 4.
 */
QImage asQImage(Img8U1& img);

 /**
 * Create QImage from 8bit unsigned image.
 * QImage requires all image data to be 32bit-aligned, so this throws
 * XInvalidParameter if @ref Img8U1::step() "img.step()" does not return a multiple of 4.
 *
 * The required setColorTable() will trigger a deep-copy internally (as originally
 * shared image data is read-only in the const& case), so the resulting QImage will NOT
 * share memory of the input img and there is no real advantage over makeQImage().
 * @note This limitation may change in Qt 6.4 or later. See #882 comment 2.
 */
QImage asQImage(const Img8U1& img);

/// Create QImage wrapper for 4-channel 8bit unsigned image (sharing img's memory)
QImage asQImage(Img8U4& img);

/**
 * Create (const) QImage wrapper for 4-channel 8bit unsigned image.
 * This will share img's memory, but mark it read-only. Non-const access to the
 * QImage will result in automatic deep-copying.
 */
QImage asQImage(const Img8U4& img);

/**
 * Convert a QImage to an image.
 */
void fromQImage(const QImage& qimage, Img<>& oImg);

/**
 * Applies a transform (translation and rotation) to a graphics scene item.
 */
void setTransform(QGraphicsItem* item, const RigidTransform2f& t);

/**
 * Enforce the use of non-native FileDialog-methods to prevent platform depending behavior differences
 */
QString getOpenFileName(QWidget* parent = NULL,
                                                const QString& caption = QString(),
                                                const QString& dir = QString(),
                                                const QString& filter = QString(),
                                                QString* selectedFilter = NULL,
                                                QFileDialog::Options options = QFileDialog::DontUseNativeDialog);

/**
 * Enforce the use of non-native FileDialog-methods to prevent platform depending behavior differences
 * @param[in] enforceExtension If a non-empty list is provided as argument, the method will make sure
 *                             one of these extensions is selected, replacing the extension entered by the user
 *                             with the first element in the list if necessary.
 */
QString getSaveFileName(QWidget* parent = NULL,
                                                const QString& caption = QString(),
                                                const QString& dir = QString(),
                                                const QString& filter = QString(),
                                                QString* selectedFilter = NULL,
                                                QFileDialog::Options options = QFileDialog::DontUseNativeDialog,
                                                const QStringList& enforceExtension = QStringList());
/**
 * Enforce the use of non-native FileDialog-methods to prevent platform depending behavior differences
 */
QString getExistingDirectory(QWidget* parent = NULL,
                                                     const QString& caption = QString(),
                                                     const QString& dir = QString(),
                                                     QFileDialog::Options options = QFileDialog::DontUseNativeDialog | QFileDialog::ShowDirsOnly);

/**
 * Init painter's pen, background and font from widget. Replaces QPainter::initFrom(QPaintDevice*) deprecated in Qt5.15.
 */
void initPainterFrom(QPainter& painter, const QWidget& widget);

///////////////////////////////////////////////////////////////////////////////

}}

#endif
