/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ViewPart.h
 *    Declaration of the ViewPart.
 *
 * @author Erik Einhorn
 * @date   2010/12/01
 */

#ifndef _MIRA_VIEWPART_H_
#define _MIRA_VIEWPART_H_

#include <rcp/WorkbenchPart.h>

#include <QDockWidget>

class QMenu;
class QToolButton;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class ViewPartTitleBar;

/**
 * A view is typically used to display information or properties. Views are
 * shown as dock windows. Only one instance of a particular view type may
 * exist within the workbench. This policy is designed to simplify the view
 * management for the user.
 */
class ViewPart : public QDockWidget, public WorkbenchPart
{
	Q_OBJECT
	MIRA_ABSTRACT_OBJECT(ViewPart)
public:
	ViewPart();
	virtual ~ViewPart();

public:
	virtual void init(IWorkbenchPartSite* site);

	IWorkbenchPartSite* getSite();

public: // overwritten from WorkbenchPart

	virtual void activate();
	virtual void deactivate();

public:

	/**
	 * Can be implemented by view implementations to provide a preferred default
	 * dock area that is assigned by default when the view is created. The
	 * user can move the view freely to other dock areas afterwards.
	 * The default implementation will return Qt::LeftDockWidgetArea.
	 */
	virtual Qt::DockWidgetArea getDefaultDockArea() const;

public:

	// overwrite QWidget::isVisible, to handle visibility of tabbified widgets correctly
	bool isVisible() const;

public:

	/// Returns the dock area, this view belongs to.
	Qt::DockWidgetArea getDockArea() const;

protected:

	/**
	 * Must be overloaded in subclasses to create the actual widget for
	 * this ViewPart.
	 * This ViewPart should be set as parent for the created widget.
	 */
	virtual QWidget* createPartControl() = 0;

	virtual void focusInEvent(QFocusEvent*);
	virtual void closeEvent(QCloseEvent*);

	/**
	 * Returns the optional view menu that is shown when the user clicks the
	 * view menu tool button.
	 * A ViewPart can add items to this menu within it's createPartControl()
	 * method, by calling this method and populating items to the menu.
	 */
	QMenu* getViewMenu();

	/// Returns the title bar of this view part.
	ViewPartTitleBar* getTitleBar();

	void paintEvent(QPaintEvent *event);

signals:

	void viewClosed(ViewPart* view);
	void viewDestroyed(ViewPart* view);
	void viewActivated(ViewPart* view);

	void minimizedRequested(ViewPart* view);

	void currentDockLocationChanged(Qt::DockWidgetArea area);

protected slots:

	virtual void onVisibilityChanged(bool visible);
	void onMinimizeRequested();
	void onDockLocationChanged(Qt::DockWidgetArea area);
	void onTopLevelChanged(bool topLevel);

protected:

	ViewPartTitleBar* mTitleBar;

private:
	IWorkbenchPartSite* mSite;
	bool mIsVisible;
	Qt::DockWidgetArea mArea;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
