/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PTPLocalTest.C
 *    Test PTP with connection to local framework
 *
 * @author Christof Schröter
 * @date   2021/12/03
 */

#include <boost/test/unit_test.hpp>

#include <serialization/Print.h>

#include <fw/Framework.h>
#include <fw/RemoteAuthority.h>

using namespace mira;

BOOST_AUTO_TEST_CASE(PTPLocalTest)
{
	const char* argv[] = {"PTPSyncTest",
	                      "${find PTPSyncTest.xml}",
	                      "--fw-name", "main"};
	Framework fw(4,(char**)argv);
	fw.load();
	fw.start();

	MIRA_SLEEP(3000) // establish connections

	std::map<std::string, std::pair<Time, Duration>> targetOffsets;

	PropertyNode* node;
	std::string name;
	Duration d;
	Time t;

	node = MIRA_FW.getProperties()->findChildNode("RemoteModule.FrameworkConnections.item[0].TimeSynch.TargetOffset");
	d = node->toTyped<Duration>()->get();
	node = MIRA_FW.getProperties()->findChildNode("RemoteModule.FrameworkConnections.item[0].TimeSynch.StartTime");
	t = node->toTyped<Time>()->get();
	node = MIRA_FW.getProperties()->findChildNode("RemoteModule.FrameworkConnections.item[0].RemoteFrameworkName");
	name = node->toTyped<std::string>()->get();
	targetOffsets[name] = std::make_pair(t, d);

	node = MIRA_FW.getProperties()->findChildNode("RemoteModule.FrameworkConnections.item[1].TimeSynch.TargetOffset");
	d = node->toTyped<Duration>()->get();
	node = MIRA_FW.getProperties()->findChildNode("RemoteModule.FrameworkConnections.item[1].TimeSynch.StartTime");
	t = node->toTyped<Time>()->get();
	node = MIRA_FW.getProperties()->findChildNode("RemoteModule.FrameworkConnections.item[1].RemoteFrameworkName");
	name = node->toTyped<std::string>()->get();
	targetOffsets[name] = std::make_pair(t, d);

	// our estimated offset to the remote fw named 'local' is 0 and has start time = epoch (no sync for a local connection)
	BOOST_CHECK_EQUAL(targetOffsets.size(), 2);
	BOOST_CHECK(targetOffsets.at("Local").first == Time::unixEpoch());
	BOOST_CHECK(targetOffsets.at("Local").second.isValid());
	BOOST_CHECK_EQUAL(targetOffsets.at("Local").second.totalMicroseconds(), 0);
	// our estimated offset to the remote fw named 'forceptp' has a current start time (sync is forced)
	BOOST_CHECK(Time::now() > targetOffsets.at("ForcePTP").first);
	BOOST_CHECK(Time::now() - targetOffsets.at("ForcePTP").first < Duration::seconds(5));
	BOOST_CHECK(targetOffsets.at("ForcePTP").second.isValid());

	// the remote frameworks both are the incoming side, so they just set their offset to 0
	json::Value v;
	v = MIRA_FW.getRPCManager().call<json::Value>("/Local#builtin",
	                                              "getPropertyJSON",
	                                              "RemoteModule.FrameworkConnections.item[0].TimeSynch.TargetOffset").get();
	BOOST_CHECK_EQUAL(json::write(v), std::string("0"));
	v = MIRA_FW.getRPCManager().call<json::Value>("/Local#builtin",
	                                              "getPropertyJSON",
	                                              "RemoteModule.FrameworkConnections.item[0].TimeSynch.StartTime").get();
	BOOST_CHECK_EQUAL(json::write(v), std::string("0"));

	v = MIRA_FW.getRPCManager().call<json::Value>("/ForcePTP#builtin",
	                                              "getPropertyJSON",
	                                              "RemoteModule.FrameworkConnections.item[0].TimeSynch.TargetOffset").get();
	BOOST_CHECK_EQUAL(json::write(v), std::string("0"));
	v = MIRA_FW.getRPCManager().call<json::Value>("/ForcePTP#builtin",
	                                              "getPropertyJSON",
	                                              "RemoteModule.FrameworkConnections.item[0].TimeSynch.StartTime").get();
	BOOST_CHECK_EQUAL(json::write(v), std::string("0"));

	MIRA_SLEEP(10000)

	// for both connections, check that at least 100 call/echo pairs have been transmitted (10/s),
	// and that the timestamp received always was the same as the timestamp sent

	v = MIRA_FW.getChannelManager().readChannel("/local/result/Count");
	BOOST_CHECK_GT(v.get_int(), 100);
	v = MIRA_FW.getChannelManager().readChannel("/local/result/MaxTSDiff");
	BOOST_CHECK_EQUAL(v.get_int(), 0);

	v = MIRA_FW.getChannelManager().readChannel("/forceptp/result/Count");
	BOOST_CHECK_GT(v.get_int(), 100);
	v = MIRA_FW.getChannelManager().readChannel("/forceptp/result/MaxTSDiff");
	BOOST_CHECK_EQUAL(v.get_int(), 0);
}
