/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FrameworkBasicTest.C
 *    Description.
 *
 * @author Erik Einhorn
 * @date   2010/12/04
 */

/**
 * Tests:
 *   - checkin and validation of authorities
 *   - channel name alias
 *   - subscription and publishing of channels
 *   - channel access restrictions
 *   - subscriber handlers, non-exclusive and exclusive ones
 *   - rpc calls
 *   - post, get and get with interpolation
 *   - channel promotion
 */

#include <platform/Platform.h>

#define ALLOWED_DELAY 10 // in ms
#define ALLOWED_RPC_DELAY 30 // in ms

#include <boost/test/unit_test.hpp>


#include <image/Img.h>
#include <math/LinearInterpolator.h>

#include <serialization/Serialization.h>

#include <fw/Framework.h>

using namespace mira;

const char* argv[] = {"FrameworkBasicTest", "-d0", "--no-colors"};
Framework fw(3,(char**)argv);

BOOST_AUTO_TEST_CASE( CheckinAndChannelAccessTests)
{
	std::cout << "CTEST_FULL_OUTPUT (Avoid ctest truncation of output)" << std::endl;

	Authority authority1("/", std::string("Authority1"));

	// must throw exceptions since authority with that name exists already
	BOOST_CHECK_THROW( Authority authority2("/", std::string("Authority1")), XInvalidParameter);

	Authority authority3("/otherNamespace", std::string("Authority1"));

	MIRA_FW.getNameRegistry().addAlias("Channel1", "/Channel1", "/otherNamespace");

	Channel<int> ch1 = authority1.publish<int>("Channel1");
	Channel<int> ch3 = authority3.subscribe<int>("Channel1");

	ChannelWrite<int> chwrite;
	ChannelRead<int> chread;

	// cannot read, since no data available
	BOOST_CHECK_THROW(chread = ch3.read(), XInvalidRead);

	chwrite = ch1.write();
	chwrite->value() = 123;
	chwrite->timestamp = Time::now();
	chwrite.finish();

	// cannot access any longer, since finish() was called
	BOOST_CHECK_THROW(chwrite->value()=333, XAccessViolation);

	chread = ch3.read();
	BOOST_CHECK_EQUAL(chread->value(), 123);

	// now perform a couple of access right checks

	// authority3 is not allowed to write:
	BOOST_CHECK_THROW(chwrite = ch3.write(), XAccessViolation);

	// authority1 is not allowed to read
	BOOST_CHECK_THROW(chread = ch1.read(), XAccessViolation);

	// make sure, that read object was not disturbed by the above assignment (which caused an exception)
	BOOST_CHECK_EQUAL(chread->value(), 123);

	chread.finish();

	// cannot access any longer, since finish() was called
	BOOST_CHECK_THROW(std::cout << chread->value(), XAccessViolation);

	// obtain read again and check again
	chread = ch3.read();
	BOOST_CHECK_EQUAL(chread->value(), 123);

	// test get() method
	BOOST_CHECK_EQUAL(ch3.get().value(), 123);

	// authority1 is not allowed to read
	BOOST_CHECK_THROW(std::cout << ch1.get().value(), XAccessViolation);

	// authority3 is not allowed to write
	BOOST_CHECK_THROW(ch3.post(555), XAccessViolation);

	ch1.post(999);

	chread = ch3.read();

	BOOST_CHECK_EQUAL(chread->value(), 999);
	BOOST_CHECK_EQUAL(ch3.get().value(), 999);

	auto ch4 = authority1.publishAndSubscribe<std::vector<int>>("Channel4");
	ch4.post({}, Time::unixEpoch());                      // post works
	BOOST_CHECK(ch4.get().value() == std::vector<int>()); // there is data on the channel, which is a default vector
	BOOST_CHECK(ch4.get().timestamp == Time::unixEpoch());

	Time ts = Time::now() + Duration(1, 0, 0);  // all timestamps are significantly different from now()
	Time ts1 =         ts + Duration(1, 0, 0);
	Time ts2 =        ts1 + Duration(1, 0, 0);

	ch4.post(makeStamped(std::vector<int>(), ts));
	BOOST_CHECK(ch4.get().timestamp == ts);

	auto stamped = makeStamped(std::vector<int>(), ts1);
	ch4.post(stamped);
	BOOST_CHECK(ch4.get().timestamp == ts1);

	const auto stamped2 = makeStamped(std::vector<int>(), ts2);
	ch4.post(stamped2);
	BOOST_CHECK(ch4.get().timestamp == ts2);
}

Stamped<Img8U1> copyImg(Stamped<Img8U1> s)
{
	s.value() = s.value().clone();
	return s;
}

BOOST_AUTO_TEST_CASE( ChannelWriteCopyTest)
{
	Authority authority("/", std::string("Authority"));

	// Storage duration ensures we do not reuse slots for writing
	Channel<long int> intChannel = authority.publishAndSubscribe<long int>("LongInt",
	                                                                       Duration::seconds(10));

	{
		intChannel.post(1234567890l);                       // create a slot, write value to it

		ChannelRead<long int> read = intChannel.read();     // read the last slot that was written
		BOOST_CHECK_EQUAL(read->value(), 1234567890l);

		ChannelWrite<long int> write = intChannel.write();  // create a new slot (the read slot is still locked)
		BOOST_CHECK_NE(read->value(), write->value());      // there is at least a 1:2^32 chance it is equal, though :(
		write.discard();

		write = intChannel.write(true);                     // copy last written value to the new write slot internally
		BOOST_CHECK_EQUAL(write->value(), 1234567890l);     // write is pre-initialized with latest channel value
	}

	Channel<Img8U1> imgChannel = authority.publishAndSubscribe<Img8U1>("Image",
	                                                                   Duration::seconds(10));

	{
		Img8U1 img(111, 55);
		img = 123;
		imgChannel.post(img.clone());

		ChannelRead<Img8U1> read = imgChannel.read();
		const Img8U1& readImg = read->value();
		BOOST_CHECK_EQUAL(readImg.width(), 111);
		BOOST_CHECK_EQUAL(readImg.height(), 55);
		BOOST_CHECK_EQUAL(readImg(0, 0), 123);               // check pixel value matching initial fill value


		ChannelWrite<Img8U1> write = imgChannel.write(true); // again, copy last written value to the new write slot internally
		BOOST_CHECK_EQUAL(write->value().width(), 111);
		BOOST_CHECK_EQUAL(write->value().height(), 55);

		BOOST_CHECK_EQUAL(write->value().data(), read->value().data()); 
		                                                     // uh, we made a shallow copy here!

		write->value() = 234;                                // manipulation of the img in the new write slot
		BOOST_CHECK_EQUAL(readImg(0, 0), 234);               // changes the img in the previous slot :(

		write.discard();

		// can we do better?
		write = imgChannel.write(true,
		                         [](const auto& s){return makeStamped(s.value().clone(), s.timestamp);});

		                         // this notation may be shorter if we need to copy frameID/sequenceID as well:
		                         // [](const auto& s){auto cp = s; cp.value() = s.value().clone(); return cp;});

		BOOST_CHECK_NE(write->value().data(), read->value().data());  // yes we can!

		write->value() = 135;                               // manipulation of the img in the new write slot
		BOOST_CHECK_EQUAL(readImg(0, 0), 234);              // does not affect the previous slot :)
		write.finish();

		write = imgChannel.write(true, &copyImg); // use function copyImg from above
		BOOST_CHECK_EQUAL(write->value().width(), 111);
		BOOST_CHECK_EQUAL(write->value().height(), 55);
		BOOST_CHECK_NE(write->value().data(), read->value().data());
	}
}

///////////////////////////////////////////////////////////////////////////////

int gBlockingCallback21=0;
void blockingCallback21(ChannelRead<int> value)
{
	MIRA_LOG(NOTICE) << "blockingCallback21";
	gBlockingCallback21 = value->value();
	BOOST_CHECK_LE((Time::now() - value->timestamp).totalMilliseconds(), ALLOWED_DELAY);
	MIRA_SLEEP(500)
}

int gExclusiveCallback22=0;
void exclusiveCallback22(ChannelRead<int> value)
{
	MIRA_LOG(NOTICE) << "exclusiveCallback22";
	gExclusiveCallback22 = value->value();
	BOOST_CHECK_LE((Time::now() - value->timestamp).totalMilliseconds(), ALLOWED_DELAY);
}

int gNonexclusiveCallback23=0;
void nonexclusiveCallback23(ChannelRead<int> value)
{
	MIRA_LOG(NOTICE) << "nonexclusiveCallback23";
	gNonexclusiveCallback23=value->value();
	BOOST_CHECK_LE((Time::now() - value->timestamp).totalMilliseconds(), 500+ALLOWED_DELAY);
}

int gNonblockingCallback31=0;
void nonblockingCallback31(ChannelRead<int> value)
{
	MIRA_LOG(NOTICE) << "nonblockingCallback31";
	gNonblockingCallback31=value->value();
	BOOST_CHECK_LE((Time::now() - value->timestamp).totalMilliseconds(), ALLOWED_DELAY);
}

int gExclusiveCallback32=0;
void exclusiveCallback32(ChannelRead<int> value)
{
	MIRA_LOG(NOTICE) << "exclusiveCallback32";
	gExclusiveCallback32=value->value();
	BOOST_CHECK_LE((Time::now() - value->timestamp).totalMilliseconds(), ALLOWED_DELAY);
}

int gNonexclusiveCallback33=0;
void nonexclusiveCallback33(ChannelRead<int> value)
{
	MIRA_LOG(NOTICE) << "nonexclusiveCallback33";
	gNonexclusiveCallback33 = value->value();
	BOOST_CHECK_LE((Time::now() - value->timestamp).totalMilliseconds(), ALLOWED_DELAY);
}

BOOST_AUTO_TEST_CASE(SubscribeCallbackTests)
{
	Authority authority1("/", std::string("Authority1"));
	Authority authority2("/", std::string("Authority2"));
	Authority authority3("/", std::string("Authority3"));

	Channel<int> ch11 = authority1.publish<int>("Channel1");
	Channel<int> ch12 = authority1.publish<int>("Channel2");
	Channel<int> ch13 = authority1.publish<int>("Channel3");

	Channel<int> ch21 = authority2.subscribe<int>("Channel1", blockingCallback21);
	Channel<int> ch22 = authority2.subscribe<int>("Channel2", exclusiveCallback22,true);
	Channel<int> ch23 = authority2.subscribe<int>("Channel3", nonexclusiveCallback23);

	Channel<int> ch31 = authority3.subscribe<int>("Channel1", nonblockingCallback31);
	Channel<int> ch32 = authority3.subscribe<int>("Channel2", exclusiveCallback32,true);
	Channel<int> ch33 = authority3.subscribe<int>("Channel3", nonexclusiveCallback33);

	// reset globals that are used to check if callbacks were called
	gBlockingCallback21=0;
	gExclusiveCallback22=0;
	gNonexclusiveCallback23=0;
	gNonblockingCallback31=0;
	gExclusiveCallback32=0;
	gNonexclusiveCallback33=0;

	MIRA_LOG(NOTICE) << "writing1";
	ch11.post(1111);
	MIRA_LOG(NOTICE) << "writing2";
	ch12.post(2222);
	MIRA_LOG(NOTICE) << "writing3";
	ch13.post(3333);

	MIRA_SLEEP(600)

	// check if callbacks were called
	BOOST_CHECK_EQUAL(gBlockingCallback21, 1111);
	BOOST_CHECK_EQUAL(gExclusiveCallback22, 2222);
	BOOST_CHECK_EQUAL(gNonexclusiveCallback23, 3333);
	BOOST_CHECK_EQUAL(gNonblockingCallback31, 1111);
	BOOST_CHECK_EQUAL(gExclusiveCallback32, 2222);
	BOOST_CHECK_EQUAL(gNonexclusiveCallback33, 3333);
}

///////////////////////////////////////////////////////////////////////////////

int gAutoTimestampTestCallback=0;
void autoTimestampTestCallback(ChannelRead<int> value)
{
	MIRA_LOG(NOTICE) << "callbackTest";
	++gAutoTimestampTestCallback;
}

BOOST_AUTO_TEST_CASE(AutoTimestampTest)
{
	Authority authority1("/", std::string("Authority1"));
	Authority authority2("/", std::string("Authority2"));

	Channel<int> ch1 = authority1.publish<int>("AutoTimestampTest");
	Channel<int> ch2 = authority2.subscribe<int>("AutoTimestampTest", autoTimestampTestCallback);

	gAutoTimestampTestCallback = 0;
	for(int i=0; i<10; ++i)
	{
		{
		auto w1 = ch1.write();
		}
		MIRA_SLEEP(10)
	}
	BOOST_CHECK_EQUAL(gAutoTimestampTestCallback, 10);
}

///////////////////////////////////////////////////////////////////////////////

class XTestException
{

};

BOOST_AUTO_TEST_CASE(InvalidTimestampTest)
{
	Authority authority1("/", std::string("Authority1"));

	authority1.publish<int>("InvalidTimestampTest");
	Channel<int> ch1 = authority1.subscribe<int>("InvalidTimestampTest");

	ChannelWrite<int> w = ch1.write();
	BOOST_CHECK_THROW(ch1.post(123, Time()), XLogical); // posting with invalid time should throw exception

	w->value() = 123;
	w->timestamp = Time(); // set invalid time

	BOOST_CHECK_THROW(w.finish(), XLogical); // writing the data should throw exception

	w = ch1.write();
	w->value() = 456;
	w->timestamp = Time::now();
	w.finish();

	ChannelRead<int> r = ch1.read();
	BOOST_CHECK_EQUAL(r->value(),456);


	w = ch1.write();
	w->value() = 789;
	w->timestamp = Time::now();
	w.finish();

	r = ch1.read();
	BOOST_CHECK_EQUAL(r->value(),789);


	// Test for ticket #492
	try {
		ChannelWrite<int> w = ch1.write();
		w->value() = 123;
		w->timestamp = Time(); // set invalid time

		// cause stack unrolling which will destruct the ChannelWrite. The
		// ChannelWrite destructor should discard the data then, instead of
		// writing it.
		throw XTestException();

	} catch(XLogical& ex) {
		std::cout << "FAILED: Exception should be XTestException, but is " << ex.what() << std::endl;
		BOOST_CHECK(false);
	} catch(XTestException&) {
		std::cout << "OK "<< std::endl;
		BOOST_CHECK(true);
	}


}

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE(ReadToleranceTest)
{
	Authority authority("/", std::string("Authority1"));

	authority.publish<int>("ReadToleranceTest");
	Channel<int> ch = authority.subscribe<int>("ReadToleranceTest");

	Time t0 = Time::now();
	auto w = ch.write();
	w->value() = 99;
	w->timestamp = t0;
	w.finish();
	auto r1 = ch.read(t0+Duration::seconds(2));
	BOOST_CHECK(r1->value() == 99);
	auto r2 = ch.read(t0+Duration::seconds(2), NEAREST_SLOT, Duration::seconds(3));
	BOOST_CHECK(r2->value() == 99);
	BOOST_CHECK_THROW(auto r3 = ch.read(t0+Duration::seconds(2), NEAREST_SLOT, Duration::seconds(1)), XInvalidRead);
}

///////////////////////////////////////////////////////////////////////////////

class MyService1
{
public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.method("method", &MyService1::method, this,
		         "A test method", "time", "");
	}

	int test;
	void method(const Time& time)
	{
		MIRA_LOG(NOTICE) << "MyService1::method() called " << time;
		BOOST_CHECK_LE((Time::now() - time).totalMilliseconds(), ALLOWED_RPC_DELAY);
		test = 8888;
	}
};

class MyService2
{
public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.method("method", &MyService2::method, this,
		         "A test method", "time", "", "param", "");
	}

	int test;
	void method(const Time& time, int param)
	{
		MIRA_LOG(NOTICE) << "MyService2::method() called " << time << ", " << param;
		BOOST_CHECK_LE((Time::now() - time).totalMilliseconds(), ALLOWED_RPC_DELAY);
		test = param;
	}
};

BOOST_AUTO_TEST_CASE(RPCBenchmark)
{
	Authority authority1("/", std::string("Authority1"));
	Authority authority2("/", std::string("Authority2"));

	MyService1 service1;
	authority1.publishService(service1);

	MyService2 service2;
	authority2.publishService(service2);

	service1.test = 0;
	RPCFuture<void> f = authority1.callService<void>("/Authority1", "method", Time::now());
	BOOST_CHECK(f.timedWait(Duration::seconds(2)));
	BOOST_CHECK_EQUAL(service1.test, 8888);

	service1.test = 0;
	f = authority2.callService<void>("/Authority1", "method", Time::now());
	BOOST_CHECK(f.timedWait(Duration::seconds(2)));
	BOOST_CHECK_EQUAL(service1.test, 8888);


	f = authority1.callService<void>("/Authority2", "method", Time::now());
	if(f.timedWait(Duration::seconds(2)))
		BOOST_CHECK_THROW(f.get(), XRPC);

	service2.test = 0;
	f = authority1.callService<void>("/Authority2", "method", Time::now(), 123);
	BOOST_CHECK(f.timedWait(Duration::seconds(2)));
	BOOST_CHECK_EQUAL(service2.test, 123);

	service2.test = 0;
	authority2.callService<void>("/Authority2", "method", Time::now(), 345).get();

	MIRA_SLEEP(200)

	BOOST_CHECK_EQUAL(service2.test, 345);
}

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE(IndependentThreadTests)
{
	Authority authority1("/", std::string("Authority1"));
	Authority authority2("/", std::string("Authority2"), Authority::INDEPENDENT_SUBSCRIBER_THREAD);

	Channel<int> ch11 = authority1.publish<int>("Channel1");

	MyService2 service;
	authority2.publishService(service);
	Channel<int> ch21 = authority2.subscribe<int>("Channel1", blockingCallback21);

	// reset globals that are used to check if callbacks were called
	gBlockingCallback21=0;

	ch11.post(111);
	auto f1 = authority1.callService<void>("/Authority2", "method", Time::now(), 123);
	auto f2 = authority2.callService<void>("/Authority2", "method", Time::now(), 345);

	MIRA_SLEEP(200)

	// check if callbacks were called
	BOOST_CHECK_EQUAL(gBlockingCallback21, 111);

	f1.get(); // make sure 'method' was executed before ending the test
	f2.get();
}

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE(InterpolationTests)
{
	Authority authority("/", std::string("Authority1"));

	authority.publish<float>("Channel");
	Channel<float> channel = authority.subscribe<float>("Channel");

	Time t0 = Time::now();
	Time t1 = Time::now()+Duration::seconds(1);

	Time tin = Time::now()+Duration::milliseconds(500);

	channel.post(2.0f, t0);
	channel.post(4.0f, t1);

	//std::cout << channel.get().value() << std::endl;
	//std::cout << channel.get(tin).value() << std::endl;
	//std::cout << channel.get(tin, LinearInterpolator()).value() << std::endl;

	BOOST_CHECK_CLOSE(channel.get(tin, LinearInterpolator()).value(), 3.0f, 0.01);

	MIRA_SLEEP(200)
}

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE(ChannelPromotionTests)
{
	Authority authority("/", std::string("Authority1"));

	Time t0 = Time::now();
	Time t1 = Time::now()+Duration::seconds(1);
	Time t2 = Time::now()+Duration::seconds(2);

	Channel<void> untypedChannel = authority.publish<void>("UntypedChannel", typeName<float>());

	// cannot publish untyped with type not matching the previous untyped publish
	Authority authority2("/", std::string("Authority2"));
	BOOST_CHECK_THROW(authority2.publish<void>("UntypedChannel", typeName<int>()), XBadCast);

	// can publish untyped with type matching the previous untyped publish
	authority2.publish<void>("UntypedChannel", typeName<float>());

	{
		Buffer<uint8> buffer;
		BinaryBufferSerializer bs(&buffer);
		ChannelWrite<void> w = untypedChannel.write();
		float value = 111.0f;
		w->timestamp = t0;
		bs.serialize(value,false);
		w.writeSerializedValue(buffer);
	}

	{
		Buffer<uint8> buffer;
		BinaryBufferSerializer bs(&buffer);
		ChannelWrite<void> w = untypedChannel.write();
		float value = 222.0f;
		w->timestamp = t1;
		bs.serialize(value,false);
		w.writeSerializedValue(buffer);
		w.finish();
	}

	// cannot promote to a type not matching the previous untyped publisher
	BOOST_CHECK_THROW(authority.subscribe<int>("UntypedChannel"), XBadCast);

	// this will induce a promotion of the channel, since we now explicitly
	// specify the type
	Channel<float> typedChannel = authority.subscribe<float>("UntypedChannel");

	BOOST_CHECK_CLOSE(typedChannel.get(t0).value(), 111.0f, 0.001);
	BOOST_CHECK_CLOSE(typedChannel.get(t1).value(), 222.0f, 0.001);

	{
		Buffer<uint8> buffer;
		BinaryBufferSerializer bs(&buffer);
		ChannelWrite<void> w = untypedChannel.write();
		float value = 333.0f;
		w->timestamp = t2;
		bs.serialize(value,false);
		w.writeSerializedValue(buffer);
		w.finish();
	}

	BOOST_CHECK_CLOSE(typedChannel.get(t2).value(), 333.0f, 0.001);

	// cannot publish untyped with a type not matching the typed channel
	Authority authority3("/", std::string("Authority3"));
	BOOST_CHECK_THROW(authority3.publish<void>("UntypedChannel", typeName<int>()), XBadCast);

	// can publish untyped with type matching the typed channel
	authority3.publish<void>("UntypedChannel", typeName<float>());
}

//////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE(IntervalTests)
{
	Authority authority("/", std::string("Authority1"));

	authority.publish<int>("Interval");
	Channel<int> channel = authority.subscribe<int>("Interval", Duration::seconds(20));

	Time t0 = Time(Date(2000, 1, 1), Duration::seconds(0));

	for (int i=0; i<10; ++i)
		channel.post(i, t0+Duration::seconds(i));

	// Requesting 4 values 2 older, 2 newer.
	ChannelReadInterval<int> iv = channel.readInterval(t0+Duration::seconds(5), 4, 2, 2);
	// Interval should contain 3,4,5,6
	int should = 3;
	foreach(int val, iv)
	{
		BOOST_CHECK_EQUAL(val, should++);
	}

	// Requesting 4 values 2 older, 2 newer.
	iv = channel.readInterval(t0+Duration::milliseconds(5001), 4, 2, 2);
	// Interval should contain 4,5,6,7
	should = 4;
	foreach(int val, iv)
	{
		BOOST_CHECK_EQUAL(val, should++);
	}

	// Requesting 4 values 1 older, 1 newer, prefer fill up with newer
	// older = 4, newer = 5
	iv = channel.readInterval(t0+Duration::seconds(5), 4, 1, 1);
	// Interval should contain 4,5,6,7
	should = 4;
	foreach(int val, iv)
	{
		BOOST_CHECK_EQUAL(val, should++);
	}

	// Requesting 4 values 1 older, 1 newer, prefer fill up with older
	// older = 4, newer = 5
	iv = channel.readInterval(t0+Duration::seconds(5), 4, 1, 1, PREFER_OLDER);
	// Interval should contain 2,3,4,5
	should = 2;
	foreach(int val, iv)
	{
		BOOST_CHECK_EQUAL(val, should++);
	}

	MIRA_SLEEP(200)
}

///////////////////////////////////////////////////////////////////////////////

class PolymorphicA : public mira::Object
{
	MIRA_OBJECT(PolymorphicA)
public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
	}

};
MIRA_CLASS_SERIALIZATION(PolymorphicA, mira::Object)

class PolymorphicB : public PolymorphicA
{
	MIRA_OBJECT(PolymorphicB)
};
MIRA_CLASS_SERIALIZATION(PolymorphicB, PolymorphicA)

class PolymorphicC : public PolymorphicB
{
	MIRA_OBJECT(PolymorphicC)
};
MIRA_CLASS_SERIALIZATION(PolymorphicC, PolymorphicB);

class PolymorphicD : public PolymorphicA
{
	MIRA_OBJECT(PolymorphicD)
};
MIRA_CLASS_SERIALIZATION(PolymorphicD, PolymorphicA);


BOOST_AUTO_TEST_CASE(PolymorphicChannelTest)
{
	Authority authority("/", std::string("Authority1"));

	authority.subscribe<PolymorphicA*>("PolymorphicChannel1");
	authority.publish<PolymorphicC*>("PolymorphicChannel1");

	authority.subscribe<PolymorphicB*>("PolymorphicChannel1");

	BOOST_CHECK_THROW(authority.subscribe<PolymorphicD*>("PolymorphicChannel1"), XBadCast);

	// BUG see ticket #390
	//authority.publish<PolymorphicC*>("PolymorphicChannel1");
}

BOOST_AUTO_TEST_CASE(PolymorphicChannelExchangeTest)
{
	Authority authority("/", std::string("Authority1"));

	Channel<PolymorphicA*> channel1 = authority.subscribe<PolymorphicA*>("PolymorphicChannel");
	Channel<PolymorphicC*> channel2 = authority.publish<PolymorphicC*>("PolymorphicChannel");

	channel2.post(new PolymorphicB, Time::now());

	ChannelRead<PolymorphicA*> r = channel1.read();
	//std::cout << r->value() << std::endl;
	//std::cout << channel1.get().value() << std::endl;
}


// tests if StampedHeader timestamp are accessed correctly when reading
// from untyped channels (pointers must be equal in this test)
BOOST_AUTO_TEST_CASE(DerivedFromObjectUntypedReadTest)
{
	Authority authority("/", std::string("Authority1"));

	Time t0 = Time::now();


	Time* timestampPtr = NULL;
	{
		Channel<PolymorphicA> ch = authority.publish<PolymorphicA>("UntypedObjChannel");
		ChannelWrite<PolymorphicA> w = ch.write();
		w->timestamp = t0;
		timestampPtr = &w->timestamp;
	}

	{
		Channel<void> ch = authority.subscribe<void>("UntypedObjChannel");
		ChannelRead<void> r = ch.read();

		BOOST_CHECK_EQUAL(timestampPtr, &r.getTimestamp());
		BOOST_CHECK_EQUAL(timestampPtr, &r->timestamp);
	}

}


void intCallback(ChannelRead<int> read)
{
}

BOOST_AUTO_TEST_CASE(HasSubscriberTest)
{
	Authority authority("/", std::string("Authority"));

	Channel<int> c1 = authority.subscribe<int>("Int1");
	BOOST_CHECK_EQUAL(c1.hasSubscriber(), true);
	Channel<int> c2 = authority.subscribe<int>("Int2", &intCallback);
	BOOST_CHECK_EQUAL(c2.hasSubscriber(), true);
}

int gHandlerOrder;

void initHandler()
{
	BOOST_CHECK_EQUAL(gHandlerOrder, 0);
	++gHandlerOrder;
}

void startupHandler()
{
	BOOST_CHECK_EQUAL(gHandlerOrder, 1);
	++gHandlerOrder;
}

void stopHandler()
{
	BOOST_CHECK_EQUAL(gHandlerOrder, 2);
	++gHandlerOrder;
}

void finalizeHandler()
{
	BOOST_CHECK_EQUAL(gHandlerOrder, 3);
	++gHandlerOrder;
}

BOOST_AUTO_TEST_CASE(HandlerOrderTest)
{
	{
		Authority a;
		a.checkin("/", "Authority");
		gHandlerOrder = 0;
		a.addImmediateHandlerFunction(&initHandler);
		a.addImmediateHandlerFunction(&startupHandler);
		a.addFinalizeHandlerFunction(&stopHandler);
		a.addFinalizeHandlerFunction(&finalizeHandler);
		a.start();
		MIRA_SLEEP(1000)
	}
	BOOST_CHECK_EQUAL(gHandlerOrder, 4);
}

///////////////////////////////////////////////////////////////////////////////

struct MoveSemantics
{
	MoveSemantics() { std::cout << "default ctor" << std::endl; }
	MoveSemantics(const MoveSemantics& other) { std::cout << "copy ctor" << std::endl; ++copied; }
	MoveSemantics(MoveSemantics&& other) noexcept { std::cout << "move ctor" << std::endl; }

	MoveSemantics& operator=(const MoveSemantics&) { std::cout << "copy assign" << std::endl; return *this; ++copied; }
	MoveSemantics& operator=(MoveSemantics&&) noexcept { std::cout << "move assign" << std::endl; return *this; }

	template <typename Reflector>
	void reflect(Reflector& r)
	{}

	static int copied;
};

int MoveSemantics::copied = 0;

BOOST_AUTO_TEST_CASE(MoveSemanticsTest)
{
	Authority authority("/", std::string("Authority"));
	auto channel = authority.publish<MoveSemantics>("MoveChannel");

	channel.post(MoveSemantics());
	BOOST_CHECK_EQUAL(MoveSemantics::copied, 0);

	std::cout << "-----------------" << std::endl;

	MoveSemantics m;
	channel.post(m);
	BOOST_CHECK_EQUAL(MoveSemantics::copied, 1);

	std::cout << "-----------------" << std::endl;

	channel.post(std::move(m));
	BOOST_CHECK_EQUAL(MoveSemantics::copied, 1);

}

///////////////////////////////////////////////////////////////////////////////

struct StructNonCopyable: boost::noncopyable
{
	template <typename Reflector>
	void reflect(Reflector& r)
	{
	}
};

BOOST_AUTO_TEST_CASE(NonCopyableTest)
{
	Authority authority("/", std::string("Authority"));
	auto channel = authority.publish<StructNonCopyable>("NonCopyable");
}

///////////////////////////////////////////////////////////////////////////////

void pose2Callback(ChannelRead<Pose2> p)
{
	// this ChannelRead has already created a copy of
	// the channel slot (Pose2 is CheapToCopy),
	// including copying its serializedValue
	std::cout << print(p->value(), COMPACT_PRINT) << std::endl;
}

void voidCallback(ChannelRead<void> v)
{
	std::cout << print(v.readSerializedValue(), COMPACT_PRINT) << std::endl;
}

// when running this with AddressSanitizer
// (LD_LD_PRELOAD=/usr/lib/gcc/x86_64-linux-gnu/13/libasan.so ASAN_OPTIONS=detect_leaks=0)
// we spot buffer overflows if Slot copying does not properly protect the serializedValue member
BOOST_AUTO_TEST_CASE(ReadSerializedValueTest)
{
	Authority authority("/", std::string("Authority"));
	auto channel = authority.publish<Pose2>("Pose2");

	// subscribe with the 2 callbacks above,
	// in independent threads (important, so they can work concurrently)
	authority.subscribe<Pose2>("Pose2", &pose2Callback, true);
	authority.subscribe<void>("Pose2", &voidCallback, true);

	Time start = Time::now();
	while (Time::now() - start < Duration::seconds(3)) {
		channel.post(Pose2());
		MIRA_SLEEP(10)
	}
}

