/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ChannelWriteXMLTest.C
 *    Test for wrteXML function for Channel<void>
 *
 * @author Tom Mehner
 */

#include <boost/test/unit_test.hpp>

#include <fw/Framework.h>
#include <serialization/XMLSerializer.h>

using namespace mira;

namespace mira::test {

struct ComplexType
{
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("String", String, "");
		r.member("Int", Int, "");
		r.member("Float", Float, "");
	}

	std::string String = "default";
	int Int = 123;
	float Float = 0.125f;
};

} // namespace mira::test

BOOST_AUTO_TEST_CASE(ChannelWriteXMLTest)
{
	const std::string xmlStr = R"(
	<Value>
		<String>Name</String>
		<Int>5</Int>
		<Float>0.5</Float>
	</Value>
	)";
	auto xml = XMLDom{};
	xml.loadFromString(xmlStr);

	const char* argv[] = {"ChannelWriteXMLTest", "-d0", "--no-colors"};
	Framework fw(3, (char**)argv);
	// now start processing
	fw.start();

	// create a void publisher for channel 'Something'
	Authority publisher("/", "VoidPublisher");
	auto chWrite = publisher.publish<void>("Something", std::string("mira::test::ComplexType"));

	// there is no typed publisher or subscriber, channel is untyped, writeXML fails
	BOOST_CHECK_THROW(chWrite.write().writeXML(xml.root()), XNotImplemented);

	// add a typed subscriber, channel is typed
	Authority subscriber("/", "TypedSubscriber");
	auto ch = subscriber.subscribe<mira::test::ComplexType>("Something");

	// the channel has no data yet
	BOOST_CHECK_THROW(ch.get(), XInvalidRead);

	// now writeXML works
	chWrite.write().writeXML(xml.root());

	mira::test::ComplexType val = ch.get();
	BOOST_CHECK_EQUAL(val.String, "Name");
	BOOST_CHECK_EQUAL(val.Int, 5);
	BOOST_CHECK_EQUAL(val.Float, 0.5);
}
