/*
 * Copyright (C) 2018 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ChannelPromoteByTypenameTest_NoAutoPromote.C
 *    Test for channel promotion by runtime typename (when disabled).
 *
 * @author Christof Schröter
 * @date   2018/08/29
 */

#include <boost/test/unit_test.hpp>

#include <fw/Framework.h>

using namespace mira;

BOOST_AUTO_TEST_CASE(ChannelPromoteByTypenameTest_NoAutoPromote)
{
	// create the framework with publisher from config file
	const char* argv[] = {"ChannelPromoteByTypenameTest_NoAutoPromote",
	                      "${find ChannelPromoteByTypenameTest.xml}",
	                      "-v", "binary=mira", "--no-autopromote"};
	Framework fw(5,(char**)argv);
	fw.load();

	// now start processing
	fw.start();

	// give it a few seconds to start and connect all channels
	MIRA_SLEEP(5000);

	ChannelManager& manager = fw.getChannelManager();

	BOOST_CHECK_EQUAL(manager.getChannels().size(), 53);

	// all have no local publishers, but local subscribers
	BOOST_CHECK_EQUAL(manager.getChannelList(true, false).size(), 26);
	BOOST_CHECK_EQUAL(manager.getChannelList(false, true).size(), 27);

	// remotely published channels have an 'internal' publisher locally
	std::map<std::string, Typename> channels = manager.getPublishedChannels(true);
	BOOST_CHECK_EQUAL(channels.size(), 52);

	BOOST_CHECK_EQUAL(channels["/1/DummyVector"],
	                           "std::vector<boost::shared_ptr<mira::test::channel_promote::ns1::DummyBase>,"
	                           "std::allocator<boost::shared_ptr<mira::test::channel_promote::ns1::DummyBase>>>");
	BOOST_CHECK_EQUAL(channels["/1/DummyMap"],
	                           "std::map<int,boost::shared_ptr<mira::test::channel_promote::ns1::DummyBase>,"
	                           "std::less<int>,std::allocator<std::pair<int const,"
	                           "boost::shared_ptr<mira::test::channel_promote::ns1::DummyBase>>>>");
	BOOST_CHECK_EQUAL(channels["/1/Int"], "int");
	BOOST_CHECK_EQUAL(channels["/1/Float"], "float");
	BOOST_CHECK_EQUAL(channels["/1/Char"], "char");
	BOOST_CHECK_EQUAL(channels["/1/Double"], "double");

	BOOST_CHECK_EQUAL(channels["/1/String"], "mira::test::channel_promote::String");

	BOOST_CHECK_EQUAL(channels["/2/DummyVector"],
	                           "std::vector<boost::shared_ptr<mira::test::channel_promote::ns2::DummyBase>,"
	                           "std::allocator<boost::shared_ptr<mira::test::channel_promote::ns2::DummyBase>>>");
	BOOST_CHECK_EQUAL(channels["/2/DummyMap"],
	                           "std::map<int,boost::shared_ptr<mira::test::channel_promote::ns2::DummyBase>,"
	                           "std::less<int>,std::allocator<std::pair<int const,"
	                           "boost::shared_ptr<mira::test::channel_promote::ns2::DummyBase>>>>");
	BOOST_CHECK_EQUAL(channels["/2/Int"], "int");
	BOOST_CHECK_EQUAL(channels["/2/Float"], "float");
	BOOST_CHECK_EQUAL(channels["/2/Char"], "char");
	BOOST_CHECK_EQUAL(channels["/2/Bool"], "bool");

	BOOST_CHECK_EQUAL(channels["/2/String"], "mira::test::channel_promote::String");

	BOOST_CHECK_EQUAL(channels["/3/Time"], "mira::Time");
	BOOST_CHECK_EQUAL(channels["/3/Int"], "int");
	BOOST_CHECK_EQUAL(channels["/3/Float"], "float");
	BOOST_CHECK_EQUAL(channels["/3/Char"], "char");
	BOOST_CHECK_EQUAL(channels["/3/Double"], "double");

	BOOST_CHECK_EQUAL(channels["/3/String"], "mira::test::channel_promote::String");

	BOOST_CHECK_EQUAL(channels["/4/Time"], "mira::Time");
	BOOST_CHECK_EQUAL(channels["/4/Int"], "int");
	BOOST_CHECK_EQUAL(channels["/4/Float"], "float");
	BOOST_CHECK_EQUAL(channels["/4/Char"], "char");
	BOOST_CHECK_EQUAL(channels["/4/Double"], "double");

	BOOST_CHECK_EQUAL(channels["/4/String"], "mira::test::channel_promote::String");

	// all channels are untyped as long as nothing is posted
	foreach (auto p, channels)
		BOOST_CHECK_EQUAL(manager.getTypeId(p.first), -1);

	manager.publish<bool>("/Signal", "", false, "bool")->write().finish();  // just an arbitrary write access to signal publishers

	// all auto-promote is disabled, so channels remain untyped even after posting
	MIRA_SLEEP(2000);	

	foreach (auto p, channels)
		BOOST_CHECK_EQUAL(manager.getTypeId(p.first), -1);


	ClassFactory& factory = ClassFactory::instance();
	auto registeredTypes = TypedChannelPromoterBase::CLASS().getDerivedClasses();
	int count = 0;
	foreach (auto c, registeredTypes) {
		if (strstr(c.first.c_str(), "framework_tests_fw") &&
		    strstr(c.first.c_str(), "_ChannelPromoteByTypenameTestUnits"))
			++count;
	}
	// called MIRA_REGISTER_CHANNELTYPE 14 times --> created 14 promoter classes for 8 distinct channel types
	BOOST_CHECK_EQUAL(count, 14);
}
