/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeRecorderUnit.C
 *    Units that act as interfaces to a TapeRecorder
 *
 * @author Tim Langner, Christof Schröter
 * @date   2013/29/01
 */

#include <serialization/SetterNotify.h>
#include <fw/MicroUnit.h>
#include <fw/TapeRecorder.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

// just a sample, as BinarySerializerCodec is abstract and real codecs are in
// separate toolbox CommonCodecs
struct SampleCodec : public BinarySerializerCodec
{
MIRA_META_OBJECT(SampleCodec,
		("FOURCC",      "SMPL")
		("Name",        "Sample")
		("Description", "Sample codec just for demonstration")
		("Category"   , "Dummy"))

public:
	SampleCodec() : quality(95) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Quality", quality, "");
	}

	virtual TypeId getSupportedTypeId() const { return TypeId();}
	virtual Buffer<uint8> encodeBuffer(TypedVoidConstPtr objectPtr) { return Buffer<uint8>(); }
	virtual void decodeBuffer(const Buffer<uint8>& data, TypedVoidPtr ioObjectPtr) {}

	int quality;
};

///////////////////////////////////////////////////////////////////////////////

class TapeRecorderUnitBase : public MicroUnit
{
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, MicroUnit);

		r.property("WarnOfDataLoss",
		           mWarnOfDataLoss,
		           setterNotify(mWarnOfDataLoss, &TapeRecorderUnitBase::setWarnOfDataLoss, this),
		           "Warn when missed update is detected for a channel with AvoidDataLoss", false);

		r.roproperty("QueueSize",
		             getter(&TapeRecorderUnitBase::storageQueueSize, this),
		             "Size of TapeRecorder storage queue");
	}

public:
	void pause() override;

public:
	size_t storageQueueSize();
	void setWarnOfDataLoss();

	void recordConfig(const TapeRecorderConfig& config);

	void stopRecording();

protected:
	bool mWarnOfDataLoss;
	boost::mutex mRecorderMutex;
	boost::shared_ptr<TapeRecorder> mRecorder;
};

///////////////////////////////////////////////////////////////////////////////

void TapeRecorderUnitBase::pause()
{
	stopRecording();
}

size_t TapeRecorderUnitBase::storageQueueSize()
{
	boost::mutex::scoped_lock lock(mRecorderMutex);
	if (!mRecorder)
		return 0;
	return mRecorder->storageQueueSize();
}

void TapeRecorderUnitBase::setWarnOfDataLoss()
{
	boost::mutex::scoped_lock lock(mRecorderMutex);
	if (mRecorder)
		mRecorder->setWarnOfDataLoss(mWarnOfDataLoss);
}

void TapeRecorderUnitBase::recordConfig(const TapeRecorderConfig& config)
{
	boost::mutex::scoped_lock lock(mRecorderMutex);
	if (mRecorder)
		mRecorder->stop();
	mRecorder.reset(new TapeRecorder());
	mRecorder->setWarnOfDataLoss(mWarnOfDataLoss);
	mRecorder->record(config);
}

void TapeRecorderUnitBase::stopRecording()
{
	{
		boost::mutex::scoped_lock lock(mRecorderMutex);
		if (!mRecorder)
			return;
		mRecorder->stop(false);
	}
	// frequently release the mRecorderMutex lock during queue processing,
	// so that QueueSize property can update
	while (true) {
		{
			boost::mutex::scoped_lock lock(mRecorderMutex);
			if (!mRecorder)
				return;
			Time start = Time::now();
			// process queue for (at least) 10ms, then sleep for 1ms
			while (Time::now() - start < Duration::milliseconds(10)) {
				if (mRecorder->storageQueueSize() == 0) {
					mRecorder.reset();
					return;
				}
				mRecorder->processStorageQueue(1);
			}
		}
		MIRA_SLEEP(1);
	}
}

///////////////////////////////////////////////////////////////////////////////

/**
 * Unit that exposes access to a TapeRecorder via RPC.
 */
class TapeRecorderUnit : public TapeRecorderUnitBase
{
	MIRA_OBJECT(TapeRecorderUnit)
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, TapeRecorderUnitBase);
		r.method("addChannel", &TapeRecorderUnit::addChannel, this,
		         "Add a channel for recording",
		         "channelname", "channel ID to add", "robot/Odometry");

		RecordedChannelInfo info;
		info.codecs.push_back(BinarySerializerCodecPtr(new SampleCodec()));
		info.interval = Duration::milliseconds(100);

		r.method("addChannelInfo", &TapeRecorderUnit::addChannelInfo, this,
		         "Add a channel for recording along with its recording settings",
		         "channelname", "channel ID to add", "robot/Odometry",
		         "info", "settings for this channel", info);

		r.method("record", &TapeRecorderUnit::record, this,
		         "Start recording using the added channels",
		         "filename", "name of tape file to write", "/home/user/1.tape");

		TapeRecorderConfig config;
		config.channels["/robot/Robot"] = info;
		config.filename = "/home/user/1.tape";

		r.method("recordConfig", &TapeRecorderUnitBase::recordConfig, (TapeRecorderUnitBase*)this,
		         "Start recording using a config",
		         "config", "recording settings", config);

		r.method("stop", &TapeRecorderUnit::stopRecording, this,
		         "Stop recording and clear all added channels");
	}

public:
	void initialize() override;
	void pause() override;

	void addChannel(const std::string& channelName);
	void addChannelInfo(const std::string& channelName, const RecordedChannelInfo& info);
	void record(const Path& filename);

	void stopRecording();

protected:
	std::map<std::string, RecordedChannelInfo> mChannels;
};

///////////////////////////////////////////////////////////////////////////////

void TapeRecorderUnit::initialize()
{
	publishService(*this);
}

void TapeRecorderUnit::pause()
{
	// call our own stopRecording() (more specific than base class)
	stopRecording();
}

void TapeRecorderUnit::addChannel(const std::string& channelName)
{
	mChannels[channelName] = RecordedChannelInfo();
}

void TapeRecorderUnit::addChannelInfo(const std::string& channelName,
                                      const RecordedChannelInfo& info)
{
	mChannels[channelName] = info;
}

void TapeRecorderUnit::record(const Path& filename)
{
	boost::mutex::scoped_lock lock(mRecorderMutex);
	if (mRecorder)
		mRecorder->stop();
	mRecorder.reset(new TapeRecorder());
	foreach(auto& channel, mChannels)
		mRecorder->addChannel(channel.first, channel.second);
	mRecorder->setWarnOfDataLoss(mWarnOfDataLoss);
	mRecorder->record(filename);
}

void TapeRecorderUnit::stopRecording()
{
	TapeRecorderUnitBase::stopRecording();
	mChannels.clear();
}

///////////////////////////////////////////////////////////////////////////////

/**
 * Unit that automatically starts recording at application startup,
 * as specified by config file.
 */
class AutoTapeRecorderUnit : public TapeRecorderUnitBase
{
	MIRA_OBJECT(AutoTapeRecorderUnit)
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, TapeRecorderUnitBase);
		r.member("RecordConfig", mConfig,
		         "The config used for recording",
		         boost::shared_ptr<TapeRecorderConfig>(),
		         REFLECT_CTRLFLAG_MEMBER_AS_ROPROPERTY);
		r.member("DefaultRecordPath", mDefaultRecordPath,
		         "Default path prefixed to configured filename (using '/') "
		         "when not selected by RecordPath input channel",
		         Path(), REFLECT_CTRLFLAG_MEMBER_AS_ROPROPERTY);

		r.member("RecordOnStart", mRecordOnStart,
		         "Start recording when the unit starts? (default=true)",
		         true, REFLECT_CTRLFLAG_MEMBER_AS_ROPROPERTY);

		r.method("record", &AutoTapeRecorderUnit::record, this,
		         "Start recording using RecordConfig, to path read from RecordPath channel "
		         "or default path if empty");

		r.method("stop", &TapeRecorderUnitBase::stopRecording, (TapeRecorderUnitBase*)this,
		         "Stop recording");
	}

public:
	void initialize() override;
	void resume() override;

public:
	void record();

	void recordTo(const Path& path);

protected:
	boost::shared_ptr<TapeRecorderConfig> mConfig;
	Path mDefaultRecordPath;
	bool mRecordOnStart = true;
	Channel<Path> mRecordPathChannel;
};

///////////////////////////////////////////////////////////////////////////////

void AutoTapeRecorderUnit::initialize()
{
	mRecordPathChannel = subscribe<Path>("RecordPath");
	publishService(*this);
}

void AutoTapeRecorderUnit::resume()
{
	if (mConfig && mRecordOnStart)
		record();
}

void AutoTapeRecorderUnit::record()
{
	if (!mConfig)
		MIRA_THROW(XInvalidParameter, "Config does not exist");

	Path filenamePrefix;

	try {
		filenamePrefix = mRecordPathChannel.get();
	}
	catch (XInvalidRead& ex) {
		filenamePrefix = mDefaultRecordPath;
	}

	TapeRecorderConfig config = *mConfig;
	config.filename = filenamePrefix / mConfig->filename;

	recordConfig(config);
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

MIRA_CLASS_SERIALIZATION(mira::SampleCodec, mira::BinarySerializerCodec)
MIRA_CLASS_SERIALIZATION(mira::TapeRecorderUnit, mira::MicroUnit);
MIRA_CLASS_SERIALIZATION(mira::AutoTapeRecorderUnit, mira::MicroUnit);
