/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file UnitLoader.C
 *    Implementation of an UnitLoader
 *
 * @author Tim Langner
 * @date   2010/09/16
 */

#include <error/Exceptions.h>

#include <fw/MicroUnit.h>
#include <fw/Framework.h>
#include <fw/UnitManager.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Loader plugin that loads, instantiates and deserializes units.
 */
class UnitLoader : public ConfigurationLoaderPlugin
{
	MIRA_META_OBJECT(UnitLoader, ("Tag", "unit"))
public:
	virtual void parseNode(const XMLDom::const_iterator& node,
	                       ConfigurationLoader* loader);

	// provided for backward compatibility
	// TODO: delete this when we drop the backward compatibility
	virtual void parseNodeLegacy(const XMLDom::const_iterator& node,
	                       ConfigurationLoader* loader);
};


void UnitLoader::parseNode(const XMLDom::const_iterator& node,
                           ConfigurationLoader* loader)
{
	auto itID = node.find_attribute("id");
	if (itID == node.attribute_end())
		MIRA_THROW(XInvalidConfig, "Error in <unit>. "
		           "No 'id' attribute specified. "
		           "Usage <unit id=\"MyUnitName\" class=\"mynamespace::MyClass\">...</unit>.");
	std::string id = itID.value();

	if ( loader->getContext().count("unit_id") > 0 )
		MIRA_THROW(XInvalidConfig, "Error in <unit>. "
		           "Units cannot be declared inside other <unit> tags. ('"
		           << id << "' inside '" << loader->getContext()["unit_id"] << "')");

	loader->getContext()["unit_id"] = id;

	bool startAtInit=true;
	auto itStart = node.find_attribute("start");
	if (itStart != node.attribute_end())
		startAtInit = fromString<bool>(itStart.value());

	auto itClass = node.find_attribute("class");
	if (itClass == node.attribute_end()) {
		// TODO: remove this and uncomment code below when we drop the backward compatibility
		parseNodeLegacy(node, loader);
		return;
		//MIRA_THROW(XInvalidConfig, "Error in <unit>. "
		//          "No 'class' attribute specified for unit '" << id <<
		//           << "' Usage <unit id=\"MyUnitName\" class=\"mynamespace::MyClass\">...</unit>.");
	}
	loader->getContext()["unit_class"] = itClass.value();

	MicroUnit* unit = NULL;
	try {
		XMLDeserializer deserializer(node);
		deserializer.deserializeFromNode("unit", unit);
	} catch(Exception& ex) {
		MIRA_RETHROW(ex, "while deserializing unit '" << id <<"' from configuration")
	}
	MicroUnitPtr unitPtr(unit);
	std::string ns = loader->getContext()["namespace"];
	std::string threadGroup;
	if (loader->getContext().count("thread") > 0)
		threadGroup = loader->getContext()["thread"];
	MIRA_FW.getUnitManager()->addUnit(unitPtr, ns, id, threadGroup, startAtInit);

	loader->getContext().erase("unit_class");
	loader->getContext().erase("unit_id");
}


// TODO: delete this when we drop the backward compatibility
void UnitLoader::parseNodeLegacy(const XMLDom::const_iterator& node,
                           ConfigurationLoader* loader)
{
	// id was already parsed and checked by parseNode
	std::string authorityID = loader->getContext()["unit_id"];

	XMLDom::const_iterator i = node.begin();
	if (node.find("instance") == node.end())
		MIRA_THROW(XInvalidConfig, "Error in <unit>. "
		           "No <instance> child found inside ('"
		           << authorityID << "'.");

	bool foundInstance = false;
	for (; i != node.end(); ++i)
	{
		if ( *i == "instance" )
		{
			if (foundInstance)
				MIRA_THROW(XInvalidConfig, "Error in <unit>. "
				           "Only one <instance> child is supported inside ('"
				           << authorityID << "'.");
			foundInstance = true;
			auto iClass = i.find_attribute("class");
			if (iClass == i.attribute_end())
				MIRA_THROW(XInvalidConfig, "Error in <instance>. "
				           "No class attribute specified inside ('"
				           << authorityID << "'. Usage <instance class=\"namespace::ClassName\">...</instance>.");

			MIRA_LOG(WARNING) << "Usage of <instance> is deprecated, use "
					"<unit"
					" id=\"" << authorityID << "\""
					" class=\""	<<iClass.value() << "\">...</unit> "
					"instead. (" << node.uri() << ":" << node.line() << ")";

			loader->getContext()["unit_class"] = iClass.value();
			MicroUnit* unit = NULL;
			try {
				XMLDeserializer deserializer(node);
				deserializer.deserialize("instance", unit);
			} catch(Exception& ex) {
				MIRA_RETHROW(ex, "while deserializing unit '" << authorityID <<"' from configuration")
			}
			MicroUnitPtr unitPtr(unit);
			std::string ns = loader->getContext()["namespace"];
			std::string threadGroup;
			if (loader->getContext().count("thread") > 0)
				threadGroup = loader->getContext()["thread"];
			MIRA_FW.getUnitManager()->addUnit(unitPtr, ns, authorityID, threadGroup);
		}
		else
			loader->parseNode(i);
	}
	loader->getContext().erase("unit_class");
	loader->getContext().erase("unit_id");
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_REGISTER(mira::UnitLoader, mira::ConfigurationLoaderPlugin)
