/*
 * Copyright (C) 2015 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RemoteConnectionPool.C
 *    Implementation of RemoteConnectionPool.h.
 *
 * @author Tom Mehner
 * @date   Feb 4, 2021
 */

#include <fw/Framework.h>

#include <fw/RemoteConnectionPool.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

void swap(RemoteConnectionProxy& a, RemoteConnectionProxy& b)
{
	using std::swap;
	swap(a.mRemoteConnectionPool, b.mRemoteConnectionPool);
	swap(a.mRemoteConnection, b.mRemoteConnection);
}

RemoteConnectionProxy::RemoteConnectionProxy(RemoteConnectionPool* pool,
                                             RemoteConnection* connection)
    : mRemoteConnectionPool(pool), mRemoteConnection(connection)
{
}

RemoteConnectionProxy::RemoteConnectionProxy(RemoteConnectionProxy&& other) noexcept
    : RemoteConnectionProxy()
{
	swap(*this, other);
}

RemoteConnectionProxy& RemoteConnectionProxy::operator=(RemoteConnectionProxy&& other) noexcept
{
	swap(*this, other);
	return *this;
}

RemoteConnectionProxy::~RemoteConnectionProxy()
{
	if (mRemoteConnectionPool)
		mRemoteConnectionPool->stopConnection(mRemoteConnection);
}

bool RemoteConnectionProxy::valid() const
{
	if (mRemoteConnection == nullptr || mRemoteConnectionPool == nullptr)
		return false;

	boost::recursive_mutex::scoped_lock lock(mRemoteConnectionPool->mConnectionsMutex);
	const auto& remoteConnections = mRemoteConnectionPool->mRemoteConnections;
	return remoteConnections.find(mRemoteConnection) != remoteConnections.end();
}

///////////////////////////////////////////////////////////////////////////////

RemoteConnectionProxy RemoteConnectionPool::createRemoteOutgoingConnection(const KnownFramework& address)
{
	RemoteOutgoingConnectionBase* connection = nullptr;
	if (address.binaryFormatVersion == BinaryBufferSerializer::getSerializerFormatVersion())
		connection = new RemoteOutgoingConnection(address);
	else if (address.binaryFormatVersion == 0)
		connection = new RemoteOutgoingConnectionLegacy(address);
	else
		MIRA_THROW(XInvalidParameter,
		           "Connection to remote framework requested with binary format version " <<
		           (int)address.binaryFormatVersion << ". "
		           "RemoteConnection only implemented for versions 0, " <<
		           (int)BinaryBufferSerializer::getSerializerFormatVersion() << ".");

	boost::recursive_mutex::scoped_lock lock(mConnectionsMutex);
	mRemoteConnections.insert(connection);
	return RemoteConnectionProxy(this, connection);
}

RemoteConnectionProxy RemoteConnectionPool::createRemoteIncomingConnection()
{
	RemoteIncomingConnection* connection = new RemoteIncomingConnection();

	boost::recursive_mutex::scoped_lock lock(mConnectionsMutex);
	mRemoteConnections.insert(connection);
	return RemoteConnectionProxy(this, connection);
}

void RemoteConnectionPool::stopConnection(RemoteConnection* connection)
{
	if (connection == nullptr)
		return;

	boost::recursive_mutex::scoped_lock lock(mConnectionsMutex);
	auto it = mRemoteConnections.find(connection);
	if (it != mRemoteConnections.end()) {
		mStoppedConnections.push_back(std::unique_ptr<RemoteConnection>(connection));
		mRemoteConnections.erase(it);
	}
}

void RemoteConnectionPool::releaseStoppedConnections()
{
	boost::recursive_mutex::scoped_lock lock(mConnectionsMutex);
	mStoppedConnections.clear();
}

///////////////////////////////////////////////////////////////////////////////

}  // namespace mira
