/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ErrorService.C
 *    Implementation of ErrorService.h.
 *
 * @author Tim Langner
 * @date   2010/11/08
 */

#include <fw/ErrorService.h>

#include <error/Exceptions.h>
#include <database/SQLiteDB.h>

///////////////////////////////////////////////////////////////////////////////

namespace mira {

///////////////////////////////////////////////////////////////////////////////

ErrorService::ErrorService(Authority& authority, const std::string& errorDBFilename) :
	mAuthority(authority),
	mErrorDBFilename(errorDBFilename)
{
	mAuthority.publishService(*this);
	checkDB();
}

void ErrorService::setError(const std::string& category,
                            const std::string& authority,
                            const std::string& errorTxt,
                            const std::string& message)
{
	Time now = Time::now();

	boost::mutex::scoped_lock lock(mMutex);
	checkDB();

	std::string query = MakeString() << "SELECT * FROM error WHERE category = \""
		<< category << "\" AND authority = \"" << authority
		<< "\" AND txt = \"" << errorTxt << "\" AND message = \"" << message << "\"";
	SQLiteQuery q = mDB->query(query);

	// no error exists
	if ( q.eof() )
	{
		std::string exec = MakeString() << "INSERT INTO error (category, authority,"
			<< "txt, count, last, message) VALUES (\""
			<< category << "\",\"" << authority << "\",\"" << errorTxt << "\",1,"
			<< now.toUnixNS() << ",\"" << message << "\");";
		mDB->exec(exec);
	}
	else
	{
		std::string exec = MakeString() << "UPDATE error SET count = "
			<< (q.getValue<int>(4) + 1) << ", last = " << now.toUnixNS()
			<< ", message = \"" << message << "\""
			<< " WHERE id = " <<
			q.getValue<int>(0) << ";";
		mDB->exec(exec);
	}
}

void ErrorService::resetError(const std::string& category,
                              const std::string& authority)
{
	boost::mutex::scoped_lock lock(mMutex);
	checkDB();

	std::string exec = MakeString() << "DELETE FROM error WHERE category = \""
		<< category << "\" AND authority = \"" << authority
		<< "\"";
	mDB->exec(exec);
}

void ErrorService::resetErrors(const std::string& authority)
{
	boost::mutex::scoped_lock lock(mMutex);
	checkDB();

	std::string exec = MakeString() <<
		"DELETE FROM error WHERE authority = \"" << authority << "\"";
	mDB->exec(exec);
}

uint32 ErrorService::getErrorCount(const std::string& authority)
{
	boost::mutex::scoped_lock lock(mMutex);
	if (!mDB)
		return 0;

	std::string exec = MakeString() <<
		"SELECT count(*) FROM error WHERE authority = \"" << authority << "\"";
	return mDB->query(exec).getValue<uint32>(0);
}

ErrorService::ErrorVector ErrorService::getErrors()
{
	boost::mutex::scoped_lock lock(mMutex);
	ErrorVector ret;
	if (!mDB)
		return ret;	

	SQLiteQuery q = mDB->query("SELECT category, authority, txt, message, count, last FROM error;");
	while ( !q.eof() )
	{
		Error e;
		e.category = q.getValue<std::string>(0);
		e.authority = q.getValue<std::string>(1);
		e.trText = q.getValue<std::string>(2);
		e.message = q.getValue<std::string>(3);
		e.count = q.getValue<uint32>(4);
		e.timestamp = Time::fromUnixNS(q.getValue<uint64>(5));
		ret.push_back(e);
		++q;
	}
	return ret;
}

std::string ErrorService::getErrorText(const std::string& error,
                                       const std::string& language)
{
	MIRA_THROW(XNotImplemented, "Not implemented");
	return "";
}

void ErrorService::checkDB()
{
	if ( !mDB )
	{
		try
		{
			mDB.reset(new SQLiteDB(mErrorDBFilename));

			if ( !mDB->tableExists("error") )
			{
				std::string exec = MakeString()
					<< "CREATE TABLE error(id INTEGER PRIMARY KEY, "
					<< "category TEXT, authority TEXT, "
					<< "txt TEXT, count INTEGER, last INTEGER, "
					<< "message TEXT);";
				mDB->exec(exec);
			}
		}
		catch(XSQLite& ex)
		{
			MIRA_LOG(ERROR) << "SQLiteUnit: " << ex.what();
			mDB.reset();
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

}
