/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Unit.h
 *    More complex base class for units that provides a configurable
 *    worker thread (process()).
 *
 * @author Tim Langner, Erik Einhorn
 * @date   2010/09/06
 */

#ifndef _MIRA_UNIT_H_
#define _MIRA_UNIT_H_

#include <fw/MicroUnit.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup FWModule
 * A more complex unit that adds a default timer to the thread dispatcher
 * of the authority that acts as a worker thread (process()).
 * The period of the timer can be altered via a property.
 */
class Unit : public MicroUnit
{
	MIRA_OBJECT(Unit)
public:

	/**
	 * Constructs a unit with given cycle time.
	 * @param[in] cycleTime The interval in which process() gets called.
	 *            Is also used as tolerance for the cycle time
	 * @param[in] flags The creating flags (see @ref Authority::Flags).
	 */
	Unit(Duration cycleTime, Flags flags = NORMAL) :
		MicroUnit(flags),
		mResetProcessTimer(true)
	{
		setCycleTime(cycleTime);
		setCycleTimeTolerance(cycleTime);
	}

	/**
	 * Constructs a unit with given cycle time and a tolerance for the
	 * cycle time.
	 * @param[in] cycleTime The interval in which process() gets called.
	 * @param[in] cycleTimeTolerance The tolerance for the timer that calls process()
	 * @param[in] flags The creating flags (see @ref Authority::Flags).
	 */
	Unit(Duration cycleTime, Duration cycleTimeTolerance, Flags flags = NORMAL) :
		MicroUnit(flags),
		mResetProcessTimer(true)
	{
		setCycleTime(cycleTime);
		setCycleTimeTolerance(cycleTimeTolerance);
	}

	virtual ~Unit()
	{}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, MicroUnit);
		r.property("CycleTime",
		           getter<Duration>(&Unit::getCycleTime, this),
		           setter<Duration>(&Unit::setCycleTime, this),
		           "The cycle time of our process", mCycleTime,
		           PropertyHints::limits<Duration>(Duration::milliseconds(0),
		                                           Duration::hours(1)) |
		           PropertyHints::step<Duration>(Duration::milliseconds(10)));
		r.property("CycleTimeTolerance",
		           getter<Duration>(&Unit::getCycleTimeTolerance, this),
		           setter<Duration>(&Unit::setCycleTimeTolerance, this),
		           "The maximum time about that we can exceed the cycle time "
		           "before issuing a warning", mCycleTimeTolerance,
		           PropertyHints::limits<Duration>(Duration::milliseconds(1),
		                                           Duration::hours(1)) |
		           PropertyHints::step<Duration>(Duration::milliseconds(10)));
	}

	/// overwritten from MicroUnit to add the process timer
	virtual void checkin(const std::string& ns, const std::string& name);

public:

	/**
	 * Set the cycle period the process(const Timer& timer) method gets called.
	 */
	void setCycleTime(Duration period);

	/**
	 * Returns the cycle period the process(const Timer& timer) method gets called.
	 */
	Duration getCycleTime() const;

	/**
	 * Set the cycle period tolerance for exceeding the period while calling
	 * the process(const Timer& timer) method
	 */
	void setCycleTimeTolerance(Duration tolerance);

	/**
	 * Returns the cycle period tolerance for exceeding the period while calling
	 * the process(const Timer& timer)
	 */
	Duration getCycleTimeTolerance() const;

	/**
	 * @name Implementation of MicroUnit
	 */
	//@{

	/// see MicroUnit::start()
	virtual void start();

	/// see MicroUnit::stop()
	virtual void stop();

	//@}

protected:

	/** @name Interface for derived classes */
	//@{
	/**
	 * Overload this method in derived class. It gets called in the specified
	 * interval if not needRecovery() is set.
	 * It will be executed in the Units main thread - the same thread
	 * initialize() was called in so there is no need to decouple call backs
	 * and the process method.
	 * You also don't need mutexes to lock your data.
	 */
	virtual void process(const Timer& timer);

	/// see MicroUnit::needRecovery()
	virtual void needRecovery(const std::string& reason = "");

	/// see MicroUnit::operational()
	virtual void operational();

private:

	void processIntern(const Timer& timer);

	Duration mCycleTime;
	Duration mCycleTimeTolerance;
	TimerPtr mProcessTimer;

	bool mResetProcessTimer;
};

///////////////////////////////////////////////////////////////////////////////

}

MIRA_NO_PUBLIC_DEFAULT_CONSTRUCTOR(Unit)

#endif
