/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ServiceProperty.h
 *    Property that can be used whenever a unit has a service as property.
 *
 * @author Erik Einhorn, Tim Langner
 * @date   2012/03/26
 */

#ifndef _MIRA_SERVICEPROPERTY_H_
#define _MIRA_SERVICEPROPERTY_H_

#include <serialization/IgnoreMissing.h>
#include <serialization/ReflectorInterface.h>
#include <serialization/XMLSerializer.h>

#include <fw/Framework.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Use this class instead of a string property whenever your module has
 * a service as property. When the user clicks on that property
 * a dialog will open where he can select a service that implements the given
 * interface out of all existing ones.
 * \code
 * class MyClass {
 * 		template <typename Reflector>
 * 		void reflect(Reflector& r) {
 * 			r.property("Service", myService, "The service name",
 *                                 ServiceProperty("ITheInterface"));
 * 		}
 * 
 * 		MyClass() : myService("ITheInterface") {}
 * 
 * 		ServiceProperty myService;
 * \endcode
 */
class ServiceProperty
{
public:

	ServiceProperty() {}

	ServiceProperty(const ServiceProperty& other) :
		mService(other.mService), mInterface(other.mInterface) {}

	/// Constructor where the interface is specified that the service must implement
	explicit ServiceProperty(const std::string& interface) :
		mInterface(interface) {}

	template <typename Reflector>
	void reflect(Reflector& r) {
		r.member("Service", mService, "The concrete service that is used", "");
		r.member("Interface", mInterface, "The interface that is required");
	}

	void reflect(XMLDeserializer& r) {
		r.member("Service", mService, "The concrete service that is used", "");

		std::string interface;
		r.member("Interface", interface, "The interface that is required",
		         serialization::IgnoreMissing(), REFLECT_CTRLFLAG_TEMP_TRACKING);

		if (interface.empty()) {
			if (mService.empty()) {
				r.atomic(mService); // if there is neither Service nor Interface node,
				                    // try to read service name directly from node's content
				                    // (for backward compatibility)
				if (!mService.empty()) {
					MIRA_LOG(WARNING) << "Expected Service and (optional) Interface nodes for "
					                     "ServiceProperty '" << *r.getNode() << "', "
					                     "found string '" << mService <<
					                     "' (assuming this for service name). "
					                     "Please update your config xml (" <<
					                     r.getNode().uri() << " , line " << r.getNode().line() << ").";
				}
			}
		} else
			mInterface = interface;

		// Interface node is only strictly required if mInterface was not initialized
		// in constructor already
		if (mInterface.empty())
			MIRA_THROW(XMemberNotFound,
			          "ServiceProperty: Node 'Interface' is missing or empty");
	}

	bool operator==(const ServiceProperty& other) const {
		return mInterface == other.mInterface && mService == other.mService;
	}

	bool operator!=(const ServiceProperty& other) const {
		return !operator==(other);
	}

	ServiceProperty& operator=(const ServiceProperty& other) {
		mService = other.mService;
		mInterface = other.mInterface;
		return *this;
	}

	ServiceProperty& operator=(const std::string& s) {
		mService = s;
		return *this;
	}

	/// Return the service that is associated with this property
	const std::string& getService() const {
		return mService;
	}

	/// Sets the interface a service must implement
	void setInterface(const std::string& interface) {
		mInterface = interface;
	}

	/// Return the interface a service must implement
	const std::string& getInterface() const {
		return mInterface;
	}

	operator std::string() const { return mService; }

	bool isEmpty() const {
		return mService.empty();
	}

	bool empty() const {
		return mService.empty();
	}

	bool isValid() const {
		if (isEmpty())
			return false;

		return MIRA_FW.getRPCManager().existsService(mService);
	}

private:

	std::string mService;   ///< The selected service or empty if none
	std::string mInterface; ///< The interface services must implement
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
