/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ServiceCall.h
 *    Class that acts as a proxy for rpc service calls to provide a function like interface
 *
 * @author Tom Mehner
 * @date   2023/12/04
 */

#ifndef MIRA_FRAMEWORK_INCLUDE_FW_SERVICECALL_H_
#define MIRA_FRAMEWORK_INCLUDE_FW_SERVICECALL_H_

#include <fw/Framework.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

template<typename>
class ServiceCall;

/**
 * A ServiceCall is a proxy for a specific method of a specific service, providing
 * a functor-like interface to that method, and encapsulation for checking the
 * availability of the service and method (name, signature).
 * See \ref Authority::createServiceCall().
 */
template<class Ret, class... ARGS>
class ServiceCall<Ret(ARGS...)>
{
public:
	/**
	 * @brief Construct a ServiceCall object referring to a
	 * specific service provider and method.
	 *
	 * @param serviceProvider The resolved name of the service provider.
	 *                        This object knows nothing about
	 *                        namespaces and is unable to resolve an alias
	 * @param method          The method name
	 */
	ServiceCall(std::string serviceProvider, std::string method)
	: mServiceProvider(std::move(serviceProvider)), mMethod(std::move(method))
	{}

	/**
	 * @brief Construct a new (invalid) ServiceCall object.
	 *        A default constructor must exist to enable creating a
	 *        ServiceCall object by deserializing its description.
	 */
	ServiceCall() = default;

	/**
	 * @brief Call operator to allow function like usage
	 *
	 * @param args expected arguments
	 * @return Ret return value
	 */
	[[nodiscard]] Ret operator()(const ARGS&... args) const
	{
		return call(args...).get();
	}

	[[nodiscard]] RPCFuture<Ret> call(const ARGS&... args) const
	{
		return MIRA_FW.getRPCManager().call<Ret>(mServiceProvider, mMethod, args...);
	}

	void waitForServiceCall() const
	{
		const auto hasServiceCall = waitForServiceCall(Duration::infinity());
	}

	[[nodiscard]] bool waitForServiceCall(Duration timeout) const
	{
		const Time end =
		    (!timeout.isValid() || timeout.isInfinity()) ? Time::eternity() : Time::now() + timeout;

		if (!MIRA_FW.getRPCManager().waitForService(mServiceProvider, timeout)) {
			return false;
		}

		auto matchSignature = [sig = getSignature()](const auto& method) { return method.signature == sig; };

		while (!boost::this_thread::interruption_requested()) {
			if (MIRA_FW.getRPCManager().getLocalServices().count(mServiceProvider)) {
				const auto& methods = MIRA_FW.getRPCManager().getLocalService(mServiceProvider).methods;
				if (std::any_of(methods.begin(), methods.end(), matchSignature)) {
					return true;
				}
			}
			// the service may unregister, it is not necessarily found in either local or remote services
			else if (MIRA_FW.getRPCManager().getRemoteServices().count(mServiceProvider)) {
				const auto& methods = MIRA_FW.getRPCManager().getRemoteService(mServiceProvider).methods;
				if (std::any_of(methods.begin(), methods.end(), matchSignature)) {
					return true;
				}
			}

			if (Time::now() > end) // handle timeout
				break;

			MIRA_SLEEP(100)
		}
		return false;
	}

	[[nodiscard]] RPCSignature getSignature() const
	{
		return mira::makeRPCSignature<Ret, ARGS...>(mMethod);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("ServiceProvider", mServiceProvider, "Provider for the service method",
		         REFLECT_CTRLFLAG_MEMBER_AS_ROPROPERTY);
		r.member("Method", mMethod, "Name of method to be called",
		         REFLECT_CTRLFLAG_MEMBER_AS_ROPROPERTY);
	}

private:
	std::string mServiceProvider;
	std::string mMethod;
};

///////////////////////////////////////////////////////////////////////////////

} // namespace mira

#endif // MIRA_FRAMEWORK_INCLUDE_FW_SERVICECALL_H_
