/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RemoteModule.h
 *    The framework module responsible for connections to remote frameworks.
 *
 * @author Tim Langner
 * @date   2010/11/16
 */

#ifndef _MIRA_REMOTEMODULE_H_
#define _MIRA_REMOTEMODULE_H_

#include <thread/CyclicRunnable.h>
#include <serialization/adapters/std/list>
#include <serialization/adapters/boost/optional.hpp>
#include <utils/UUID.h>
#include <security/RSAKey.h>

#include <fw/DiscoverService.h>
#include <fw/PerformanceStatistics.h>
#include <fw/RemoteConnection.h>
#include <fw/RemoteServer.h>
#include <fw/RemoteConnectionPool.h>
#include <fw/ServiceLevel.h>

namespace mira {

class RemoteServer;

///////////////////////////////////////////////////////////////////////////////

/**
 * Manages connections to other remote frameworks. Exchanges channel data,
 * authorities, RPC calls and handles time synchronization between frameworks.
 * The handshake protocol for establishing a connection between two frameworks
 * is as follows:
 * \code
 * No Authentication
 * 
 * Client                                            Server
 * |                                                    |
 * | CONNECT_MSG: Version, Group, ClientIDs, AuthMode 0 |
 * |--------------------------------------------------->|
 * |                                                    |
 * | CONNECT_ACCEPT_MSG: ServerIDs Version              |
 * |<---------------------------------------------------|
 * 
 * On error (e.g. wrong version, wrong authentication, etc)
 * | CONNECT_DENIED_MSG: message                        |
 * |<---------------------------------------------------|
 * 
 * 
 * Password Authentication
 * 
 * Client                                            Server
 * | CONNECT_MSG: Version, ClientIDs, AuthMode 1, PW    |
 * |--------------------------------------------------->|
 * |                                                    |
 * | CONNECT_ACCEPT_MSG: ServerIDs Version              |
 * |<---------------------------------------------------|
 * 
 * On error
 * | CONNECT_DENIED_MSG: message                        |
 * |<---------------------------------------------------|
 * 
 * Key Authentication
 * 
 * Client                                            Server
 * | CONNECT_MSG: Version, ClientIDs, AuthMode 2, Msg1  |
 * |--------------------------------------------------->|
 * |                                                    |
 * | SERVER_AUTH_MSG: Signed Msg1, Msg2                 |
 * |<---------------------------------------------------|
 * |                                                    |
 * | CLIENT_AUTH_MSG: Signed Msg2                       |
 * |--------------------------------------------------->|
 * |                                                    |
 * | CONNECT_ACCEPT_MSG: ServerIDs Version              |
 * |<---------------------------------------------------|
 * 
 * On error
 * | CONNECT_DENIED_MSG: message                        |
 * |<---------------------------------------------------|
 * 
 * CONNECT_DENIED_MSG is sent from server only,
 * clients simply close the connection on error.
 * \endcode
 */
class RemoteModule
{
public:
	typedef boost::shared_ptr<RemoteServer> RemoteServerPtr;

	typedef std::list<RemoteConnectionProxy> ConnectionList;
	typedef std::map<UUID, RemoteConnectionProxy> ConnectionMap;

	typedef std::list<KnownFramework> KnownFrameworkList;
	typedef std::list<ServiceLevel> ServiceLevelList;
	typedef std::map<std::string, ServiceLevel> ServiceLevelMap;
	typedef std::list<TypeServiceLevel> TypeServiceLevelList;
	typedef std::map<Typename, TypeServiceLevel> TypeServiceLevelMap;

	friend class RemoteConnection;
	friend class RemoteIncomingConnection;
	friend class RemoteOutgoingConnectionBase;
	friend class DiscoverService;
	friend class RemoteServer;

	/// Authentication mode
	enum AuthMode
	{
		AUTH_NONE=0,
		AUTH_PASSWORD=1,
		AUTH_KEY=2
	};

	/**
	 * Contains the authentication settings.
	 * Each framework may belong to a certain workgroup. The framework will be
	 * able to connect to other frameworks within the same group only.
	 * Moreover, there are three modes for authentication:
	 * -# No additional authentication
	 * -# Weak authentication using a password
	 * -# Strong authentication using a RSA key pair
	 */
	struct AuthSettings
	{
		std::string group;

		template<typename Reflector>
		void reflect(Reflector& r)
		{
			r.member("Group", group,
			         "The workgroup, this framework belongs to", "");
			r.member("Password", mPassword, setter(&AuthSettings::setPassword, this),
			         "Password that is used for the weak authentication",
			         serialization::IgnoreMissing());
			r.member("Key", mKeyFile, setter(&AuthSettings::setKeyFile, this),
			         "Path to the file containing the private key "
			         "for strong authentication.",
			         serialization::IgnoreMissing());
		}

		AuthMode getMode() const {
			if(mKey)
				return AUTH_KEY;
			else if(mPassword)
				return AUTH_PASSWORD;
			else
				return AUTH_NONE;
		}

		/// Clears authentication settings
		void clearAuth();

		/**
		 * Returns the authentication password
		 * @throw XInvalidParameter if no password is set
		 */
		const std::string& getPassword() const;

		/// Sets a password
		void setPassword(const boost::optional<std::string>& passwd);

		/**
		 * Returns the authentication key
		 * @throw XInvalidParameter if no key is set
		 */
		const RSAKey& getKey() const;

		/// Sets a key
		void setKey(const boost::optional<std::string>& str);

		/// Sets a key file
		void setKeyFile(const boost::optional<std::string>& file);

	private:

		boost::optional<std::string> mPassword;
		boost::optional<std::string> mKeyFile;
		boost::optional<RSAKey> mKey;
	};

	RemoteModule();

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		serialization::VersionType version = r.version(2, this);
		r.member("Port", mPort,
		         "The port where the framework server runs on. 0 means auto port", 0);
		r.member("IOThreadCount", mIOThreadCount,
		         "How many threads are used for communication", 1);
		r.member("CycleTime", mCyclicRunnable.cycleTime,
		         "The cycle time of remote framework thread that observes connections",
		         Duration::seconds(1));

		r.property("EnablePTPSync", mEnablePTPSync, "Enable/Disable synchronization of clocks via PTP (effective for new connections)", true);
		r.property("EnablePingTimeout", mEnablePingTimeout, "Enable/Disable ping timeout (effective for new connections)", false);
		r.member("PingInterval", mPingInterval,
		         "The interval of sending ping messages.", Duration::seconds(1));
		r.member("PingTimeout", mPingTimeout,
		         "The maximum time before a connection is assumed as dead "
		         "when no ping messages are received.", Duration::seconds(10));

		r.member("ExitOnDisconnectTimeout", mExitOnDisconnectTimeout,
		         "Exits/Closes the framework if any connection could not be established or is closed for a certain time",
		         Duration::invalid());

		r.member("Authentication", mAuthSettings, "Authentication Settings",
		         AuthSettings());

		if (version >= 2)
			r.member("KnownFrameworks", mKnownFrameworks,
			         "The list of remote frameworks", KnownFrameworkList());
		else {
			// old KnownFramework serialization = delegate to just address
			std::vector<std::string> tmpFWs;
			r.member("KnownFrameworks", tmpFWs, "",
			         std::vector<std::string>(), REFLECT_CTRLFLAG_TEMP_TRACKING);
			mKnownFrameworks.clear();
			foreach (const std::string& a, tmpFWs) {
				mKnownFrameworks.emplace_back();
				mKnownFrameworks.back().address = a;
			}
		}

		r.member("ServiceLevels", mServiceLevels,
		         "The service level agreement per channel", ServiceLevelList());
		r.member("ServiceLevelsByType", mTypeServiceLevels,
		         "The service level agreement per channel type", TypeServiceLevelList());

		r.property("FrameworkConnections", mConnections,
		           "The list of active remote framework connections", serialization::IgnoreMissing());

		r.interface("IRemoteModule");
		r.method("migrateUnitToThisFramework", &RemoteModule::migrateUnitToThisFramework, this,
		         "Requests a unit from an other framework to migrate into this framework",
		         "id", "id of the unit to migrate", "/navigation/Pilot");
		r.method("isFrameworkConnected", &RemoteModule::isFrameworkConnected, this,
		         "Returns if a framework with ID is connected to this framework",
		         "frameworkID", "framework ID to check", "Framework_81ab60d5-0e9a-43d6-aced-7e33a3cd875a");
		r.method("isConnectedTo", &RemoteModule::isConnectedTo, this,
		         "Returns if a framework with address is connected to this framework",
		         "address", "address to check", "127.0.0.1:1234");
		r.method("disconnectFramework", &RemoteModule::disconnectFramework, this,
		         "Disconnects a framework with ID (if connected)",
		         "frameworkID", "framework ID to disconnect", "Framework_81ab60d5-0e9a-43d6-aced-7e33a3cd875a",
		         "autoReconnect", "if true, keep as known framework (thus try reconnecting)", false);
		r.method("disconnectFrom", &RemoteModule::disconnectFrom, this,
		         "Disconnects from a framework with address",
		         "address", "address to disconnect", "127.0.0.1:1234",
		         "autoReconnect", "if true, keep as known framework (thus try reconnecting)", false);
		r.method("connectTo",
		         (void(RemoteModule::*)(const std::string&))(&RemoteModule::addKnownFramework), this,
		         "Add an address to the list of known remote frameworks and try to connect",
		         "address", "Address of the remote framework as Host:Port", "127.0.0.1:1234");
		r.method("connectTo",
		         (void(RemoteModule::*)(const std::string&, bool, bool, bool, const Duration&))(&RemoteModule::addKnownFramework), this,
		         "Add an address to the list of known remote frameworks and try to connect",
		         "address", "Address of the remote framework as Host:Port", "127.0.0.1:1234",
		         "forcePTP", "Force PTP time sync", false,
		         "legacyBinaryFormat", "Use legacy binary serializer to encode data (connect to legacy framework)", false,
		         "monitorOnly", "Do not publish local channels, services and authorities to remote", false,
		         "delayConnect", "Wait for this duration before trying to connect", Duration::seconds(1));
	}

	/**
	 * Returns the version of the communication protocol, that is supported
	 * by THIS framework.
	 */
	static uint32 getCurrentVersion();

	/**
	 * Set the port of our TCP server.
	 * If not specified it uses auto port
	 * @param[in] port The port
	 */
	void setPort(uint16 port) { mPort = port; }

	/**
	 * Returns the port we listen on for incoming connections.
	 */
	uint16 getPort() const;


	/**
	 * Enable/Disable synchronization of clocks via PTP.
	 */
	void enablePTPSync(bool enable=true) { mEnablePTPSync = enable; }

	/**
	 * Returns true if PTP clock synchronization is enabled.
	 */
	bool isPTPSyncEnabled() const { return mEnablePTPSync; }

	/**
	 * Enable/Disable the ping timeout
	 */
	void enablePingTimeout(bool enable=true) { mEnablePingTimeout = enable; }

	/**
	 * Returns true if ping timeout is enabled.
	 */
	bool isPingTimeoutEnabled() const { return mEnablePingTimeout; }

	/**
	 * Return the ping interval.
	 */
	Duration getPingInterval() const { return mPingInterval; }

	/**
	 * Return the ping timeout.
	 */
	Duration getPingTimeout() const { return mPingTimeout; }

	/**
	 * Sets the working group for this framework. The framework will be
	 * able to connect to other frameworks within the same group only.
	 * The default group is "".
	 */
	void setAuthGroup(const std::string& group);

	/**
	 * Resets passwords and keys, and sets authentication to "none".
	 */
	void setNoAuth();

	/**
	 * Enables the weak password based authentication mode and sets the
	 * password. Previously set RSA keys will be dropped.
	 */
	void setAuthPassword(const std::string& password);

	/**
	 * Enables the strong RSA key based authentication mode and sets the
	 * key. A previously set password will be dropped.
	 */
	void setAuthKey(const std::string& key);

	/**
	 * Enables the strong RSA key based authentication mode and sets the
	 * key file. A previously set password will be dropped.
	 */
	void setAuthKeyFile(const std::string& keyfile);

	/**
	 * Returns the current authentication settings.
	 */
	const AuthSettings& getAuthSettings() const;

	/**
	 * Sets the timeout to exit on disconnect, closes the framework
	 * if a connection is closed or can not be established for this time
	 * (set Duration::invalid() to disable)
	 */
	void setExitOnDisconnectTimeout(Duration timeout) { mExitOnDisconnectTimeout = timeout; }

	/**
	 * Returns the unique ID of the remote framework.
	 */
	const UUID& getID() const { return mID; }

	/**
	 * Get the level of service for given channel.
	 */
	ServiceLevel getServiceLevel(const std::string& channelID, const Typename& channelType);

	/**
	 * Start the remote framework. Including the server.
	 */
	void start();

	/**
	 * Stops the remote framework.
	 */
	void stop();


	/**
	 * Returns the map of connections.
	 */
	ScopedAccess<const ConnectionMap, boost::recursive_mutex> getConnections() const;

	/**
	 * Adds the address of a framework to the list of known remote frameworks.
	 * This module will try to connect to all known frameworks. It will also
	 * reconnect on connection loss.
	 * @param[in] address Address of the remote framework in the form Host:Port.
	 */
	void addKnownFramework(const std::string& address);

	/**
	 * Adds the address of a framework to the list of known remote frameworks.
	 * This module will try to connect to all known frameworks. It will also
	 * reconnect on connection loss.
	 * @param[in] address Address of the remote framework in the form Host:Port.
	 * @param[in] forcePTP Force PTP time sync
	 * @param[in] legacyBinaryFormat If true, encode all binary data for remote
	 *                               connection using legacy serializer, thus
	 *                               it can communicate with legacy framework.
	 * @param[in] monitorOnly        If true, do not publish local channels, services
	 *                               and authorities to remote.
	 * @param[in] delayConnect       Wait for this duration before trying to connect.
	 */
	void addKnownFramework(const std::string& address,
	                       bool forcePTP,
	                       bool legacyBinaryFormat,
	                       bool monitorOnly,
	                       const Duration& delayConnect = Duration::seconds(0));

	/**
	 * Disconnects framework with given ID (if it is connected).
	 * If autoReconnect is false the framework is removed from the list 
	 * of known remote frameworks so there will be no reconnect.
	 */
	void disconnectFramework(const std::string& frameworkID,
	                         bool autoReconnect = false);

	/**
	 * Disconnects framework with given address (if it is connected).
	 * If autoReconnect is false the framework is removed from the list 
	 * of known remote frameworks so there will be no reconnect.
	 * @param[in] address Address of the remote framework in the form Host:Port.
	 * @param[in] autoReconnect If true there will be a reconnect
	 */
	void disconnectFrom(const std::string& address,
	                    bool autoReconnect = false);

	/**
	 * Returns if a framework with given ID is connected with us.
	 */
	bool isFrameworkConnected(const std::string& frameworkID) const;
	
	/**
	 * Returns if a framework with given address is connected to us.
	 * @param[in] address Address of the remote framework in the form Host:Port.
	 */
	bool isConnectedTo(const std::string& address) const;

	/**
	 * Requests that the unit with the given id migrates to this framework.
	 * If the unit is already located in this framework nothing happens.
	 * Otherwise the framework will ask all connected frameworks if they run 
	 * a unit with the given id and if a request is sent to move the unit into
	 * this framework. The remote framework will serialize the unit, destroy it
	 * and send it to this framework where it is created out of its serialized
	 * state.
	 */
	void migrateUnitToThisFramework(const std::string& id);

	/**
	 * @name Internal functions that should not be called directly but cannot
	 *       be protected. 
	 */
	//@{

	/**
	 * Publishes an authority to all connected remote frameworks.
	 */
	void publishAuthority(const AuthorityDescription& authority);

	/**
	 * Unpublishes an authority to all connected remote frameworks.
	 */
	void unpublishAuthority(const AuthorityDescription& authority);

	/**
	 * Publishes a service to all connected remote frameworks.
	 */
	void publishService(const std::string& service);

	/**
	 * Unpublishes a service to all connected remote frameworks.
	 */
	void unpublishService(const std::string& service);

	/**
	 * Notify all connected remote frameworks about the fact that we have a
	 * publisher for the given channel.
	 * All frameworks will subscribe if they have a subscriber for this channel
	 * @param[in] channelID The channel we have a publisher for.
	 * @param[in] type The typename of the channels type.
	 */
	void publishChannel(const std::string& channelID, const Typename& type);

	/**
	 * Notify all connected remote frameworks about the fact that we have no
	 * longer a publisher for the given channel. All frameworks will unsubscribe
	 * if they were subscribed to that channel.
	 * @param[in] channelID The channel we have no more publishers for.
	 */
	void unpublishChannel(const std::string& channelID);

	/**
	 * Notify all connected remote frameworks about the fact that we have a
	 * subscriber for the given channel.
	 * Subscribes on all remote frameworks that have a publisher for this channel.
	 * Remote frameworks will notify us on data changes.
	 * @param[in] channelID The channel we have a subscriber for.
	 */
	void subscribeChannel(const std::string& channelID);

	/**
	 * Notify all connected remote frameworks that we no longer have a subscriber 
	 * on the given channel. We will not longer receive updates on that channel.
	 * @param[in] channelID The channel we have no more subscribers for.
	 */
	void unsubscribeChannel(const std::string& channelID);

	//@}

	/// Update statistics about incoming data
	void updateIncomingStats(std::size_t size);
	/// Update statistics about outgoing data
	void updateOutgoingStats(std::size_t size);

	std::size_t getIncomingBytesPerSecond() const;
	std::size_t getOutgoingBytesPerSecond() const;

protected:

	RemoteConnectionProxy createIncomingConnection();

	/**
	 * Move a connection to the list of pending incoming connections.
	 * @param[in] start if true, call proxy->start()
	 */
	RemoteConnectionProxy& addPendingIncomingConnection(RemoteConnectionProxy&& proxy,
	                                                    bool start = true);

	/**
	 * If there is a proxy for connection in the list of pending incoming connections,
	 * remove it.
	 */
	void removePendingIncomingConnection(RemoteConnection* connection);

	/**
	 * Move a connection to the list of pending outgoing connections.
	 * @param[in] start if true, call proxy->start()
	 */
	RemoteConnectionProxy& addPendingOutgoingConnection(RemoteConnectionProxy&& proxy,
	                                                    bool start = true);

	/**
	 * If there is a proxy for connection in the list of pending outgoing connections,
	 * remove it.
	 */
	void removePendingOutgoingConnection(RemoteConnection* connection);

	/**
	 * Called when a new remote framework was discovered (e.g. via multi cast)
	 */
	void onRemoteFrameworkDiscovered(const std::string& host, uint16 port, UUID id);

	/**
	 * Add a connection to our internal map of connections
	 * (and update the property node).
	 * @param[in] iConnection The connection pointer
	 */
	void storeConnection(RemoteConnectionProxy iConnection);

	/**
	 * Remove a connection from our internal map of connections
	 * (and update the property node).
	 * @param[in] it Iterator to the entry to be removed
	 */
	void eraseConnection(ConnectionMap::iterator it);
	
	/**
	 * Called whenever a client connects to our server
	 * @param[in] iConnection The connection pointer
	 * @return true if connection is accepted, false if not
	 */
	bool onIncomingConnected(RemoteIncomingConnection* iConnection);

	/**
	 * Called whenever a client disconnects from our server
	 * @param[in] iConnection The connection pointer
	 */
	void onIncomingDisconnected(RemoteIncomingConnection* iConnection);

	/**
	 * Called whenever we connect to a remote server
	 * @param[in] iConnection The connection pointer
	 * @return true if connection is accepted, false if not
	 */
	bool onOutgoingConnected(RemoteOutgoingConnectionBase* iConnection);

	/**
	 * Called whenever one of our client disconnects from a remote server
	 * @param[in] iConnection The connection pointer
	 */
	void onOutgoingDisconnected(RemoteOutgoingConnectionBase* iConnection);

	/**
	 * The main process method of the remote module.
	 * Tries to connect to known remote frameworks and does
	 * time synchronization between frameworks.
	 */
	void process();

	void startDisconnectTimeout();
	void stopDisconnectTimeout();
	bool checkDisconnectTimeout();

protected:

	PerformanceStatistics mIncomingStats;
	PerformanceStatistics mOutgoingStats;
	uint32 mIncomingData;
	uint32 mOutgoingData;

	CyclicRunnable mCyclicRunnable;
	DiscoverServicePtr mDiscoverService;
	boost::thread mThread;
	uint16 mPort;
	bool mEnablePTPSync;
	bool mEnablePingTimeout;
	Duration mPingInterval;
	Duration mPingTimeout;
	uint16 mIOThreadCount;
	UUID mID;
	RemoteServerPtr mServer;

	ConnectionList mPendingOutgoingConnections;
	ConnectionList mPendingIncomingConnections;
	ConnectionMap mConnections;

	RemoteConnectionPool mRemoteConnectionPool;

	AuthSettings mAuthSettings;
	Duration mExitOnDisconnectTimeout;
	Time mDisconnectedSince;
	KnownFrameworkList mKnownFrameworks;
	ServiceLevelList mServiceLevels;
	TypeServiceLevelList mTypeServiceLevels;
	ServiceLevelMap mChannelServiceLevels;
	TypeServiceLevelMap mChannelTypeServiceLevels;
	mutable boost::recursive_mutex mConnectionMutex;
};

///////////////////////////////////////////////////////////////////////////////

/// Typedef for a pointer to RemoteModule
typedef boost::shared_ptr<RemoteModule> RemoteModulePtr;

///////////////////////////////////////////////////////////////////////////////

template <>
class IsCopyAssignable<RemoteModule::ConnectionMap> : public std::false_type {};

///////////////////////////////////////////////////////////////////////////////

}

#endif
