/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RobotDataTypesConversion.h
 *    Contains mira2ros and ros2mira specializations.
 *
 * @author Christian Reuther, Tim Langner
 * @date   2013/06/20
 */

#include <ros/ros.h>
#include <fw/Framework.h>
#include <std_msgs/Bool.h>
#include <std_msgs/Int8.h>
#include <std_msgs/Int16.h>
#include <std_msgs/Int32.h>
#include <std_msgs/Int64.h>
#include <std_msgs/Int8.h>
#include <std_msgs/Int16.h>
#include <std_msgs/Int32.h>
#include <std_msgs/Int64.h>
#include <std_msgs/UInt8.h>
#include <std_msgs/UInt16.h>
#include <std_msgs/UInt32.h>
#include <std_msgs/UInt64.h>
#include <std_msgs/UInt8.h>
#include <std_msgs/UInt16.h>
#include <std_msgs/UInt32.h>
#include <std_msgs/UInt64.h>
#include <std_msgs/String.h>

#include <robot/Odometry.h>
#include <nav_msgs/Odometry.h>
#include <tf/tf.h>

#include <robot/RangeScan.h>
#include <sensor_msgs/LaserScan.h>

/*
#include <robot/BatteryState.h>
#include <pr2_msgs/BatteryState2.h>
*/

#include <maps/OccupancyGrid.h>
#include <nav_msgs/OccupancyGrid.h>

///////////////////////////////////////////////////////////////////////////////

const float BATTERY_GOOD_VOLTAGE = 24.0f;
const float BATTERY_SHUTDOWN_VOLTAGE = 23.0f;

///////////////////////////////////////////////////////////////////////////////

template <typename T>
struct MIRA2ROSTrait
{
	typedef T type;
};

#define MIRA2ROSTRAIT(ctype, rostype) \
template <>                           \
struct MIRA2ROSTrait<ctype>           \
{                                     \
	typedef rostype type;             \
};

MIRA2ROSTRAIT(std::string, std_msgs::String)
MIRA2ROSTRAIT(bool, std_msgs::Bool)
MIRA2ROSTRAIT(int8, std_msgs::Int8)
MIRA2ROSTRAIT(int16, std_msgs::Int16)
MIRA2ROSTRAIT(int32, std_msgs::Int32)
MIRA2ROSTRAIT(int64, std_msgs::Int64)
MIRA2ROSTRAIT(uint8, std_msgs::UInt8)
MIRA2ROSTRAIT(uint16, std_msgs::UInt16)
MIRA2ROSTRAIT(uint32, std_msgs::UInt32)
MIRA2ROSTRAIT(uint64, std_msgs::UInt64)

///////////////////////////////////////////////////////////////////////////////

/**
 * Specialize this function for special data type conversion of MIRA channel data
 * to a ROS topic data
 */
template <typename T>
void mira2ros(mira::ChannelRead<T> data, typename MIRA2ROSTrait<T>::type& out)
{
	out.data = data->value();
}

///////////////////////////////////////////////////////////////////////////////

/**
 * Specialize this function for special data type conversion of a ROS topic data
 * to MIRA channel data
 */
template <typename T>
void ros2mira(const typename MIRA2ROSTrait<T>::type::ConstPtr& data,
              mira::ChannelWrite<T>& out)
{
	out->value() = data->data;
}

///////////////////////////////////////////////////////////////////////////////

/*
 * Odometry
 */

MIRA2ROSTRAIT(mira::robot::Odometry2, nav_msgs::Odometry)

template<> void mira2ros<mira::robot::Odometry2>(mira::ChannelRead<mira::robot::Odometry2> data, MIRA2ROSTrait<mira::robot::Odometry2>::type& out) {
	out.header.stamp = ros::Time::fromBoost(data.getTimestamp());
	out.header.frame_id = data->frameID;

	out.pose.pose.position.x = data->value().pose.x();
	out.pose.pose.position.y = data->value().pose.y();
	out.pose.pose.orientation = tf::createQuaternionMsgFromYaw(data->value().pose.phi());

	out.twist.twist.linear.x = data->value().velocity.x();
	out.twist.twist.linear.y = data->value().velocity.y();
	out.twist.twist.angular.z = data->value().velocity.phi();
}

/*
 * Laser range data
 */

MIRA2ROSTRAIT(mira::robot::RangeScan, sensor_msgs::LaserScan)

template<> void mira2ros<mira::robot::RangeScan>(mira::ChannelRead<mira::robot::RangeScan> data, MIRA2ROSTrait<mira::robot::RangeScan>::type& out) {
	out.header.stamp = ros::Time::fromBoost(data.getTimestamp());
	out.header.frame_id = data->frameID;

	out.angle_min = data->value().startAngle.value();
	out.angle_max = data->value().startAngle.value() + (data->value().range.size() * data->value().deltaAngle.rad());
	out.angle_increment = data->value().deltaAngle.rad();

	// No such data in MIRA
	out.time_increment = 0.0f;
	out.scan_time = 0.0f; 

	out.range_min = data->value().minimumRange;
	out.range_max = data->value().maximumRange;

	out.ranges.resize(data->value().range.size());
	for(unsigned int i = 0; i < data->value().range.size(); ++i) {
		out.ranges[i] = data->value().range[i];
	}

	out.intensities.resize(data->value().reflectance.size());
	for(unsigned int i = 0; i < data->value().reflectance.size(); ++i) {
		out.intensities[i] = data->value().reflectance[i];
    }
}

/*
 * Map
 */

MIRA2ROSTRAIT(mira::maps::OccupancyGrid, nav_msgs::OccupancyGrid)

template<> void mira2ros<mira::maps::OccupancyGrid>(mira::ChannelRead<mira::maps::OccupancyGrid> data, MIRA2ROSTrait<mira::maps::OccupancyGrid>::type& out) {
	out.header.stamp = ros::Time::fromBoost(data.getTimestamp());
	out.header.frame_id = data->frameID;

	out.info.map_load_time = ros::Time::now();
	out.info.resolution = data->value().getCellSize();
	out.info.width = data->value().width();
	out.info.height = data->value().height();
	out.info.origin.position.x = -data->value().getWorldOffset().x();
	out.info.origin.position.y = -data->value().getWorldOffset().y();
	out.info.origin.position.z = 0.0f;
	out.info.origin.orientation = tf::createQuaternionMsgFromYaw(0.0f);

	const unsigned int numberOfCells = data->value().width() * data->value().height();
	out.data.resize(numberOfCells);

	int threshold = 35;	
	for(unsigned int i = 0; i < numberOfCells; ++i) {
	    out.data[i] = (data->value().data()[i] < threshold) ? 0 : 100;
	} 	  
}

/*
 * Battery data - Commented out, because it is not really sure what battery server
 * the ROS side is expecting. Maybe only voltage and current are of interest?
 */
/*
MIRA2ROSTRAIT(mira::robot::BatteryState, pr2_msgs::BatteryServer2)

template<> void mira2ros<mira::robot::BatteryState>(mira::ChannelRead<mira::robot::BatteryState> data, MIRA2ROSTrait<pr2_msgs::BatteryServer2>::type& out) {
	out.id = 1;
	out.last_system_update = ros::Time::now();
	out.time_left = ros::Duration(data->value().lifeTime * 60, 0);
	out.average_charge = data->value().lifePercent;

	if(data->value().voltage > BATTERY_GOOD_VOLTAGE)
	  out.message = "Battery good.";
	else if(data->value().voltage > BATTERY_SHUTDOWN_VOLTAGE)
	  out.message = "Starting shutdown when voltage drops by another " + boost::lexical_cast<std::string>(data->value().voltage - BATTERY_SHUTDOWN_VOLTAGE) + " volts";
	else
	  out.message = "Shutting down due to low battery voltage.";

	out.last_controller_update = ros::Time::now();
	out.battery.resize(1);
	out.battery[0].present = true;
	out.battery[0].charging = data->value().charging;
	out.battery[0].discharging = data->value().current > 0.5f; // battery charging: current < 0, discharging: current > 0
	out.battery[0].power_present = data->value().voltage > 22.0f;
	out.battery[0].power_no_good = data->value().voltage < 22.0f;
	out.battery[0].inhibited = false;
	out.battery[0].last_battery_update = ros::Time::now();

	std::fill(out.battery[0].battery_register.begin(), out.battery[0].battery_register.end(), 0);

	// The registers have never been updated
	std::fill(out.battery[0].battery_update_flag.begin(), out.battery[0].battery_update_flag.end(), false);

	// I hope ROS doesn't choke when the registers have never been updated, but the last update was just now
	std::fill(out.battery[0].battery_register_update.begin(), out.battery[0].battery_register_update.end(), ros::Time::now());
}
*/
