/*
 * Copyright (C) 2025 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TestXMLMacroChain.C
 *    Tests for builtin macros.
 *
 * @author Adrian Kriegel
 * @date   Mon Feb 10 2025
 */
#include <boost/test/unit_test.hpp>

#include <geometry/Point.h>
#include <serialization/XMLSerializer.h>

#include <xml/macros/Utils.h>
#include <xml/macros/XMLMacroProcessor.h>

#include <xml/macros/TestUtils.h>

using namespace mira;
using namespace mira::xmlmacros;

BOOST_AUTO_TEST_CASE(TestForMacro)
{
	testEqual(
	    R"(
		<macro:For>
			<MyIterator>
				<item>1</item>
				<item>2</item>
				<item>3</item>
			</MyIterator>
			<Do>
				<macro:MyIterator />
			</Do>
		</macro:For>)",
	    //
	    "123");

	testEqual(
	    R"(
		<macro:For>
			<MyIterator>
				<item>
					<Foo>foo</Foo>
				</item>
				<item>
					<Foo>Bar</Foo>
				</item>
				<item>
					<Foo>Baz</Foo>
				</item>
			</MyIterator>
			<Do>
				<macro:MyIterator />
			</Do>
		</macro:For>)",
	    //
	    "<Foo>foo</Foo><Foo>Bar</Foo><Foo>Baz</Foo>");
}

BOOST_AUTO_TEST_CASE(TestImportMacro)
{
	// Inline import and call.
	testEqual(
	    R"(
		<macro:MyImportedMacro From="MIRABase:tests/xml/macros/etc/ImportedMacro.xml">
			<MyParam>Bar</MyParam>
		</macro:MyImportedMacro>
		)",
	    //
	    R"(<Foo>Bar</Foo>)");
}

BOOST_AUTO_TEST_CASE(TestSplitMacro)
{
	testEqual(
	    R"(
		<macro:Split>foo bar baz</macro:Split>)",
	    //
	    R"(<item>foo</item><item>bar</item><item>baz</item>)");
	
	testEqual(
	    R"(
		<macro:Split Words="foo bar baz" />)",
	    //
	    R"(<item>foo</item><item>bar</item><item>baz</item>)");

	testEqual(
	    R"(
		<var myVariable="foo bar baz" />
		<macro:Split Words="${myVariable}" />)",
	    //
	    R"(<item>foo</item><item>bar</item><item>baz</item>)");
		
	testEqual(
	    R"(
		<macro:Define_ExecSplit>
			<Body>
				<macro:Split Words="${myVariable}" />
			</Body>
		</macro:Define_ExecSplit>

		<var myVariable="foo bar baz" />
		<macro:ExecSplit />

		<var myVariable="boo baz far" overwrite="true"/>
		<macro:ExecSplit />
		)",
	    //
	    R"(<item>foo</item><item>bar</item><item>baz</item><item>boo</item><item>baz</item><item>far</item>)");
}

BOOST_AUTO_TEST_CASE(TestSplitSegfault) 
{
	testEqual(
	    R"(
		<macro:Define_Serials>
			<Parameters></Parameters>
			<Body></Body>
		</macro:Define_Serials>

		<macro:Split Words="this should work" />)",
	    //
	    R"(<item>this</item><item>should</item><item>work</item>)");
}

BOOST_AUTO_TEST_CASE(TestZipMacro)
{
	testEqual(
	    R"(
		<macro:Zip>
			<Foo>
				<item>1</item>
				<item>2</item>
				<item>3</item>
			</Foo>
			<Bar>
				<item>A</item>
				<item>B</item>
				<item>C</item>
			</Bar>
		</macro:Zip>)",
	    //
	    R"(<item><Foo>1</Foo><Bar>A</Bar></item><item><Foo>2</Foo><Bar>B</Bar></item><item><Foo>3</Foo><Bar>C</Bar></item>)");
}

BOOST_AUTO_TEST_CASE(TestZipAndFor)
{
	testEqual(
	    R"(
		<var ids   = "2 4" />
		<var names = "foo bar" />

		<macro:For>
			<Object>
				<macro:Zip>
					<Name>
						<macro:Split Words="${names}" />
					</Name>
					<ID>
						<macro:Split Words="${ids}" />
					</ID>
				</macro:Zip>
			</Object>
			<Do>
				<Object>
					<ID><macro:Object.ID /></ID>
					<Name><macro:Object.Name /></Name>
				</Object>
			</Do>
		</macro:For>)",
	    //
	    R"(
		<Object>
			<ID>2</ID>
			<Name>foo</Name>
		</Object>
		<Object>
			<ID>4</ID>
			<Name>bar</Name>
		</Object>)");
}
