/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SerializationVeryBasicTest.C
 *    Test for serialization framework.
 *
 * @author Erik Einhorn
 */

#include "BasicClasses.h"
#include "CommonTest.h"

#include <error/LogTxtStreamSink.h>
#include <serialization/adapters/std/vector>

using namespace std;
using namespace mira;


template <typename T>
void serializeAll(T& value, const char* baseName, int num)
{
	string xmlfilename = (boost::format("%s%d.xml") % baseName % num).str();
	string jsonfilename = (boost::format("%s%d.json") % baseName % num).str();

	{
	XMLDom xml;
	XMLSerializer s(xml);
	s.serialize("value", value);
	xml.saveToFile(xmlfilename);
	}

	{
	JSONSerializer s;
	json::Value v = s.serialize(value);
	ofstream fs(jsonfilename.c_str());
	json::write(v,fs,true);
	fs.flush();

	T value2;
	JSONDeserializer d(v);
	d.deserialize(value2);
	}
}

//////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( Test1 )
{
	int v = 1234;
	serializeAll(v, "VeryBasic", 1);
}

BOOST_AUTO_TEST_CASE( Test2 )
{
	std::string v = "Hello World";
	serializeAll(v, "VeryBasic", 2);
}

BOOST_AUTO_TEST_CASE( Test3 )
{
	float v[] = {1.23f,2.34f,4.56f,5.67f};
	serializeAll(v, "VeryBasic", 3);
}

//////////////////////////////////////////////////////////////////////////////

class SimpleClass
{
public:
	SimpleClass() : a(123) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("a", a, "");
	}

	int a;
};

//////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( Test4 )
{
	SimpleClass c;
	serializeAll(c, "VeryBasic", 4);
}

BOOST_AUTO_TEST_CASE( Test5 )
{
	SimpleClass *p = NULL;
	serializeAll(p, "VeryBasic", 5);
}

BOOST_AUTO_TEST_CASE( Test6 )
{
	SimpleClass *p = new SimpleClass();
	serializeAll(p, "VeryBasic", 6);
	delete p;
}

BOOST_AUTO_TEST_CASE( Test7 )
{
	SimpleClass v[] = {SimpleClass(),SimpleClass(),SimpleClass(),SimpleClass()};
	serializeAll(v, "VeryBasic", 7);
}

//////////////////////////////////////////////////////////////////////////////

class ClassWithPointer
{
public:
	ClassWithPointer() : p(NULL) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("v", v, "");
		r.member("p", p, "");

	}

	SimpleClass* p;
	SimpleClass v;
};

//////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( Test8 )
{
	ClassWithPointer v;
	v.p = new SimpleClass;
	serializeAll(v, "VeryBasic", 8);
	delete v.p;
}
