/*
 * Copyright (C) 2015 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SerializationVariantTest.C
 *    Tests for the variant serialization adapter
 *
 * @author Tom Mehner
 * @date   Mar 14, 2023
 */

#include <boost/test/unit_test.hpp>

#include "CommonTest.h"

#include <math/Angle.h>
#include <serialization/IgnoreMissing.h>
#include <serialization/adapters/boost/optional.hpp>
#include <serialization/adapters/boost/tuple.hpp>
#include <xml/XMLDomReflect.h>

#include <serialization/adapters/boost/variant.hpp>

// A class that deliberately does not reflect anything.
// Serialization/deserialization should work regardless.
struct Struct
{
	template<typename Reflector>
	void reflect(Reflector& r)
	{}
};

class ComplexTypeBoost
{
public:
	ComplexTypeBoost() = default;

	// set member to some values
	ComplexTypeBoost(bool)
	{
		opt1.reset(123);
		opt2.reset();
		tup = boost::make_tuple(12, 3.45f, "678");
		guard = 0xABCD;
		phi = 1.25;
		xml.root().add_child("Node");
		variantMember = std::string("test string");
	}

	// returns true, if the values are as required
	void check()
	{
		BOOST_CHECK_EQUAL(*opt1, 123);
		BOOST_CHECK_EQUAL((bool)opt2, false);
		BOOST_CHECK_EQUAL(tup.get<0>(), 12);
		BOOST_CHECK_EQUAL(tup.get<1>(), 3.45f);
		BOOST_CHECK_EQUAL(tup.get<2>(), "678");
		BOOST_CHECK_EQUAL(guard, 0xABCD);
		BOOST_CHECK_EQUAL(phi.rad(), 1.25);
		BOOST_CHECK_EQUAL(*xml.root().begin(), "Node");
		BOOST_CHECK_EQUAL(boost::get<std::string>(variantMember), "test string");
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("opt1", opt1, "", serialization::IgnoreMissing());
		r.member("opt2", opt2, "", serialization::IgnoreMissing());
		r.member("tup", tup, "");
		r.member("guard", guard, "");
		r.member("phi", phi, "");
		r.member("xml", xml, "");
		r.member("struct", st, "");
		r.member("variantMember", variantMember, "");
	}

	boost::optional<int> opt1;
	boost::optional<int> opt2;
	boost::tuple<int, float, std::string> tup;
	int guard{};
	Anglef phi;
	XMLDom xml;
	Struct st;

	boost::variant<int, std::string, bool> variantMember;
};

class ComplexerTypeBoost
{
public:
	ComplexerTypeBoost() = default;

	// set member to some values
	ComplexerTypeBoost(bool)
	{
		variantMember1 = ComplexTypeBoost{true};
		variantMember2 = std::string("another test string");
		variantMember3 = true;
	}

	void check()
	{
		boost::get<ComplexTypeBoost>(variantMember1).check();
		BOOST_CHECK_EQUAL(boost::get<std::string>(variantMember2), "another test string");
		BOOST_CHECK_EQUAL(boost::get<bool>(variantMember3), true);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("variantMember1", variantMember1, "");
		r.member("variantMember2", variantMember2, "");
		r.member("variantMember3", variantMember3, "");
	}

	boost::variant<ComplexTypeBoost, std::string, bool> variantMember1;
	boost::variant<std::string, bool> variantMember2;
	boost::variant<std::string, ComplexTypeBoost, bool> variantMember3;
};


BOOST_AUTO_TEST_CASE(VariantMemberTestBoost)
{
	testAll<ComplexTypeBoost>("VariantMemberClass", 1);
	testAll<ComplexerTypeBoost>("VariantMemberClass2", 1);
}

////////////////////////////////////////////////////////////////////////////////////////////

#include <serialization/adapters/std/variant>

class ComplexTypeStd
{
public:
	ComplexTypeStd() = default;

	// set member to some values
	ComplexTypeStd(bool)
	{
		opt1.reset(123);
		opt2.reset();
		tup = boost::make_tuple(12, 3.45f, "678");
		guard = 0xABCD;
		phi = 1.25;
		xml.root().add_child("Node");
		variantMember = std::string("test string");
	}

	// returns true, if the values are as required
	void check()
	{
		BOOST_CHECK_EQUAL(*opt1, 123);
		BOOST_CHECK_EQUAL((bool)opt2, false);
		BOOST_CHECK_EQUAL(tup.get<0>(), 12);
		BOOST_CHECK_EQUAL(tup.get<1>(), 3.45f);
		BOOST_CHECK_EQUAL(tup.get<2>(), "678");
		BOOST_CHECK_EQUAL(guard, 0xABCD);
		BOOST_CHECK_EQUAL(phi.rad(), 1.25);
		BOOST_CHECK_EQUAL(*xml.root().begin(), "Node");
		BOOST_CHECK_EQUAL(std::get<std::string>(variantMember), "test string");
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("opt1", opt1, "", serialization::IgnoreMissing());
		r.member("opt2", opt2, "", serialization::IgnoreMissing());
		r.member("tup", tup, "");
		r.member("guard", guard, "");
		r.member("phi", phi, "");
		r.member("xml", xml, "");
		r.member("struct", st, "");
		r.member("variantMember", variantMember, "");
	}

	boost::optional<int> opt1;
	boost::optional<int> opt2;
	boost::tuple<int, float, std::string> tup;
	int guard{};
	Anglef phi;
	XMLDom xml;
	Struct st;

	std::variant<int, std::string, bool> variantMember;
};

class ComplexerTypeStd
{
public:
	ComplexerTypeStd() = default;

	// set member to some values
	ComplexerTypeStd(bool)
	{
		variantMember1 = ComplexTypeStd{true};
		variantMember2 = std::string("another test string");
		variantMember3 = true;
	}

	void check()
	{
		std::get<ComplexTypeStd>(variantMember1).check();
		BOOST_CHECK_EQUAL(std::get<std::string>(variantMember2), "another test string");
		BOOST_CHECK_EQUAL(std::get<bool>(variantMember3), true);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("variantMember1", variantMember1, "");
		r.member("variantMember2", variantMember2, "");
		r.member("variantMember3", variantMember3, "");
	}

	std::variant<ComplexTypeStd, std::string, bool> variantMember1;
	std::variant<std::string, bool> variantMember2;
	std::variant<std::string, ComplexTypeStd, bool> variantMember3;
};


BOOST_AUTO_TEST_CASE(VariantMemberTestStd)
{
	testAll<ComplexTypeStd>("VariantMemberClass", 1);
	testAll<ComplexerTypeStd>("VariantMemberClass2", 1);
}

BOOST_AUTO_TEST_CASE(XMLSerializationTest)
{
	const std::string XMLStr = R"(
	<root>
		<VariantWrongType>
			<Type>int</Type>
			<Content>Hello World</Content>
		</VariantWrongType>
		<Variant>
			<Type>std::string</Type>
			<Content>Hello World</Content>
		</Variant>
	</root>)";
	std::variant<std::string, bool> variant;

	XMLDom domIn;
	domIn.loadFromString(XMLStr);

	XMLDeserializer deserializer(domIn);

	BOOST_CHECK_THROW(deserializer.deserialize("VariantWrongType", variant),
	                  XInvalidConfig);

	deserializer.deserialize("Variant", variant);
	BOOST_CHECK_EQUAL(std::get<std::string>(variant), "Hello World");

	XMLDom domOut;
	XMLSerializer serialize{domOut};
	serialize.serialize("Variant", variant);

	auto it = domOut.root().find("Variant");
	BOOST_CHECK(it != domOut.root().end());

	auto itType = it.find("Type");
	BOOST_CHECK(itType != it.end());
	BOOST_CHECK_EQUAL(*itType.content_begin(), std::string("std::string"));

	auto contentIt = it.find("Content");
	BOOST_CHECK(contentIt != it.end());
	BOOST_CHECK_EQUAL(*contentIt.content_begin(), std::string("Hello World"));
}

BOOST_AUTO_TEST_CASE(JSONSerializationTest)
{
	std::variant<std::string, int, bool> variant;
	JSONValue json;
	JSONDeserializer deserializer(json);

	// type index 3 does not exist
	json::read("{\"Type\":3, \"Content\":123 }", json);
	BOOST_CHECK_THROW(deserializer.deserialize(variant),
	                  XInvalidConfig);

	// string content "123" cannot be read for type index 1 = int
	json::read("{\"Type\": 1, \"Content\":\"123\" }", json);
	BOOST_CHECK_THROW(deserializer.deserialize(variant),
	                  XInvalidConfig);

	json::read("{ \"Type\":1, \"Content\":123 }", json);
	deserializer.deserialize(variant);
	BOOST_CHECK_EQUAL(std::get<int>(variant), 123);

	JSONSerializer serializer;
	json::Value v = serializer.serialize(variant);
	BOOST_CHECK_EQUAL(json::write(v, false), "{\"Content\":123,\"Type\":1}");
}
