/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file IgnoredXMLTagsTest.C
 *    Test for warnings if config contains tags that are ignored by the deserialization
 *
 * @author Tom Mehner
 * @date   2025/01/09
 */

#include <boost/test/unit_test.hpp>

#include <serialization/adapters/std/vector>
#include <serialization/Serialization.h>

#include <error/LogTxtStreamSink.h>

#include <math/Eigen.h>
#include <utils/ToString.h>

using namespace std;
using namespace mira;

class SomeObject{
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("String", aString, "");
	}

	std::string aString;
};

class TestClass
{
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Vector", v, "");
		r.member("Object", obj, "");
	}

	std::vector<int> v;
	SomeObject obj;
};



BOOST_AUTO_TEST_CASE( IgnoredXMLTagsTest )
{
	const auto xmlConfig = R"(
	<root>
		<Test>
			<Vector>
				<item>42</item>
				<item>42</item>
				<itme>42</itme><!-- wrong -->
				<item>42</item>
			</Vector>
			<Object>
				<String>A String</String>
				<Unexpected>spanish inquisition</Unexpected>
			</Object>
		</Test>
	</root>)";

	std::stringstream ss;
	MIRA_LOGGER.registerSink(LogTxtStreamSink(ss));
	TestClass test;
	XMLDom xml;
	xml.loadFromString(xmlConfig);
	XMLDeserializer s(xml.croot());
	s.deserialize("Test", test);

	const auto output = ss.str();

	BOOST_CHECK(output.find("XML tag itme (originating from xml:7) is not used during deserialization") != std::string::npos);
	BOOST_CHECK(output.find("XML tag Unexpected (originating from xml:12) is not used during deserialization") != std::string::npos);
}

BOOST_AUTO_TEST_CASE( IgnoredXMLTagsMayBeIgnored )
{
	const auto xmlConfig = R"(
	<root>
		<Test>
			<Vector>
				<item>42</item>
				<item>42</item>
				<itme>42</itme><!-- wrong -->
				<item>42</item>
			</Vector>
			<Object>
				<String>A String</String>
				<Unexpected __may_be_ignored__="true">spanish inquisition</Unexpected>
			</Object>
		</Test>
	</root>)";

	std::stringstream ss;
	MIRA_LOGGER.registerSink(LogTxtStreamSink(ss));
	TestClass test;
	XMLDom xml;
	xml.loadFromString(xmlConfig);
	XMLDeserializer s(xml.croot());
	s.deserialize("Test", test);

	const auto output = ss.str();
	BOOST_CHECK(output.find("XML tag itme (originating from xml:7) is not used during deserialization") != std::string::npos);
	BOOST_CHECK(output.find("XML tag Unexpected originating from xml:12) is not used during deserialization") == std::string::npos);
}
