/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ImgBaseTest.C
 *    Test cases for img class.
 *
 * @author Michael Volkhardt
 * @date   2010/10/20
 */

#include <iostream>

#include <boost/test/unit_test.hpp>
#include <boost/geometry/strategies/strategies.hpp>

//#include <opencv2/highgui.h>
//#include <opencv2/highgui.hpp>

// For cvRedirectError, cvStdErrReport, cvErrorStr
#include <opencv2/core/version.hpp>
#if CV_MAJOR_VERSION >= 4
#  include <opencv2/core/core_c.h>
#endif

#include <platform/Types.h>
#include <utils/Foreach.h>
#include <geometry/Rect.h>
#include <image/Img.h>

using namespace mira;

///////////////////////////////////////////////////////////////////////////////

static int openCVErrorHandler(int code, const char *func_name,
		const char *err_msg, const char *file, int line, void*) {
	const char* errorStr = cvErrorStr(code);
	std::cerr << "OpenCV Error: " << errorStr << " (" << err_msg << ") in "
			<< func_name << ", " << file << ":" << line << std::endl;
	cvStdErrReport(code, func_name, err_msg, file, line, NULL);
	abort(); // this abort is essential, to allow the debugger to show a detailed stack trace
	return 0;
}


// some test cases for code coverage
BOOST_AUTO_TEST_CASE( ImgBaseConstructionTest )
{
	// construction
	Img8U3 img1;
	Img<float, 2> img2;
	Img<> img3;
	BOOST_CHECK(img1.isEmpty());
	BOOST_CHECK(img2.empty());
	BOOST_CHECK(img3.isEmpty());

	Img8U3 img4(Size2i(200, 300));
	BOOST_CHECK(!img4.isEmpty());
	BOOST_CHECK(img4.size() == Size2i(200,300));
	BOOST_CHECK(img4.width() == 200 && img4.height() ==300);
	BOOST_CHECK(img4.step()==600);
	BOOST_CHECK(img4.channels() == 3);
	BOOST_CHECK(img4.total() == 60000);
	BOOST_CHECK(img4.depth() == CV_MAT_DEPTH(CV_8UC3));
	cv::Mat tMat(100, 100, CV_8UC1);
	Img<> img5(tMat);
	BOOST_CHECK(img5.size() ==Size2i(100,100));
	BOOST_CHECK(img5.depth() ==CV_MAT_DEPTH(CV_8UC1));

	Img8U3 img6(img1);
	Img<> img7(img1);
	BOOST_CHECK((img6 == img1) && (img1== img7));
}

#if MIRA_USE_IPL_IMAGE > 0
BOOST_AUTO_TEST_CASE( ImgBaseConstructionCopyTest )
{
	IplImage *iplImg = cvCreateImage(cvSize(40,30), IPL_DEPTH_8U, 3);
	cvRectangle(iplImg, cvPoint(10, 10), cvPoint(30, 20), cvScalar(255,255,255), CV_FILLED);
	Img<> imgCpy1(iplImg, true);
	Img<> imgCpy2(iplImg, false);
	BOOST_CHECK(imgCpy1 == imgCpy2);
	
	cvReleaseImage(&iplImg);
}
#endif

BOOST_AUTO_TEST_CASE( ImgBaseComparisionTest )
{
	Img8U3 img(10, 10);
	img = Img8U3::Pixel(10,10,10);
	Img8U3 img2(10, 10);
	img2 = Img8U3::Pixel(10,10,10);
	BOOST_CHECK(img == img2);

	Img8U3 img3(img);
	Img<> img4(img);
	BOOST_CHECK((img == img3) && (img == img4) );

	img2 = Color::Red;
	BOOST_CHECK(img != img2);

	Img8U3 img5(11, 11);
	BOOST_CHECK(img != img5);

	Img8U1 img6(10, 10);
	BOOST_CHECK(img != img6);
}

BOOST_AUTO_TEST_CASE( ImgBaseOpenCVTest )
{
	Img8U1 img(100, 100);
	cv::Mat mat = img;
	cv::Mat const& mat2 = img.getMat();
	BOOST_CHECK(mat.size()== cv::Size(100,100));
	BOOST_CHECK(mat.total() == 10000);
	BOOST_CHECK(mat2.size()== cv::Size(100,100));
	BOOST_CHECK(mat2.total() == 10000);

#if MIRA_USE_IPL_IMAGE > 0
	IplImage iplimg = mat2;
	Img<> img3(&iplimg);
	IplImage iplimg2 = img;
	Img<> img4(&iplimg2);
	BOOST_CHECK(img3.size() == Size2i(100,100));
	BOOST_CHECK(img4.size() == Size2i(100,100));
	BOOST_CHECK((img == img3) &&( img== img4));
#endif
}

BOOST_AUTO_TEST_CASE( ImgBaseFunctionsTest )
{

	Img8U1 img(10, 10);
	img = Img8U1::Pixel(0);;
	uint8* data = img.data();
	*data = 1;
	BOOST_CHECK(img(0,0) == 1);
	*img.data() = 2;
	BOOST_CHECK(img(0,0) == 2);
	uint8 const* data2 = img.data();
	uint8 *data3 = img.data(1);
	++data3;
	*data3 = 3;
	BOOST_CHECK(img(1,1) ==3);
	uint8 const* data4 = img.data(2);

	BOOST_CHECK(img(5,5) == img[5][5]);
}
//test cases for basic functionality
BOOST_AUTO_TEST_CASE( ImgBaseTest )
{
	cvRedirectError(openCVErrorHandler); // set our own opencv handler

	Img8U3 roi;

	{
		Img8U3 img1(320, 240);
		Img8U1 img2(320, 240);
		Img<> img3 = img1; // untyped image
		// pixel and iteration
		Img8U3::Pixel p(0, 128, 255);
		img1 = p;
		foreach(Img8U3::Pixel& p, img1) {
			BOOST_CHECK(p == Img8U3::Pixel(0, 128, 255));
		}
		//cloning
		img3 = img1.clone();
		BOOST_CHECK(img3 == img1);
		Img8U3::Pixel p2(255, 128, 255);
		img1 = p2;
		BOOST_CHECK(img3 != img1);

		//iterator fancy color
		int i = 0;
		foreach(Img8U3::Pixel& p, img1) {
			p[0] = i % 255;
			p[1] = i % 255;
			p[2] = (255 - i) % 255;
			++i;
		}
		BOOST_CHECK(img1(0,0) == Img8U3::Pixel(0,0,0));
		BOOST_CHECK(img1(1,0) == Img8U3::Pixel(1,1,254));
		//iterator begin, end
		for (Img<>::iterator it = img3.begin(); it < img3.end(); ++it) {
			it->at<uint8> (0) = 255;
			it->at<uint8> (1) = 255;
			it->at<uint8> (2) = 0;
		}
		for (Img<>::iterator it = img3.begin(); it < img3.end(); ++it) {
			BOOST_CHECK( it->at<uint8> (0) == 255);
			BOOST_CHECK( it->at<uint8> (1) == 255);
			BOOST_CHECK( it->at<uint8> (2) == 0);
		}

		//iterator over ROI
		Img<> untypedroi = img3(Rect2i(10, 10, 100, 100));
		for (Img<>::iterator it = untypedroi.begin(); it < untypedroi.end(); ++it) {
			it->at<uint8> (2) = 255;
		}
		for (Img<>::const_iterator it = untypedroi.begin(); it < untypedroi.end(); ++it) {
			BOOST_CHECK(it->at<uint8> (0) == 255);
			BOOST_CHECK(it->at<uint8> (1) == 255);
			BOOST_CHECK(it->at<uint8> (2) == 255);

		}
		//single channel image
		img2 = Img8U1::Pixel(127);
		foreach(Img8U1::Pixel& p, img2) {
			BOOST_CHECK(p == 127);
		}
		// roi that is painted in one color (used to check for existence later on)
		roi = img1(Rect2i(10, 10, 100, 100));
		foreach(Img8U3::Pixel& p, roi) {
			p = Img8U3::Pixel(50, 200, 20);
		}

		//conversion
		Img<uint8, 3> imgColor(200, 200);
		imgColor = Img8U3::Pixel(127, 0, 255);
		
		Img<uint8, 1> imgGray;
		imgGray = Img<uint8, 1>::convertFrom(imgColor); // copies first src channel
		BOOST_CHECK(imgColor.size() == imgGray.size());
		foreach(Img8U1::Pixel& p,imgGray) {
			// if this fails, check Img.h line 601
			BOOST_CHECK(p == Img8U1::Pixel(127));
			p = Img8U1::Pixel(85);
		}
		
		imgColor = Img<uint8, 3>::convertFrom(imgGray); // copies 1 src channel to all 3 dest channels
		foreach(Img8U3::Pixel& p, imgColor) {
			BOOST_CHECK(p == Img8U3::Pixel(85, 85, 85));
		}
	}

	// destructed original image, only roi available
	foreach(Img8U3::Pixel& p, roi) {
		BOOST_CHECK(p == Img8U3::Pixel(50, 200, 20));
	}
}

BOOST_AUTO_TEST_CASE( ImgROIAssignmentTest )
{
	Img8U3 img1(320, 240);
	Img8U3 img2(320, 240);
	img1 = Img8U3::Pixel(255,128,50);
	img2.clear();

	foreach(Img8U3::Pixel p, img2)
		BOOST_CHECK(p == Img8U3::Pixel(0,0,0));

	Rect2i r(100,100,50,50);
	img2.assignROI(r, img1(r));

	Img8U3 roi = img2(r);
	foreach(Img8U3::Pixel& p, roi) {
		BOOST_CHECK(p == Img8U3::Pixel(255,128,50));
		p = Img8U3::Pixel(0,0,0);
	}

	foreach(Img8U3::Pixel p, img2)
		BOOST_CHECK(p == Img8U3::Pixel(0,0,0));

	//cv::imshow("img1", img1);
	//cv::imshow("img2", img2);
	//cv::waitKey();
}

BOOST_AUTO_TEST_CASE( ImgMaskAssignmentTest )
{
	Img8U3 img1(320, 240);
	Img8U3 img2(320, 240);
	img1 = Img8U3::Pixel(255,128,50);
	img2.clear();

	foreach(Img8U3::Pixel p, img2)
		BOOST_CHECK(p == Img8U3::Pixel(0,0,0));

	Img8U1 mask(50, 50);
	mask = 0;
	for (int y = 0; y < mask.width(); ++y)
		for (int x = y; x < mask.height(); ++x)
			mask(x,y) = 1;

	Rect2i r(100,100,50,50);
	img2.assignMask(r, img1(r), mask);

	for (int y = 0; y < mask.width(); ++y)
		for (int x = y; x < mask.height(); ++x) {
			BOOST_CHECK(img2(100+x,100+y) == Img8U3::Pixel(255,128,50));
			img2(100+x,100+y) = Img8U3::Pixel(0,0,0);
	}

	foreach(Img8U3::Pixel p, img2)
		BOOST_CHECK(p == Img8U3::Pixel(0,0,0));
}

BOOST_AUTO_TEST_CASE( ImgPolyAssignmentTest )
{
	Img8U3 img1(320, 240);
	Img8U3 img2(320, 240);
	img1 = Img8U3::Pixel(255,128,50);
	img2.clear();

	foreach(Img8U3::Pixel p, img2)
		BOOST_CHECK(p == Img8U3::Pixel(0,0,0));

	Polygon2i poly;
	poly.emplace_back(10,20);
	poly.emplace_back(25,20);
	poly.emplace_back(25,25);
	poly.emplace_back(35,25);
	poly.emplace_back(35,40);
	poly.emplace_back(10,40);
	poly.emplace_back(10,20);

	Rect2i r(100,100,50,50);
	img2.assignPolygon(r, img1(r), poly);

	for (int y = 0; y < 50; ++y)
		for (int x = 0; x < 50; ++x) {	
			if (boost::geometry::covered_by(Point2i(x,y), poly)) {
				BOOST_CHECK(img2(100+x,100+y) == Img8U3::Pixel(255,128,50));
				img2(100+x,100+y) = Img8U3::Pixel(0,0,0);
			}
		}

	foreach(Img8U3::Pixel p, img2)
		BOOST_CHECK(p == Img8U3::Pixel(0,0,0));

	////////////////////////////////////////////////////////////////
	// check that reversing polygon points order makes no difference

	img2.clear();

	foreach(Img8U3::Pixel p, img2)
		BOOST_CHECK(p == Img8U3::Pixel(0,0,0));

	Polygon2i poly2;
	poly2.emplace_back(10,20);
	poly2.emplace_back(10,40);
	poly2.emplace_back(35,40);
	poly2.emplace_back(35,40);
	poly2.emplace_back(35,25);
	poly2.emplace_back(25,25);
	poly2.emplace_back(25,20);
	poly2.emplace_back(10,20);

	img2.assignPolygon(r, img1(r), poly2);

	for (int y = 0; y < 50; ++y)
		for (int x = 0; x < 50; ++x) {

			// according to 
			// https://www.boost.org/doc/libs/1_62_0/libs/geometry/doc/html/geometry/reference/algorithms/covered_by/covered_by_2.html
			// and
			// https://www.boost.org/doc/libs/1_62_0/libs/geometry/doc/html/geometry/reference/algorithms/within/within_2.html
			// this check does not depend on the order of points (when using default strategy)
			if (boost::geometry::covered_by(Point2i(x,y), poly)) {

				BOOST_CHECK(img2(100+x,100+y) == Img8U3::Pixel(255,128,50));
				img2(100+x,100+y) = Img8U3::Pixel(0,0,0);
			}
		}

	foreach(Img8U3::Pixel p, img2)
		BOOST_CHECK(p == Img8U3::Pixel(0,0,0));
}

BOOST_AUTO_TEST_CASE( ImgToTypeTest )
{
	Img8U3 imgT(320, 240);
	Img<> imgU = imgT;
	Img8U3 tmp = img_cast<uint8, 3>(imgU);
	BOOST_CHECK(tmp == imgT);

	BOOST_CHECK_THROW((img_cast<uint8, 1>(imgU)), mira::XBadCast);
	BOOST_CHECK_THROW((img_cast<float, 3>(imgU)), mira::XBadCast);
	BOOST_CHECK_THROW((img_cast<int32, 2>(imgU)), mira::XBadCast);

	bool hasCorrectType = imgU.hasType<uint8,3>();
	BOOST_CHECK_EQUAL(hasCorrectType, true);
	hasCorrectType = imgU.hasType<int8,4>();
	BOOST_CHECK_EQUAL(hasCorrectType, false);
}

BOOST_AUTO_TEST_CASE( ImgResizeTest )
{
	//test if a zero-sized image initializes the OpenCV data type to use
	//the image after resizing
	Img8U3 img;
	img.resize(20,20);
	for (int y = 0; y < img.height(); ++y ) {
		for (int x = 0; x < img.width(); ++x ) {
			img(x,y) =  Img8U3::Pixel(128,255,0);
			BOOST_CHECK(img(x,y) == Img8U3::Pixel(128,255,0));
		}
	}
}

BOOST_AUTO_TEST_CASE( ImgEmptyIteratorTest )
{
	//test if iterators on empty images (width==0 || height==0) are handled
	//correctly, tests, if there's no abnormal program termination
	{
	std::cout << "1:" << std::endl;
	Img8U1 img(1, 1);
	for (Img8U1::iterator it = img.begin(); it != img.end(); ++it)
		std::cout << it.pos().x() << ", " << it.pos().y() << std::endl;
	}

	{
	std::cout << "2:" << std::endl;
	Img8U1 img(100, 0);
	for (Img8U1::iterator it = img.begin(); it != img.end(); ++it)
		std::cout << it.pos().x() << ", " << it.pos().y() << std::endl;
	}

	{
	std::cout << "3:" << std::endl;
	Img8U1 img(0, 100);
	for (Img8U1::iterator it = img.begin(); it != img.end(); ++it)
		std::cout << it.pos().x() << ", " << it.pos().y() << std::endl;
	}

}
