/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ColorTest.C
 *    test cases for the color classes.
 *
 * @author Erik Einhorn
 */

#include <boost/test/unit_test.hpp>

#include <opencv2/imgproc/imgproc.hpp>

#include <image/Img.h>
#include <utils/Foreach.h>
#include <error/Exceptions.h>

using namespace mira;

///////////////////////////////////////////////////////////////////////////////

bool rgbInRange(const Color::RGB &rgbColor) {
	bool inrange = ((rgbColor.r >= 0.0) && (rgbColor.r <= 1.0)
	     && (rgbColor.g >= 0.0) && (rgbColor.g <= 1.0)
	     && (rgbColor.b >= 0.0) && (rgbColor.b <= 1.0));
	BOOST_CHECK_EQUAL(inrange, rgbColor.isInRange());
	return inrange;
}

void checkRGB(const Color::RGB& rgb, float r, float g, float b)
{
//	std::cout << print("c:", rgb) << std::endl;

	BOOST_CHECK(fabs(rgb.r - r)<0.05);
	BOOST_CHECK(fabs(rgb.g - g)<0.05);
	BOOST_CHECK(fabs(rgb.b - b)<0.05);
}

#define CHECK_MIN_MAX_PARAMETERS(COLORSPACE, X_MIN, X_MAX, Y_MIN, Y_MAX, Z_MIN, Z_MAX, STEP)			\
	BOOST_CHECK_NO_THROW(COLORSPACE(X_MIN, Y_MIN, Z_MIN).toRGB());						\
	BOOST_CHECK_NO_THROW(COLORSPACE(X_MIN+STEP, Y_MIN, Z_MIN).toRGB());					\
	BOOST_CHECK_NO_THROW(COLORSPACE(X_MIN, Y_MIN+STEP, Z_MIN).toRGB());					\
	BOOST_CHECK_NO_THROW(COLORSPACE(X_MIN, Y_MIN, Z_MIN+STEP).toRGB());					\
	BOOST_CHECK_NO_THROW(COLORSPACE(X_MAX, Y_MAX, Z_MAX).toRGB());						\
	BOOST_CHECK_NO_THROW(COLORSPACE(X_MAX-STEP, Y_MAX, Z_MAX).toRGB());					\
	BOOST_CHECK_NO_THROW(COLORSPACE(X_MAX, Y_MAX-STEP, Z_MAX).toRGB());					\
	BOOST_CHECK_NO_THROW(COLORSPACE(X_MAX, Y_MAX, Z_MAX-STEP).toRGB());

#define CHECK_PARAMETERS_IN_RANGE(COLORSPACE, X_MIN, X_MAX, Y_MIN, Y_MAX, Z_MIN, Z_MAX, STEP)			\
	for (float x = X_MIN; x <= X_MAX; x += STEP) {								\
		for (float y = Y_MIN; y <= Y_MAX; y += STEP) {							\
			for (float z = Z_MIN; z <= Z_MAX; z += STEP) {						\
				BOOST_CHECK(rgbInRange(COLORSPACE(x, y, z).toRGB()));				\
			}											\
		}												\
	}

BOOST_AUTO_TEST_CASE( ColorHSVToRGBConvertTest )
{
	BOOST_CHECK(Img8U3::Pixel(Color::HSV(0.15f, 0.1f, 0.7f).toRGB()) == Img8U3::Pixel(Color::RGB(0.7f, 0.693f, 0.63f)));
	BOOST_CHECK(Img8U3::Pixel(Color::HSV(0.30f, 0.2f, 0.8f).toRGB()) == Img8U3::Pixel(Color::RGB(0.672f, 0.8f, 0.64f)));
	BOOST_CHECK(Img8U3::Pixel(Color::HSV(0.45f, 0.3f, 0.9f).toRGB()) == Img8U3::Pixel(Color::RGB(0.63f, 0.9f, 0.819f)));
	BOOST_CHECK(Img8U3::Pixel(Color::HSV(0.60f, 0.4f, 0.1f).toRGB()) == Img8U3::Pixel(Color::RGB(0.06f, 0.076f, 0.1f)));
	BOOST_CHECK(Img8U3::Pixel(Color::HSV(0.75f, 0.5f, 0.2f).toRGB()) == Img8U3::Pixel(Color::RGB(0.15f, 0.1f, 0.2f)));
	BOOST_CHECK(Img8U3::Pixel(Color::HSV(0.90f, 0.6f, 0.3f).toRGB()) == Img8U3::Pixel(Color::RGB(0.3f, 0.12f, 0.228f)));

	CHECK_MIN_MAX_PARAMETERS( Color::HSV,     0, 1,     0, 1,     0, 1,     0.001f);
	CHECK_PARAMETERS_IN_RANGE(Color::HSV,     0, 1,     0, 1,     0, 1,     0.3f);
}

BOOST_AUTO_TEST_CASE( ColorXYZToRGBConvertTest )
{

	checkRGB(Color::XYZ(41,21,2).toRGB(), 1.0, 0.0, 0.0);
	checkRGB(Color::XYZ(36,72,12).toRGB(), 0.0, 1.0, 0.0);
	checkRGB(Color::XYZ(18,7,95).toRGB(), 0.0, 0.0, 1.0);
	checkRGB(Color::XYZ(95,100,108.8f).toRGB(), 1.0, 1.0, 1.0);
	checkRGB(Color::XYZ(53,38,25).toRGB(), 1.0, 0.5, 0.5);
	checkRGB(Color::XYZ(0,0,0).toRGB(), 0.0, 0.0, 0.0);
}

BOOST_AUTO_TEST_CASE( ColorLabToRGBConvertTest )
{
	checkRGB(Color::Lab(65,-41,40).toRGB(), 0.396f, 0.686f, 0.323f);
}

BOOST_AUTO_TEST_CASE( ColorTest )
{
	Img8U3 img1(320,200);

	float h = 0.0f;
	foreach(Img8U3::Pixel& p, img1)
	{
		p=Color::HSV(h,1.0f,1.0f); // operater= inits a ImgPixel in RGB-format (converts HSV to RGB)
		h+=0.0001f;

		if(h>1.0f)
		h=0.0f;
	}

	// check some background pixels in RGB
	BOOST_CHECK(img1(100,100) == Img8U3::Pixel(0, 255, 188));
	BOOST_CHECK(img1(100,101) == Img8U3::Pixel(0, 255, 139));
	BOOST_CHECK(img1(100,102) == Img8U3::Pixel(0, 255, 90));

	// red rectangle
	Img8U3 roi = img1(cv::Rect(10, 10, 40, 40));
	foreach(Img8U3::Pixel& p, roi) {
		p=Color::Red;
	}
	foreach(Img8U3::Pixel& p, roi) {
		BOOST_CHECK(p[0]==0 && p[1]==0 && p[2]==255); //note: image has BGR order
	}
	// green
	roi = img1(cv::Rect(30,30,40,40));
	foreach(Img8U3::Pixel& p, roi) {
		p=Color::Green;
	}
	Img8U3::Pixel green(0,255,0);
	foreach(Img8U3::Pixel& p, roi) {
		BOOST_CHECK(p == green);
	}
	//blue
	Img8U3::Pixel blue(Color::Blue);
	roi = img1(cv::Rect(50,50,40,40));
	foreach(Img8U3::Pixel& p, roi) {
		p=Color::Blue;
	}
	foreach(Img8U3::Pixel& p, roi) {
		BOOST_CHECK(p==blue);
	}

	// test opencv drawing with color class -> colored circles
	cv::circle((cv::Mat&)img1, cv::Point(160,140), 50, Color::Red, 10);
	cv::circle((cv::Mat&)img1, cv::Point(160,140), 50, Color::Blue, 2);

	BOOST_CHECK(img1(160,190) == Img8U3::Pixel(Color::Blue));
	BOOST_CHECK(img1(160,185) == Img8U3::Pixel(Color::Red));
}

BOOST_AUTO_TEST_CASE( ColorScalarTest )
{
	Color::RGBA color(0.8f, 0.5f, 0.1f, 0.8f);
	Img8U4::Pixel pixel(26, 128, 204, 204);
	Img8U4 imgFromPixel(320,200);
	imgFromPixel = pixel;
	Img8U4 imgFromColor(320,200);
	imgFromColor = color;

	BOOST_CHECK(imgFromPixel == imgFromColor);

	cv::Mat cvImgFromScalar(cv::Size(320,200),CV_8UC4);
	cvImgFromScalar = cv::Scalar(26,128,204,204);

	cv::Mat cvImgFromColor(cv::Size(320,200),CV_8UC4) ;
	cvImgFromColor = color;

	BOOST_CHECK(imgFromPixel == cvImgFromScalar);
	BOOST_CHECK(imgFromPixel == cvImgFromColor);
}

